//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Thread_h)
#define FeatStd_Platform_Thread_h

#include <FeatStd/Platform/Base.h>
#include <FeatStd/Config.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
/** Thread run status */
    namespace ThreadStatus {
        enum Enum {
            /// Thread object is available but not initialized
            Invalid,
            /// Set after Thread::Run() was called (thread itself must not run yet)
            Started,
            /// Set after Thread::OnExecute(), when thread effectively running
            Running,
            // Suspended,
            /// Set after Thread::OnTerminate(), when thread ended
            Terminated
        };
    }

    /** ThreadPriority */
    namespace ThreadPriority {
        enum Enum {
            IDLE,
            LOW,
            BELOW_NORMAL,
            NORMAL,
            ABOVE_NORMAL,
            HIGH,
            REAL_TIME,
            NO_PERMISSION_RETURN,
            ERROR_RETURN,
            _PLATFORM_DEFAULT // FeatStd internal use only
        };
    }

    namespace ThreadInvariant {
        /** Maximum thread name length */
        static const UInt NameLength = 15;
        static const UInt NameBufferLength = NameLength + 1;
        static const Char ThreadNameFormat[] = "FeatStd-0x%x";
    }
/// @}
}}

#include FEATSTD_PLATFORM_INCLUDE(Thread)

#ifndef _FEATSTD_OS_INTERNAL_THREAD_NOT_SUPPORTED

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** A thread is the basic unit to which the operating system allocates processor time. */
    class Thread : private Impl::Thread {
        typedef Impl::Thread Base;
    public:
        /** Returns ID of the currently running thread.
            @return Current thread ID. */
        static UInt32 GetCurrentId();

        /** Returns ID of the calling thread. This id is system-wide visible.
        @return Current thread ID. */
        static UInt32 GetCurrentSystemId();

       /** Suspends current thread for the given amount of time.
            @param time Sleep time in milliseconds */
        static void Sleep(UInt32 time);

        /** Suspends the thread for given seconds and nano seconds
            @param seconds number of seconds to suspend thread
            @param nanoSeconds number of nano seconds to suspend thread (valid range 0 - 999 999 999)*/
        static inline void NanoSleep(UInt32 seconds, UInt32 nanoSeconds);

        /** Causes the calling thread to yield execution to another thread
            that is ready to run on the current processor.
            The operating system selects the thread to yield to.
            @return     <em>true</em> if the switch of execution to another thread was successful,
                        <em>false</em> if the switch fails, or no other thread is ready to execute. */
        static bool Yield();

        typedef Base::SetPriorityCallback SetPriorityCallback;

        /** Creates the Thread and runs the user-implemented <em>ThreadFn</em> method.
            @param priority Which priority the thread should run at. The actual priority should be checked after
                   with GetPriority
            @param callback Function to be used instead of native OS API call to set thread priority.
                   See \e SetPriority
            @return             <em>true</em> if thread is running,
                                <em>false</em> if thread failed to start up. */
        bool Run(
                ThreadPriority::Enum priority = ThreadPriority::_PLATFORM_DEFAULT,
                SetPriorityCallback Callback = static_cast<Base::SetPriorityCallback>(0)
                );

        /** Get the thread priority
            @return Thread priority */
        ThreadPriority::Enum GetPriority() const;

        /** Set the running thread priority.
            @param thread priority
            @param callback (optional) Function to be used instead of native OS API call to set thread priority.
                   Function declaration is OS/API-dependent:
                    - POSIX: void(*SetPriorityCallback)(::pid_t pid, ::pthread_t *thread, const int policy, const ::sched_param *priority);
                    - Other OS: NA
            @return Actual priority that the thread was set to or an error value. */
        ThreadPriority::Enum SetPriority(
                ThreadPriority::Enum priority,
                SetPriorityCallback Callback = static_cast<SetPriorityCallback>(0)
                );

        typedef Base::SetNiceCallback SetNiceCallback;

        /** Get the thread nice level (only has effect for Linux/NPTL Posix implementations).
         *  @return Nice value of thread */
        Int16 GetNiceLevel();

        /** Atempt to set the thread nice value (only has effect for Linux/NPTL Posix implementation).
         *  NOTE: To check if nice level was set successfully, call <em>GetNiceLevel</em> after thread was started.
         *  @param nice Nice value for thread.
         *  @param callback (optional) Function to be used instead of native OS API call to set thread nice level.
         *         Function declaration is OS/API-dependent:
         *         - POSIX: void(*SetNiceCallback)(const ::pid_t tid, const int nice)
         *         - Other OS: NA */
        void SetNiceLevel(
                Int16 nice,
                SetNiceCallback Callback = static_cast<SetNiceCallback>(0)
                );

        /** Set the name of the thread.
            @param name Thread name to set */
        void SetName(const Char * name);

        /** Get name of the thread.
            @return Thread name */
        const Char * GetName() const;

        /** Get ID of the thread.
            @return Thread ID */
        UInt32 GetId() const;

        /** Returns the child thread Id which is also system-wide.
        Use this API from Parent thread in order to find out the thread's IDs of it's children.
        @return Thread ID - if returned value is 0, then the thread ended it's execution. */
        UInt32 GetSystemId() const;

       /** Informs about the current run status of thread.
            @return Thread run status */
        ThreadStatus::Enum GetStatus() const;

        /** Causes the thread instance to be terminated and destroyed.
            Kill() is a dangerous function that should only be used in the most extreme cases.
            You should call Kill() only if you know exactly what the target thread is doing,
            and you control all of the code that the target thread could possibly be running
            at the time of the destruction.
            @param exitCode  The exit code used to terminate the thread.
            @return     <em>true</em> if thread termination was successful,
                        <em>false</em> if thread termination fails. */
        bool Kill(Int exitCode);

        /** Allows  application to wait for a thread to terminate.
            @return     <em>true</em> joined successfully,
                        <em>false</em> failed to join. */
        bool Join();

    protected:
        /** Specifies the entry function of the task.
            This function will be executed in thread context.
            @return Exit code of thread function */
        virtual Int ThreadFn() = 0;

        /** This method is called when starting the thread.
            It is running in the context of that method which is calling Thread::Run. */
        virtual void OnStart();

        /** This method is called when the thread starts its execution.
            It is running in the thread context and is called right before ThreadFn is
            invoked. */
        virtual void OnExecute();

        /** This method is called when the thread is terminating.
            It is running in the thread context and is called right after ThreadFn is
            returning.
            @param exitCode  The exit code used to terminate the thread. */
        virtual void OnTerminate(Int exitCode);
    };

    /*  --------------------------------------------------------------------
        GetCurrentId
        -------------------------------------------------------------------- */
    inline UInt32 Thread::GetCurrentId()
    {
        return Base::GetCurrentId();
    }

    /*  --------------------------------------------------------------------
        GetCurrentSystemWideId
    -------------------------------------------------------------------- */
    inline UInt32 Thread::GetCurrentSystemId()
    {
        return Base::GetCurrentSystemId();
    }

    /*  --------------------------------------------------------------------
        Sleep
        -------------------------------------------------------------------- */
    inline void Thread::Sleep(UInt32 time)
    {
        return Base::Sleep(time);
    }

    /*  --------------------------------------------------------------------
        NanoSleep
        -------------------------------------------------------------------- */
    inline void Thread::NanoSleep(UInt32 seconds, UInt32 nanoSeconds)
    {
        return Base::NanoSleep(seconds, nanoSeconds);
    }

    /*  --------------------------------------------------------------------
        Yield
        -------------------------------------------------------------------- */
    inline bool Thread::Yield()
    {
        return Base::Yield();
    }

    /*  --------------------------------------------------------------------
        Run
        -------------------------------------------------------------------- */
    inline bool Thread::Run(ThreadPriority::Enum priority, Base::SetPriorityCallback Callback)
    {
        return Base::Run(priority, Callback);
    }

    /*  --------------------------------------------------------------------
        GetPriority
        -------------------------------------------------------------------- */
    inline ThreadPriority::Enum Thread::GetPriority() const
    {
        return Base::GetPriority();
    }

    /*  --------------------------------------------------------------------
        SetPriority
        -------------------------------------------------------------------- */
    inline ThreadPriority::Enum Thread::SetPriority(ThreadPriority::Enum priority, Base::SetPriorityCallback Callback)
    {
        return Base::SetPriority(priority, Callback);
    }

    inline void Thread::SetNiceLevel(Int16 nice, SetNiceCallback Callback)
    {
        Base::SetNiceLevel(nice, Callback);
    }

    inline Int16 Thread::GetNiceLevel()
    {
        return Base::GetNiceLevel();
    }

    /*  --------------------------------------------------------------------
        SetName
        -------------------------------------------------------------------- */
    inline void Thread::SetName(const Char * name)
    {
        Base::SetName(name);
    }

    /*  --------------------------------------------------------------------
        GetName
        -------------------------------------------------------------------- */
    inline const Char * Thread::GetName() const
    {
        return Base::GetName();
    }

    /*  --------------------------------------------------------------------
        GetId
        -------------------------------------------------------------------- */
    inline UInt32 Thread::GetId() const
    {
        return Base::GetId();
    }

    /*  --------------------------------------------------------------------
        GetSystemWideId
    -------------------------------------------------------------------- */
    inline UInt32 Thread::GetSystemId() const
    {
        return Base::GetSystemId();
    }

    /*  --------------------------------------------------------------------
        GetStatus
        -------------------------------------------------------------------- */
    inline ThreadStatus::Enum Thread::GetStatus() const
    {
        return Base::GetStatus();
    }

    /*  --------------------------------------------------------------------
        Kill
        -------------------------------------------------------------------- */
    inline bool Thread::Kill(Int exitCode)
    {
        return Base::Kill(exitCode);
    }

    /*  --------------------------------------------------------------------
        OnStart
        -------------------------------------------------------------------- */
    inline void Thread::OnStart()
    {
        Base::OnStart();
    }

    /*  --------------------------------------------------------------------
        OnExecute
        -------------------------------------------------------------------- */
    inline void Thread::OnExecute()
    {
        Base::OnExecute();
    }

    /*  --------------------------------------------------------------------
        OnTerminate
        -------------------------------------------------------------------- */
    inline void Thread::OnTerminate(Int exitCode)
    {
        Base::OnTerminate(exitCode);
    }

    /*  --------------------------------------------------------------------
        Join
    -------------------------------------------------------------------- */
    inline bool Thread::Join()
    {
        bool joinStatus = false;
        // Check first if thread execution has finish. If so, don't join threads anymore.
        if (ThreadStatus::Terminated == GetStatus())
        {
            joinStatus = true;
        }
        else {
            // Need to forward call to base class to handle differently for Windows or Linux case.
            joinStatus = Base::Join();
        }
        return joinStatus;
    }
/// @}
}}
#endif //_FEATSTD_OS_INTERNAL_THREAD_NOT_SUPPORTED
#endif
