//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Semaphore_h)
#define FeatStd_Platform_Semaphore_h

#include <FeatStd/Platform/Base.h>

#include FEATSTD_PLATFORM_INCLUDE(Semaphore)

#ifndef _FEATSTD_OS_INTERNAL_SEMAPHORE_NOT_SUPPORTED

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** <em>Semaphore</em> object is a synchronization object that maintains a count between zero
        and a specified maximum value. The count is decremented each time a thread completes a wait
        for the semaphore object and incremented each time a thread releases the semaphore. When
        the count reaches zero, no more threads can successfully wait for the semaphore object state
        to become signaled. The state of a semaphore is set to signaled when its count is greater than
        zero, and non-signaled when its count is zero.

        <em>Semaphore</em> is useful in controlling a shared resource that can support a limited
        number of users. It acts as a gate that limits the number of threads sharing the resource
        to a specified maximum number. */
    class Semaphore : private Impl::Semaphore {
        typedef Impl::Semaphore Base;
    public:
        /** @param  initVal     Specifies an initial count for the semaphore object.
                                The state of a semaphore is signaled when its count is greater than zero
                                and non-signaled when it is zero. The count is decreased by one whenever
                                a wait function releases a thread that was waiting for the semaphore.
                                The count is increased by a specified amount by calling the Release method.
            @return             <em>true</em> if succeeded.
                                <em>false</em> if failed. */
        bool Create(Int initVal = 0);

        /** <em>Obtain</em> returns if the specified object is in the signaled state. */
        bool Obtain();

        /** <em>Obtain</em> returns when one of the following occurs:
            - The specified object is in the signaled state.
            - The time-out interval elapses.
            @param  timeout     Specifies the time-out interval, in milliseconds. The function returns
                                if the interval elapses, even if the object's state is non-signaled.
                                If <em>timeout</em> is zero, the method tests the object's state and
                                returns immediately. If <em>timeout</em> is INFINITE, the method's
                                time-out interval never elapses.
            @return             <em>true</em> if the state is signaled.
                                <em>false</em> if the time-out interval elapsed. */
        bool Obtain(const Ticks & timeout);

        /** <em>Release</em> increases the count of the semaphore object.
            If count exceeds <em>maxVal</em> that was specified when the semaphore was created, the count
            is not changed.
            @return <em>true</em> if releasing the semaphore succeeded,
                    <em>false</em> otherwise. */
        bool Release();

        /** <em>Destroy</em> frees the semaphore resources. */
        bool Destroy();
    };

    /*  --------------------------------------------------------------------
        Create
        -------------------------------------------------------------------- */
    inline bool Semaphore::Create(Int initVal)
    {
        return Base::Create(initVal);
    }

    /*  --------------------------------------------------------------------
        Create
        -------------------------------------------------------------------- */
    inline bool Semaphore::Obtain(const Ticks & timeout)
    {
        return Base::Obtain(timeout);
    }

    /*  --------------------------------------------------------------------
        Create
        -------------------------------------------------------------------- */
    inline bool Semaphore::Obtain()
    {
        return Base::Obtain();
    }

    /*  --------------------------------------------------------------------
        Release
        -------------------------------------------------------------------- */
    inline bool Semaphore::Release()
    {
        return Base::Release();
    }

    /*  --------------------------------------------------------------------
        Destroy
        -------------------------------------------------------------------- */
    inline bool Semaphore::Destroy()
    {
        return Base::Destroy();
    }
/// @}
}}
#endif // _FEATSTD_OS_INTERNAL_SEMAPHORE_NOT_SUPPORTED
#endif
