//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Posix_PosixThreadLocalPointer_h)
#define FeatStd_Platform_Posix_PosixThreadLocalPointer_h

#include <FeatStd/Platform/Base.h>
#include <FeatStd/Diagnostics/Log.h>
#include <pthread.h>

namespace FeatStd { namespace Internal { namespace Posix {

    FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdPlatform);

/// @addtogroup FEATSTD_PLATFORM_POSIX
/// @{


FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 5-18-1: false positive, comma operator is not used")
/** 
  Posix implementation of thread-local pointers.
  */
class PosixThreadLocalPointer {
public:
    PosixThreadLocalPointer(void);
    virtual ~PosixThreadLocalPointer(void);

protected:

    /** 
      Retrieves the thread-local instance of this pointer.
      @return The thread-local instance of this pointer.
      */
    inline void* GetData() const;

    /** 
      Sets the thread-local instance of this pointer to a new value.
      @param The new value.
      */
    inline void SetData(void* data);

private:
    pthread_key_t m_tlsIndex;
    bool m_valid;

    inline bool IsValid() const { return m_valid; }
};

inline void* PosixThreadLocalPointer::GetData() const
{
    void* pData = 0;

    if (IsValid()) {
        pData = ::pthread_getspecific(m_tlsIndex);
    } else {
        FEATSTD_LOG_ERROR("Trying to retrieve thread-local storage using invalid key.");
    }
    return pData;
}

inline void PosixThreadLocalPointer::SetData(void* data)
{
    FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixThreadLocalPointer::SetData, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
    FEATSTD_LINT_SYMBOL(818, data, "[MISRA C++ Rule 7-1-2] The symbol in question is not declared as "const" by intention because it typically modifies the object in a direct, indirect, or virtual way.")
    FEATSTD_LINT_SYMBOL(550, data, "[MISRA C++ Rule 0-1-4] data is not accessed is a false positive")
        
    if (IsValid()) {
        static_cast<void>(::pthread_setspecific(m_tlsIndex, data));
    } else {
        FEATSTD_LOG_ERROR("Trying to set thread-local storage using invalid key.");
    }
}

}}}

namespace FeatStd { namespace Internal { namespace Impl {
    typedef FeatStd::Internal::Posix::PosixThreadLocalPointer ThreadLocalPointer;
}}}

#endif // !defined(FeatStd_Platform_Posix_PosixThreadLocalPointer_h)
