//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Posix_PosixThread_h)
#define FeatStd_Platform_Posix_PosixThread_h

#include <FeatStd/Platform/Base.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <pthread.h>

namespace FeatStd { namespace Internal { namespace Posix {
    /// @addtogroup FEATSTD_PLATFORM_POSIX
    /// @{
    /** Posix basic thread functionality implementation */
    class PosixThread {
    public:
        ///
        static SizeType GetCurrentId();
        static SizeType GetCurrentSystemId();

        ///
        static void Sleep(UInt32 timeMs);
        static void NanoSleep(UInt32 seconds, UInt32 nanoSeconds);

        ///
        static bool Yield();

        ///
        PosixThread();

        ///
        virtual ~PosixThread();

        ///
        typedef void(*SetPriorityCallback)(const SizeType threadId, const int policy, const ::sched_param *priority);

        ///
        bool Run(ThreadPriority::Enum thPriority, SetPriorityCallback Callback);

        ///
        ThreadPriority::Enum GetPriority() const;

        ///
        ThreadPriority::Enum SetPriority(ThreadPriority::Enum thPriority, SetPriorityCallback Callback);

        ///
        typedef void(*SetNiceCallback)(const pid_t tid, const int nice);

        ///
        Int16 GetNiceLevel();

        ///
        void SetNiceLevel(Int16 nice, SetNiceCallback Callback);

        ///
        void SetName(const Char * name);

        ///
        const Char * GetName() const { return mName; }

        ///
        SizeType GetId() const;

        ///
        SizeType GetSystemId() const;

        ///
        ThreadStatus::Enum GetStatus() const;

        ///
        bool Kill(Int exitCode);

        ///
        bool Join();

    protected:
        ///
        virtual Int ThreadFn() = 0;

        ///
        virtual void OnStart();

        ///
        virtual void OnExecute();

        ///
        virtual void OnTerminate(Int exitCode);

        /** Set to true if Join() function was called successfully one time upon the same thread. */
        bool mJoined;
    private:

        /** Thread run status. */
        mutable ThreadStatus::Enum mStatus;

        /** Thread handle. */
        pthread_t mThread;

        /** Local thread critical section for synchronizing e.g. thread status. */
        CriticalSection mCs;

        /** Thread's System-Wide ID **/
        SizeType mCachedSystemId;

        /** Nice level desired */
        Int16 mNice;

        /** Callback function to set nice level */
        SetNiceCallback mNiceCallback;

        /** Thread name. */
        Char mName[ThreadInvariant::NameBufferLength];

        /** General thread start procedure procedure.
            This procedure implements the method hooks (OnExecute() and OnTerminate()) and
            starts the thread specific start function (ThreadFn()).
            @param arg  The thread data passed to the function.
            @return     Indicates the success or failure of this function. */
        static void * StartProc(void * arg);

        /** Safely set thread status.
            @param status Thread status to set. */
        void SetStatus(ThreadStatus::Enum status);

        void SetNiceLevelImpl();

        FEATSTD_MAKE_CLASS_UNCOPYABLE(PosixThread);
    };
    /// @}
}}}

namespace FeatStd { namespace Internal { namespace Impl {
    typedef FeatStd::Internal::Posix::PosixThread Thread;
}}}

#endif
