//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Platform/TcpIp.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Diagnostics/Log.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Util/WarningMacros.h>

FEATSTD_LINT_FILE(829, stdio.h, "platform dependent file")
//lint -efile(829, errno.h)  platform dependent file
FEATSTD_LINT_FILE(829, string.h, "platform dependent file")
#include <stdio.h>
#include <cstdlib>
#include <string.h>
#include <errno.h>

#include <unistd.h> // POSIX
#include <sys/time.h>
#include <arpa/inet.h>
#include <netdb.h>


FEATSTD_LINT_NEXT_EXPRESSION(1923, "INVALID_SOCKET is OS dependent and cannot be a constant.")
#define INVALID_SOCKET (~0)
FEATSTD_LINT_NEXT_EXPRESSION(1923, "SOCKET_ERROR is OS dependent and cannot be a constant.")
#define SOCKET_ERROR   (~0)
FEATSTD_LINT_NEXT_EXPRESSION(1923, "WSA_STARTUP is OS dependent and cannot be a constant.")
#define WSA_STARTUP(a) 0
#define closesocket(s) close(s)

#define WSACleanup() 0
#define WSAGetLastError() errno




namespace FeatStd {
    namespace Internal {
        namespace Posix {
            FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::FeatStdPlatform);

            FEATSTD_LINT_CURRENT_SCOPE(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: ~): changing to unsigned type might change the end result of the expression")
            FEATSTD_LINT_CURRENT_SCOPE(1013, "false positive, c_cc, c_cflag, c_lflag and c_iflag are members of termios")
            FEATSTD_LINT_SYMBOL(818, buffer, "[MISRA C++ Rule 7-1-2] The symbol in question is not declared as 'const' by intention because it typically modifies the object in a direct, indirect, or virtual way.")
            FEATSTD_LINT_SYMBOL(550, buffer, "[MISRA C++ Rule 0-1-4] buffer is not accessed is a false positive")
            FEATSTD_LINT_CURRENT_SCOPE(948, "false positive - comparison does not always result in false/true")
            FEATSTD_LINT_CURRENT_SCOPE(774, "false positive - comparison does not always result in false/true")
                
            // ------------------------------------------------------------------------
            
            static inline SOCKET GetInvalidSocket(){
                return INVALID_SOCKET; //lint !e1924 !e1960    1924: C-style cast; 1960: Bitwise operator applied to signed underlying type
            }

            static void InitAddrInfo(bool isServer, struct addrinfo& addrInfo){
                FEATSTD_LINT_NEXT_EXPRESSION(826, "Necessary to cast byte adressable buffer to use the data stored inside.")
                FeatStd::Internal::Memory::Set(&addrInfo, 0, sizeof(addrInfo));
                addrInfo.ai_protocol = static_cast<FeatStd::Int>(IPPROTO_TCP);
                addrInfo.ai_socktype = SOCK_STREAM;
                if (isServer)
                {
                    addrInfo.ai_family = AF_INET;
                    addrInfo.ai_flags = AI_PASSIVE;
                }
                else
                {
                    addrInfo.ai_family = AF_UNSPEC;
                }
            }
            
            static UInt IsClientAvailable(SOCKET lSocket){
                
                FEATSTD_LINT_SYMBOL(550, timeout, "[MISRA C++ Rule 0-1-4] timeout is not accessed is a false positive")
                
                struct timeval timeout;
                fd_set readSet;
                timeout.tv_sec = 0;
                timeout.tv_usec = 0;

                FEATSTD_LINT_NEXT_EXPRESSION(534, "FD_ZERO is macro and can contain OS specific implementations. Has no return value.")
                FEATSTD_LINT_NEXT_EXPRESSION(1055, "FD_ZERO is macro and can contain OS specific implementations. Has no return value.")
                FD_ZERO(&readSet);//lint !e1924 C-style cast
                FEATSTD_LINT_SYMBOL(534, FD_SET, "FD_SET is macro and can contain OS specific implementations. Has no return value.")
                FEATSTD_LINT_NEXT_EXPRESSION(1055, "FD_SET is macro and can contain OS specific implementations. Has no return value.")
                    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) {
                    static_cast<void>(FD_SET(lSocket, &readSet)); //lint !e1960 !e1924 !e717 !e909   1960: Re-use of C++ identifier pattern; 1924: C-style cast; 717: WinSock2.h;
                }

                return ::select(lSocket + 1, &readSet, 0, 0, &timeout);
            }

            static SOCKET CloseSocket(SOCKET lSocket, const Char* msg)
            {
                FEATSTD_UNUSED(msg); // if logging is disabled, msg is not used
                static_cast<void>(::closesocket(lSocket));
                lSocket = GetInvalidSocket(); 
                FEATSTD_LOG_ERROR(msg);
                static_cast<void>(WSACleanup());
                FEATSTD_UNUSED(lSocket);
                return GetInvalidSocket();
            }


            static SOCKET AcceptSingleClient(SOCKET lSocket)
            {
                UInt selectResult = IsClientAvailable(lSocket);
                if (selectResult != 1)
                {
                    return GetInvalidSocket();
                }
                // Accept a client socket
                SOCKET cSocket = ::accept(lSocket, NULL, NULL);
                if (cSocket == GetInvalidSocket()) 
                {   
                    return CloseSocket(lSocket, "Unable to accept client!\n");
                }
                // No longer need server socket
                static_cast<void>(::closesocket(lSocket));

                FEATSTD_DEBUG_ASSERT(cSocket != GetInvalidSocket());
                return cSocket;
            }
            

            static SOCKET AcceptSingleClient(struct addrinfo * ptr, SOCKET lSocket, bool waitForAvailableClient, bool * isConnecting){
                bool clientAvailable = waitForAvailableClient;
                // Setup the TCP listening socket
                Int iResult = ::bind(lSocket, ptr->ai_addr, static_cast<Int>(ptr->ai_addrlen));
                if (iResult == SOCKET_ERROR) {
                    static_cast<void>(::closesocket(lSocket));
                    lSocket = GetInvalidSocket();   
                }

                // Should really try the next address returned by getaddrinfo
                // if the connect call failed
                // But for this simple example we just free the resources
                // returned by getaddrinfo and print an error message

                static_cast<void>(::freeaddrinfo(ptr));

                if (lSocket == GetInvalidSocket()) {    
                    FEATSTD_LOG_ERROR("Unable to start server!\n");
                    static_cast<void>(WSACleanup());
                    return GetInvalidSocket();
                }

                iResult = ::listen(lSocket, SOMAXCONN);
                if (iResult == SOCKET_ERROR) 
                {
                    return CloseSocket(lSocket, "Unable to start listening!\n");
                }
                if (!waitForAvailableClient){
                    UInt selectResult = IsClientAvailable(lSocket);
                    clientAvailable = (selectResult == 1);
                    if (!clientAvailable)
                    {
                        *isConnecting = true;
                        return lSocket;
                    }
                }
                SOCKET cSocket = GetInvalidSocket();  
                // Accept a client socket
                if (clientAvailable){
                    cSocket = ::accept(lSocket, NULL, NULL);
                }
                if (cSocket == GetInvalidSocket()) 
                {  
                    return CloseSocket(lSocket, "Unable to accept client!\n");
                }
                // No longer need server socket
                static_cast<void>(::closesocket(lSocket));



                FEATSTD_DEBUG_ASSERT(cSocket != GetInvalidSocket());
                return cSocket;
            }

            static SOCKET ConnectClientToServer(struct addrinfo * ptr, SOCKET lSocket)
            {
                FEATSTD_LINT_SYMBOL(818, ptr, "[MISRA C++ Rule 7-1-2] The symbol in question is not declared as 'const' by intention because it typically modifies the object in a direct, indirect, or virtual way.")
                FEATSTD_LINT_SYMBOL(550, ptr, "[MISRA C++ Rule 0-1-4] ptr is not accessed is a false positive")
                
                // Connect to server.
                Int iResult = ::connect(lSocket, ptr->ai_addr, static_cast<Int>(ptr->ai_addrlen));
                if (iResult == SOCKET_ERROR) {
                    static_cast<void>(::closesocket(lSocket));
                    lSocket = GetInvalidSocket();   
                }

                // Should really try the next address returned by getaddrinfo
                // if the connect call failed
                // But for this simple example we just free the resources
                // returned by getaddrinfo and print an error message

                ::freeaddrinfo(ptr);

                if (lSocket == GetInvalidSocket()) {    
                    FEATSTD_LOG_ERROR("Unable to connect to server!\n");
                    static_cast<void>(WSACleanup());
                    return GetInvalidSocket();
                }

                FEATSTD_DEBUG_ASSERT(lSocket != GetInvalidSocket());
                return lSocket;
            }


            PosixTcpIp::PosixTcpIp() :mSocket(GetInvalidSocket()), mIsConnecting(false), mListenerSocket(GetInvalidSocket()) 
            {

            }

            bool PosixTcpIp::DataAvailable(bool& available){
                FEATSTD_DEBUG_ASSERT(Connected());

                fd_set lSockets;
                FEATSTD_LINT_NEXT_EXPRESSION(534, "FD_ZERO is macro and can contain OS specific implementations. Has no return value.")
                FD_ZERO(&lSockets); //lint !e1924 C-style cast
                FEATSTD_LINT_SYMBOL(534, FD_SET, "FD_SET is macro and can contain OS specific implementations. Has no return value.")
                    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) {
                    static_cast<void>(FD_SET(mSocket, &lSockets)); //lint !e1960 !e1924 !e717 !e909   1960: Re-use of C++ identifier pattern; 1924: C-style cast; 717: WinSock2.h;
                }
                FEATSTD_LINT_SYMBOL(550, lTimeout, "[MISRA C++ Rule 0-1-4] lTimeout is not accessed is a false positive")
                
                struct timeval lTimeout;
                lTimeout.tv_sec = 0;
                lTimeout.tv_usec = 0;

                Int lSelectRes = ::select(static_cast<Int>(mSocket + 1),
                    &lSockets, NULL, NULL, &lTimeout);
                if (lSelectRes < 0) {
                    FEATSTD_LOG_ERROR("Error at socket(): %d\n", WSAGetLastError());

                    static_cast<void>(::closesocket(mSocket));
                    mSocket = GetInvalidSocket();  
                    static_cast<void>(WSACleanup());

                    return false;
                }else if (lSelectRes == 0) {
                    available = false;
                    return true;
                }
                else{
                    FEATSTD_DEBUG_ASSERT(lSelectRes == 1);
                    available = true;
                }
                return true;
            }

            Int32 PosixTcpIp::Read(UInt8*buffer, UInt32 length){
                
                FEATSTD_LINT_SYMBOL(818, buffer, "[MISRA C++ Rule 7-1-2] The symbol in question is not declared as 'const' by intention because it typically modifies the object in a direct, indirect, or virtual way.")
                FEATSTD_LINT_SYMBOL(550, buffer, "[MISRA C++ Rule 0-1-4] buffer is not accessed is a false positive")
                
                FEATSTD_DEBUG_ASSERT(buffer != NULL);
                FEATSTD_DEBUG_ASSERT(length <= 0x7FFFFFFF);

                FEATSTD_DEBUG_ASSERT(Connected());

                Int iResult = ::recv(mSocket,
                    reinterpret_cast<Char*>(buffer),
                    static_cast<Int>(length),
                    0);
                if (iResult > 0) {
                    FEATSTD_DEBUG_ASSERT(iResult == static_cast<Int>(length));
                    return iResult;
                }

                if (iResult == 0) {
                    FEATSTD_LOG_INFO("Connection closed\n");
                }
                else {
                    FEATSTD_LOG_ERROR("recv failed: %d\n", WSAGetLastError());
                }

                mSocket = GetInvalidSocket();
                return iResult;
            }

            Int32 PosixTcpIp::Write(const UInt8* buffer, UInt32 length){
                FEATSTD_LINT_SYMBOL(550, buffer, "[MISRA C++ Rule 0-1-4] buffer is not accessed is a false positive")
                
                FEATSTD_DEBUG_ASSERT(buffer != NULL);
                FEATSTD_DEBUG_ASSERT(length <= 0x7FFFFFFF);

                Int iResult = ::send(mSocket,
                    reinterpret_cast<const Char*>(buffer),
                    static_cast<Int>(length),
                    0);
                if (iResult < 0) {
                    FEATSTD_LOG_ERROR("send failed: %d\n", WSAGetLastError());
                    static_cast<void>(::closesocket(mSocket));
                    mSocket = GetInvalidSocket();   
                    static_cast<void>(WSACleanup());

                    return iResult;
                }

                return iResult;
            }

            bool PosixTcpIp::Close(){
                bool res = ::closesocket(mSocket) == 0;
                mSocket = GetInvalidSocket();
                return res;
            }

            bool PosixTcpIp::Connect(ConnectionHandshake handshake){
                FEATSTD_DEBUG_ASSERT(!Connected());
                if ((!mIsConnecting) && (!Connected()))
                {
                FEATSTD_LOG_INFO("Try to start server %s : %s !\n", mSettings.Server, mSettings.Port);

                // Initialize Winsock
                Int iResult = WSA_STARTUP(0x0202);
                FEATSTD_LINT_NEXT_EXPRESSION(774, "WSA_STARTUP has OS dependent functionality.")
                FEATSTD_LINT_NEXT_EXPRESSION(948, "WSA_STARTUP has OS dependent functionality.")
                if (iResult != 0) {
                    FEATSTD_LOG_ERROR("WSAStartup failed: %d\n", iResult);
                    return false;
                }

                // Create client socket
                struct addrinfo* result = NULL;
                struct addrinfo* ptr;
                struct addrinfo hints;

                InitAddrInfo(mSettings.isServer, hints);

                // Resolve the server address and port
                FEATSTD_LINT_NEXT_EXPRESSION(64, "False-positive: No type mismatch performed as getaddrinfo returns an int")
                iResult = ::getaddrinfo(mSettings.Server, mSettings.Port, &hints, &result);
                if (iResult != 0) {
                    FEATSTD_LOG_ERROR("getaddrinfo failed: %d\n", iResult);
                    static_cast<void>(WSACleanup());
                    return false;
                }

                // Attempt to connect to the first address returned by
                // the call to getaddrinfo
                FEATSTD_LINT_CURRENT_SCOPE(438, "ptr is used subsequently to create socket.")
                FEATSTD_LINT_CURRENT_SCOPE(550, "ptr is used subsequently to create socket.")
                ptr = result;

                // Create a SOCKET for connecting to server
                SOCKET lSocket = ::socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol);
                if (lSocket == GetInvalidSocket()) {  
                    FEATSTD_LOG_ERROR("Error at socket(): %d\n", WSAGetLastError());
                    static_cast<void>(::freeaddrinfo(result));
                    static_cast<void>(WSACleanup());
                    return false;
                }
                if (mSettings.isServer)
                {
                        mSocket = AcceptSingleClient(ptr, lSocket, mSettings.waitForAvailableClient, &mIsConnecting);
                        if (mIsConnecting)
                        {
                            mListenerSocket = mSocket;
                            mSocket = GetInvalidSocket();
                }
                    }
                else
                {
                    mSocket = ConnectClientToServer(ptr, lSocket);
                    }
                if (GetInvalidSocket() == mSocket){
                        return false;
                    }
                }
                else
                {
                    FEATSTD_DEBUG_ASSERT(mSettings.isServer);
                    FEATSTD_DEBUG_ASSERT(!mSettings.waitForAvailableClient);
                    mSocket = AcceptSingleClient(mListenerSocket);
                    if (GetInvalidSocket() == mSocket){
                        return false;
                    }
                    else
                    {
                        mListenerSocket = GetInvalidSocket();
                        mIsConnecting = false;
                    }
                }
                bool handshakeResult = true;
                if (handshake != 0){
                    handshakeResult = (*handshake)();
                }
                else
                {
                    FEATSTD_UNUSED(handshake);
                }
                return (GetInvalidSocket() != mSocket) && handshakeResult;    


            }

            bool PosixTcpIp::Configure(Settings const * configData){

                FEATSTD_LINT_NEXT_EXPRESSION(1924, "5-2-4 this cast warning is caused by system includes of INADDR_NONE")
                if (::inet_addr(configData->Server) == INADDR_NONE) {
                    FEATSTD_LOG_ERROR("CommTcpIp::Configure: Invalid TCP/IP address specified");
                    return false;
                }

                Int portNum = atoi(configData->Port);    //lint !e586    atoi is deprecated
                if ((portNum < 1) || (portNum > 65535)) {
                    FEATSTD_LOG_ERROR("CommTcpIp::Configure: Invalid TCP/IP port specified");
                    return false;
                }
                return TcpIp::CopySettings(&mSettings, configData);
            }

            bool PosixTcpIp::Connected()
            {
                return (mSocket != GetInvalidSocket()); 
            }//lint !e1762    1762: [MISRA C++ Rule 9-3-3] not const because of base class function declaration

        }
    }
}
