//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Platform/SerialPort.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Diagnostics/Log.h>

FEATSTD_LINT_FILE(829, stdio.h, "platform dependent file")
//lint -efile(829, errno.h)  platform dependent file
FEATSTD_LINT_FILE(829, string.h, "platform dependent file")

#include <poll.h>
#include <sys/select.h>
#include <sys/time.h>
#include <stdio.h>   /* Standard input/output definitions */
#include <string.h>  /* String function definitions */
#include <unistd.h>  /* UNIX standard function definitions */
#include <fcntl.h>   /* File control definitions */
#include <errno.h>   /* Error number definitions */

#include <iostream>

#ifndef ETIME
#define ETIME 62                //lint !e1923    1923: [MISRA C++ Rule 16-2-2] not const var: Flag defined in library - replacement if flag does not exist
#endif

#ifndef CRTSCTS
// define in lib:020000000000 as octal
#define CRTSCTS 0x80000000    //lint !e1923    1923: [MISRA C++ Rule 16-2-2] not const var: Flag defined in library - replacement if flag does not exist
#endif

#ifndef B460800
// define in lib: 0010004 as octal
#define B460800 0x1004         //lint !e1923    1923: [MISRA C++ Rule 16-2-2] not const var: Flag defined in library - replacement if flag does not exist
#endif

#ifndef B921600
// define in lib: 0010007 as octal
#define B921600 0x1007         //lint !e1923    1923: [MISRA C++ Rule 16-2-2] not const var: Flag defined in library - replacement if flag does not exist
#endif


namespace FeatStd {
    namespace Internal {
        namespace Posix {
            FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::FeatStdPlatform);
            namespace {
                template<typename T, typename FT>
                static inline void ResetMaskBits(T &var, const FT mask)
                {
                    var &= static_cast<T>(~static_cast<T>(mask));
                }

                template <typename T, typename FT>
                static inline void SetMaskBits(T &var, const FT mask)
                {
                    var |= static_cast<T>(mask);
                }
            }

            FEATSTD_LINT_CURRENT_SCOPE(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: |): changing to unsigned type might change the end result of the expression")
            FEATSTD_LINT_CURRENT_SCOPE(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: ~): changing to unsigned type might change the end result of the expression")
            FEATSTD_LINT_CURRENT_SCOPE(1960, "MISRA C++ 2008 Required Rule 5-0-21, Bitwise operator applied to signed underlying type: &): changing to unsigned type might change the end result of the expression")
            FEATSTD_LINT_CURRENT_SCOPE(1013, "false positive, c_cc, c_cflag, c_lflag and c_iflag are members of termios")
            FEATSTD_LINT_CURRENT_SCOPE(1013, "false positive, tv_sec and tv_usec are members of timeval")
            // ------------------------------------------------------------------------
            static bool SetRecvSize(Int32 port, UInt32 recvSize)
            {
                FEATSTD_LINT_SYMBOL(550, options, "[MISRA C++ Rule 0-1-4] options is not accessed is a false positive")
                FEATSTD_LINT_SYMBOL(550, port, "[MISRA C++ Rule 0-1-4] port is not accessed is a false positive")
        
                FEATSTD_DEBUG_ASSERT(port != 0);
                struct termios options;
                if(recvSize > 255){
                    recvSize = 255;
                }
                if (::tcgetattr(port, &options) == -1)
                {
                    FEATSTD_UNUSED(recvSize);
                    return false;
                }
                options.c_cc[VMIN] = recvSize;
                options.c_cc[VTIME] = 0;
                if (::tcsetattr(port, TCSANOW, &options) == -1){
                    return false;
                }
                return true;
            }
            static Int IsSelect(Int32 port, UInt timeoutUSec)
            {
                fd_set fds;
                struct timeval lTimeout;
                FEATSTD_LINT_NEXT_EXPRESSION(534, "FD_ZERO is macro and can contain OS specific implementations. Has no return value.")
                FEATSTD_LINT_NEXT_EXPRESSION(1055, "FD_ZERO is macro and can contain OS specific implementations. Has no return value.")
                FD_ZERO(&fds); //lint !e1924 C-style cast
                FEATSTD_LINT_SYMBOL(534, FD_SET, "FD_SET is macro and can contain OS specific implementations. Has no return value.")
                FEATSTD_LINT_NEXT_EXPRESSION(1055, "FD_SET is macro and can contain OS specific implementations. Has no return value.")
                FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) {
                    static_cast<void>(FD_SET(port, &fds)); //lint !e1960 !e1924 !e717 !e909   1960: Re-use of C++ identifier pattern; 1924: C-style cast; 717: WinSock2.h;
                }
                lTimeout.tv_sec = 0;
                lTimeout.tv_usec = static_cast<Int32>(timeoutUSec);
                return ::select(static_cast<Int>(port + 1), &fds, NULL, NULL, &lTimeout);
            }//lint !e550 e550: [MISRA C++ Rule 0-1-4] timeoutUSec is not accessed is a false positive 

            PosixSerialPort::PosixSerialPort() :hPort(0), mConnected(false),mConnecting(false)
            {
                mSettings.Baudrate = BaudRate_9600;
                mSettings.StopBits = StopBit_One;
                mSettings.Parity = Parity_None;
                mSettings.ReadTimeout = 500;
                mSettings.WriteTimeout = 500;
            }

            bool PosixSerialPort::DataAvailable(bool& available)
            {
                return DataAvailable(available, 0);
            }//lint !e1762    1762: [MISRA C++ Rule 9-3-3] not const because of base class function declaration

            bool PosixSerialPort::DataAvailable(bool& available, UInt timeoutUSec) const
            {
                Int lSelectRes = IsSelect(hPort, timeoutUSec);
                if (lSelectRes < 0) {
                    FEATSTD_LOG_ERROR("Error checking for available data!");
                    return false;
                }else if (lSelectRes == 0) {
                    available = false;
                    return true;
                }
                else{
                    FEATSTD_DEBUG_ASSERT(lSelectRes == 1);
                    available = true;
                }
                return true;
            }

            Int32 PosixSerialPort::Read(UInt8* buffer, UInt32 length)
            {
                FEATSTD_LINT_SYMBOL(818, buffer, "[MISRA C++ Rule 7-1-2] The symbol in question is not declared as 'const' by intention because it typically modifies the object in a direct, indirect, or virtual way.")
                FEATSTD_LINT_SYMBOL(550, buffer, "[MISRA C++ Rule 0-1-4] buffer is not accessed is a false positive")
                FEATSTD_LINT_SYMBOL(715, buffer, "[MISRA C++ Rule 0-1-4] buffer is not accessed is a false positive")
                
                bool dataAvailable = false;
                UInt32 readBytes = 0;
                if (mSettings.enableReadPendingOnly)
                {
                    if(!DataAvailable(dataAvailable))
                    {
                        return -1;
                    }
                    if(!dataAvailable)
                    {
                        return 0;
                    }
                    ssize_t tmpBytes = ::read(hPort,&buffer[0], length);
                    if((tmpBytes < 0) && (errno != ETIME)){
                        //perror("");
                        return -1;
                    }
                    readBytes = static_cast<UInt32>(tmpBytes);
                }
                else
                {
                    if (!SetRecvSize(hPort, length)){
                        return -1;
                    }
                    if(!DataAvailable(dataAvailable,static_cast<UInt>(mSettings.ReadTimeout*1000)))
                    {
                        return -1;
                    }
                    if(dataAvailable)
                    {
                        ssize_t tmpBytes = ::read(hPort,&buffer[0], length);
                        if((tmpBytes < 0) && (errno != ETIME)){
                            // perror("");
                            return -1;
                        }
                        readBytes = static_cast<UInt32>(tmpBytes);
                    }
                    if (!SetRecvSize(hPort, 1)){
                        return -1;
                    }
                }
                return static_cast<Int32>(readBytes);
            }

            Int32 PosixSerialPort::Write(const UInt8* buffer, UInt32 length)
            {
                return ::write(hPort, buffer, length);
            }//lint !e1762    1762: [MISRA C++ Rule 9-3-3] not const because of base class function declaration

            bool PosixSerialPort::Configure(Settings const * configData){
                return SerialPort::CopySettings(&mSettings, configData);
            }

            bool PosixSerialPort::Close(){
                mConnecting = false;
                mConnected = false;
                static_cast<void>(::close(hPort));
                return true;
            }

            bool PosixSerialPort::Connect(ConnectionHandshake handshake){
                if (!mConnecting)
                {
                    struct termios options;
                    if(hPort > 0)
                    {
                        static_cast<void>(Close());
                    }
                    hPort = ::open(mSettings.SerialPort, O_RDWR | O_NOCTTY | O_NDELAY);
                    if(hPort == -1)
                    {
                        hPort = 0;
                        return false;
                    }
    
                    if (::tcgetattr(hPort, &options) == -1){
                        static_cast<void>(Close());
                        return false;
                    }
    
                    options.c_cc[VMIN]=0;
                    options.c_cc[VTIME] = 0;
                    options.c_cflag &= ~CRTSCTS;
                    options.c_cflag |= (CLOCAL | CREAD);
                    options.c_lflag = 0;
                    if( (!SetParity(options)) || (!SetBaudrate(options)))
                    {
                        static_cast<void>(Close());
                        return false;
                    }
                    if(::tcflush(hPort,TCIOFLUSH) != 0){
                        static_cast<void>(Close());
                        return false;
                    }
                    if (::tcsetattr(hPort, TCSANOW, &options) == -1){
                        static_cast<void>(Close());
                        return false;
                    }
                    static_cast<void>(::fcntl(hPort, F_SETFL, 0));
                }
                bool result = true;
                if (handshake != 0){
                    mConnecting = true;
                    bool varEnableReadPendingOnly = mSettings.enableReadPendingOnly;
                    mSettings.enableReadPendingOnly = true;
                    result = (*handshake)();
                    mSettings.enableReadPendingOnly = varEnableReadPendingOnly;
                    if ((!mSettings.waitUntilHandshakeConnect) && (!result))
                    {
                        return false;
                    }
                    mConnecting = false;
                }
                else
                {
                    FEATSTD_UNUSED(handshake);
                }
                if(!result)
                {
                    static_cast<void>(Close());
                }
                mConnected = result;
                return result;
            }

            bool PosixSerialPort::Connected(){
                return mConnected;
            }//lint !e1762    1762: [MISRA C++ Rule 9-3-3] not const because of base class function declaration

            bool PosixSerialPort::SetParity(struct termios& options){
                FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSerialPort::SetParity, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
                FEATSTD_LINT_CURRENT_SCOPE(1025, "Linting in windows might not recognize termios structure. Therefore, it resolves the templates wrongly under windows. If this is an issue it would result in a compile error which it doesnt.")
                FEATSTD_LINT_CURRENT_SCOPE(1058, "Linting in windows might not recognize termios structure. Therefore, it resolves the templates wrongly under windows. If this is an issue it would result in a compile error which it doesnt.")
                if(mSettings.Parity == Parity_None)//8N1
                {
                    ResetMaskBits(options.c_cflag, PARENB);
                    ResetMaskBits(options.c_cflag, CSTOPB);
                    ResetMaskBits(options.c_cflag, CSIZE);
                    SetMaskBits(options.c_cflag, CS8);
                }else if(mSettings.Parity == Parity_Even){//7E1
                    SetMaskBits(options.c_cflag, PARENB); 
                    ResetMaskBits(options.c_cflag, PARODD);
                    ResetMaskBits(options.c_cflag, CSTOPB);
                    ResetMaskBits(options.c_cflag, CSIZE);
                    SetMaskBits(options.c_cflag, CS7);
                    SetMaskBits(options.c_iflag, (INPCK | ISTRIP));
                }else if(mSettings.Parity == Parity_Odd){//7O1
                    SetMaskBits(options.c_cflag, PARENB);
                    SetMaskBits(options.c_cflag, PARODD);
                    ResetMaskBits(options.c_cflag, CSTOPB);
                    ResetMaskBits(options.c_cflag, CSIZE);
                    SetMaskBits(options.c_cflag, CS7);
                    SetMaskBits(options.c_iflag, (INPCK | ISTRIP));
                }else{//Default = No parity (8N1)
                    ResetMaskBits(options.c_cflag, PARENB);
                    ResetMaskBits(options.c_cflag, CSTOPB);
                    ResetMaskBits(options.c_cflag, CSIZE);
                    SetMaskBits(options.c_cflag, CS8);
                }
                return true;
            }

            bool PosixSerialPort::SetBaudrate(struct termios& options){
                FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSerialPort::SetBaudrate, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
                
                static const UInt32 baudrates[] = { B0, B300, B1200, B2400, B4800, B9600, B19200, B38400, B57600, B115200, B230400, B460800, B921600 };
                static_cast<void>(::cfsetispeed(&options, baudrates[mSettings.Baudrate]));
                static_cast<void>(::cfsetospeed(&options, baudrates[mSettings.Baudrate]));
                return true;
            }
}}}
