//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Platform/Semaphore.h>

#include <FeatStd/Diagnostics/Log.h>

FEATSTD_LINT_FILE(829, errno.h, "platform dependent file")
//lint -efile(829, errno.h)  platform dependent file
FEATSTD_LINT_FILE(829, time.h, "platform dependent file")

#include <errno.h>
#include <time.h>

namespace FeatStd { namespace Internal { namespace Posix {

FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdPlatform);

#define LOG_SYSCALL_ERROR(syscall)   \
    if (errno != 22) { /* TODO Hugo */ \
        FEATSTD_LOG_ERROR("Semaphore %0xp: ::%s() failed (Error %d)", &mSem, syscall, errno); \
    }
    
    FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 5-18-1: false positive, comma operator is not used")
    FEATSTD_LINT_CURRENT_SCOPE(774, "Violates MISRA C++ 2008 Required Rule 0-1-2: false positive as wrongly identified as comma operator")
    FEATSTD_LINT_CURRENT_SCOPE(948, "Violates MISRA C++ 2008 Required Rule 0-1-9: false positive as wrongly identified as comma operator")
    FEATSTD_LINT_CURRENT_SCOPE(944, "Violates MISRA C++ 2008 Required Rule 0-1-1: false positive as wrongly identified as comma operator")
    FEATSTD_LINT_CURRENT_SCOPE(1013, "false positive, tv_sec and tv_nsec are members of timespec")
    
    // ------------------------------------------------------------------------
    PosixSemaphore::PosixSemaphore()
    {
    }

    // ------------------------------------------------------------------------
    PosixSemaphore::~PosixSemaphore()
    {
    }

    // ------------------------------------------------------------------------
    bool PosixSemaphore::Create(Int initVal)
    {
        FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSemaphore::Create, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
        bool lRc = (0 == ::sem_init(&mSem, 0, initVal));
        if (!lRc) {
            LOG_SYSCALL_ERROR("sem_init");
        }
        return (lRc);
    }

    // ------------------------------------------------------------------------
    bool PosixSemaphore::Obtain(const Ticks &timeout)
    {
        FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSemaphore::Obtain, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
        FEATSTD_LINT_SYMBOL(550, lTimeoutPeriod, "[MISRA C++ Rule 0-1-4] lTimeoutPeriod is not accessed is a false positive")
        FEATSTD_LINT_SYMBOL(550, lTimeoutTime, "[MISRA C++ Rule 0-1-4] lTimeoutTime is not accessed is a false positive")
        FEATSTD_LINT_SYMBOL(550, lNow, "[MISRA C++ Rule 0-1-4] lNow is not accessed is a false positive")
        FEATSTD_LINT_CURRENT_SCOPE(438, "lTimeoutPeriod is used to calculate lTimeoutTime")
        
        // Get current system time
        struct timespec lNow;
        static_cast<void>(::clock_gettime(CLOCK_REALTIME, &lNow));

        /* Convert to Posix time */
        const UInt32 lTimeoutPeriod = timeout.GetTime();
        struct timespec lTimeoutTime;

        lTimeoutTime.tv_sec = lNow.tv_sec + lTimeoutPeriod / 1000;
        lTimeoutTime.tv_nsec = lNow.tv_nsec + (lTimeoutPeriod % 1000) * 1000000U;
        if (lTimeoutTime.tv_nsec >= 1000000000) {
            lTimeoutTime.tv_nsec -= 1000000000U;
            ++lTimeoutTime.tv_sec;
        }

        bool lRc = false;
        do {
            lRc = (0 == ::sem_timedwait(&mSem, &lTimeoutTime));
        } while ((!lRc) && (errno == EINTR));

        if ((!lRc) && (ETIMEDOUT != errno)) {
            LOG_SYSCALL_ERROR("sem_timedwait");
        }

        return (lRc);
    }

    // ------------------------------------------------------------------------
    bool PosixSemaphore::Obtain()
    {
        FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSemaphore::Obtain, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
        bool lRc = false;
        do {
            lRc = (0 == ::sem_wait(&mSem));
        } while ((!lRc) && (errno == EINTR));

        if (!lRc) {
            LOG_SYSCALL_ERROR("sem_wait");
        }
        return (lRc);
    }

    // ------------------------------------------------------------------------
    bool PosixSemaphore::Release()
    {
        FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSemaphore::Release, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
        if (0 != ::sem_post(&mSem)) {
            LOG_SYSCALL_ERROR("sem_post");
            return false;
        }
        return true;
    }

    // ------------------------------------------------------------------------
    bool PosixSemaphore::Destroy()
    {
        FEATSTD_LINT_SYMBOL(1762, FeatStd::Internal::Posix::PosixSemaphore::Destroy, "Violates MISRA C++ 2008 Required Rule 9-3-3: function has side effects and is therefore considered non-const")
        if (0 != ::sem_destroy(&mSem)) {
            LOG_SYSCALL_ERROR("sem_close");
            return false;
        }
        return true;
    }
}}}
