//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_PerfCounter_h)
#define FeatStd_Platform_PerfCounter_h

#include <FeatStd/Platform/Base.h>

#if !defined(FEATSTD_PERFCOUNTER_RESOLUTION)
    /** defines resolution of performance counters in micro seconds
        default is 20 micro seconds. 20 micro seconds resolution enable timings
        with UInt16 up to 1 second and almost 24 hours with UInt32.
        Increase resolution to extend monitoring times, decrease to increase
        accuracy of performance monitoring. */
    #define FEATSTD_PERFCOUNTER_RESOLUTION 20
#endif

#include FEATSTD_PLATFORM_INCLUDE(PerfCounter)

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** Provides high resolution performance counter support.
     */
    struct PerfCounter : private Impl::PerfCounter {
        typedef Impl::PerfCounter Base;

        /**
         * Provides the current time in 20us units. To get the actual time in microseconds,
         * call function @see PerfCounterPlatform::ToMicroSec. This time is only to be
         * used for time measurements.
         * @return The current time in 20us units.
         */
        static UInt32 Now() {
            return Base::Now();
        }

        /** return elapsed time of given performance counter
            @param counter the counter variable setup by @see PerfCounterPlatform::Start
            @return elapsed time in 20us units */
        static UInt32 Elapsed(UInt32 counter) {
            return Base::Elapsed(counter);
        }

        /** return elapsed time of given performance counter for time periods smaller than 1sec
            @param counter the counter variable setup by @see PerfCounterPlatform::Start
            @return elapsed time in 20us units */
        static UInt16 ElapsedShort(UInt32 counter) {
            return static_cast<UInt16>(Elapsed(counter));
        }

        /**
         * Computes the duration of a time period delimited by two 32-bit
         * unsigned integer values.
         *
         * @param start Start time of the period to compute.
         * @param end   End time of the period to compute.
         * @return The duration starting from start and ending at end.
         */
        static UInt32 Duration(UInt32 start, UInt32 end) {
            return Base::Duration(start, end);
        }

        /**
         * Computes the duration of a time period delimited by two 16-bit
         * unsigned integer values.
         *
         * @param start Start time of the period to compute.
         * @param end   End time of the period to compute.
         * @return The duration starting from start and ending at end.
         */
        static UInt32 ShortDuration(UInt16 start, UInt16 end) {
            return Base::ShortDuration(start, end);
        }

        /** Converts time returned by @see PerfCounterPlatform::Elapsed and PerfCounterPlatform::ElapsedShort to us
            @param duration in 20 us units.
            @return time elapsed in us */
        static UInt32 ToMicroSec(UInt32 duration) {
            return duration * FEATSTD_PERFCOUNTER_RESOLUTION;
        }
    };
/// @}
}}
#endif
