#ifndef FeatStd_Platform_ICommNode_h
#define FeatStd_Platform_ICommNode_h

#define FEATSTD_COM_TYPE_NONE (0)
#define FEATSTD_COM_TYPE_TCPIP (1)
#define FEATSTD_COM_TYPE_SERIALPORT (2)
#define FEATSTD_COM_TYPE_FILE_DUMP (3)
#define FEATSTD_COM_TYPE_MEMORY_DUMP (4)

#include <FeatStd/Platform/CommunicationSettings.h>
#include <FeatStd/Platform/Base.h>
namespace FeatStd{
    namespace Internal{
        /// @addtogroup FEATSTD_PLATFORM
        /// @{
        /**<em>ICommNode</em> is an interface for common communication types within FeatStd.
        It abstracts the basic operations of different communication types, so it can be easily replaced by each other.
        Only the configuration structure has to be changed when communication type has changed.
        */
        class ICommNode {
        public:
            /** <em>Configure</em> passes an undefined structure pointer through to concrete communication type. \p
            This will add new settings infromation. To activate these settings Close() an opened connection
            and reopen it with Open() \p
            The structure contains all settings that can be changed by given communication type \p
            For further settings details and the structures @see FeatStd/Platform/CommunicationSettings.h
            @param configData   pointer to settings structure 
            @return             <em>true</em> if copying all data succeeded
                                <em>false</em> if error occurred while copying the settings  */
            virtual bool Configure(void * configData) = 0;
            /** <em>Open</em> opens connection of given communication type without handshake
            @return             <em>true</em>  if opening succeeded
            <em>false</em> if error occurred while opening connection */
            virtual bool Open() = 0;
            /** <em>Open</em> opens connection of given communication type
            @param handshake    handshake function pointer to establish a connection
            @return             <em>true</em>  if opening succeeded
            <em>false</em> if error occurred while opening connection */
            virtual bool Open(ConnectionHandshake handshake) = 0;
            /** <em>Close</em> closes current communication
            @return             <em>true</em> if closing succeeded
                                <em>false</em> if error occurred while closing connection */
            virtual bool Close() = 0;
            /** <em>Connected</em> checks whether connection is established or not
            @return             <em>true</em> if connection is established
                                <em>false</em> if connection is not established  */
            virtual bool Connected() = 0;
            /** <em>DataAvailable</em> checks whether data is available or not
            @param available    out-parameter which is set to true when data is available else it is false
            @return             <em>true</em> if setting available worked
            <em>false</em> if error occurred while retrieving available state  */
            virtual bool DataAvailable(bool& available) = 0;
            /** <em>Read</em> reads given amount of bytes (as a maximum).\p
            Further configuration is set by settings and/or the specific communication type
            @param buffer       buffer in which retrieved data will be stored.
            @param length       Amount of bytes to be read
            @return             <0: Error occurred while reading data \p >=0: Amount of bytes read */
            virtual Int32 Read(UInt8* buffer, UInt32 length) = 0;
            /** <em>Write</em> writes given buffer into the communication stream
            @param buffer       buffer which is sent
            @param length       Amount of bytes to be written
            @return             <0: Error occurred while writing data \p >=0: Amount of bytes written  */
            virtual Int32 Write(const UInt8* buffer, UInt32 length) = 0;
            /** <em>GetConfiguration</em> 
            @return             returns current configuration of  communication node  */
            virtual void const * GetConfiguration() const = 0;
            /** Destructor of <em>ICommNode</em> */
            virtual ~ICommNode(){}
        protected:
            ICommNode(){}
        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(ICommNode);
        };
        /// @}
    }
}
#endif
