//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Generic_GenericMemory_h)
#define FeatStd_Platform_Generic_GenericMemory_h

#include <FeatStd/Platform/Base.h>
#include <FeatStd/Platform/Memory.h>

#include <stdlib.h>
#include <string.h>

namespace FeatStd { namespace Internal { namespace Generic {
/// @addtogroup FEATSTD_PLATFORM_GENERIC
/// @{
/**
* @brief Abstraction layer for memory methods.
*/
struct GenericMemory {
    /**
    * Copies bytes between buffers.
    * @param destination      The destination buffer.
    * @param source           The source buffer.
    * @param count            The count in bytes.
    */
    static void Copy(void* destination, const void* source, SizeType count){
        memcpy(destination, source, static_cast<size_t>(count));
    }

    /**
    * Moves bytes between buffers (std::memmove semantics).
    * @param destination      The destination buffer.
    * @param source           The source buffer.
    * @param count            The count in bytes.
    */
    static void Move(void* destination, const void* source, SizeType count){
        memmove(destination, source, static_cast<size_t>(count));
    }

    /**
    * Sets buffers to a specified value.
    * @param destination      The destination buffer.
    * @param value            The value to set.
    * @param count            The count in bytes.
    */
    static void Set(void* destination, UInt8 value, SizeType count) {
        memset(destination, static_cast<int>(value), static_cast<size_t>(count));
    }

    /**
    * Compares two memory buffers
    * @param l          left-hand buffer
    * @param r          right-hand buffer
    * @param nBytes     number of bytes to compare
    * @return <0: l less than r \p >0: l greater than r, \p 0: l identical to r
    */
    static Int32 Compare(const void *l, const void *r, SizeType nBytes) {
        return static_cast<Int32>(memcmp(l, r, static_cast<size_t>(nBytes)));
    }

    /**
    *  Allocates nbytes memory
    *  @param nbytes number of bytes which have been allocated
    *  @return pointer to memory. 0, if no memory way allocated.
    */
    static void* MemAlloc(SizeType nbytes) {
        return malloc(static_cast<size_t>(nbytes));
    }

    /**
    *  Reallocate memory
    *  functionality according to standard c realloc function
    *  @param p buffer to allocated memory
    *  @param nbytes number of bytes which have been allocated
    *  @return pointer to memory. 0, if no memory allocated.
    */
    static inline void* MemRealloc(void *p, SizeType nbytes) {
        return realloc(p, static_cast<size_t>(nbytes));
    }

    /**
    *  Releases previously allocated memory.
    *  @param p pointer to the buffer to be released
    */
    static void MemFree(void *p) {
        return free(p);
    }

    /**
    *  Estimates the largest available block of memory which is available for allocation.
    *  @return The largest available block in memory in bytes.
    */
    static SizeType MemAvailable() {
        return static_cast<SizeType>(-1);
    }
};
/// @}
}}}

namespace FeatStd { namespace Internal { namespace Impl {
    typedef FeatStd::Internal::Generic::GenericMemory Memory;
}}}
#endif
