//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Generic_GenericMath_h)
#define FeatStd_Platform_Generic_GenericMath_h

#include <FeatStd/Platform/Base.h>
#include <FeatStd/Util/FeatLimits.h>

FEATSTD_LINT_FILE(829, math.h, "maintained for backward compatibility")
FEATSTD_LINT_FILE(829, float.h, "maintained for backward compatibility")
FEATSTD_LINT_FILE(829, stdlib.h, "maintained for backward compatibility")

#include <float.h>
#include <stdlib.h>
#if __cplusplus >= 201103L
#include <cmath>
#else
#include <math.h>
#endif

#ifdef FEATSTD_64BIT_PLATFORM
    #define LLABS llabs
#endif

#define ABS abs

namespace FeatStd { namespace Internal { namespace Generic {
    /** @addtogroup GenericOS
    *  @{
    */

    /**
    * @brief Generic, libc based abstraction layer for maths methods.
    */
    struct GenericMath {
        /**
         * Calculates the square root for FeatStd::Float types.
         * @param f        A non-negative FeatStd::Floating-point value.
         * @return         The square-root of f. If f is negative, SquareRoot returns an indefinite value.
         */
        static inline Float SquareRoot(Float f);

        /**
         * Calculates the square root for double types.
         * @param d        A non-negative FeatStd::Floating-point value.
         * @return         The square-root of d. If d is negative, SquareRoot returns an indefinite value.
         */
        static inline Double SquareRoot(Double d);

        /**
         * Computes the sine of a given value in radians.
         * @param val      A FeatStd::Float value in radians.
         * @return         The sine of the given FeatStd::Float value.
         */
        static inline Float Sine(Float val);

        /**
         * Computes the arcsine of a given value.
         * @param val A float value inside [-1, 1] interval.
         * @return the arcsine of the given value in radians.
         */
        static inline Float ASine(Float val);

        /**
         * Computes the cosine of a given value in radians.
         * @param val      A FeatStd::Float value in radians.
         * @return         The cosine of the given FeatStd::Float value.
         */
        static inline Float Cosine(Float val);

        /**
         * Computes the arccosine of a given value.
         * @param val A float value inside [-1, 1] interval.
         * @return the arccosine of the given value in radians.
         */
        static inline Float ACosine(Float val);

        /**
         * Computes the tangent of a given value in radians.
         * @param val      A FeatStd::Float value in radians.
         * @return         The tangent of the given FeatStd::Float value.
         */
        static inline Float Tangent(Float val);

        /**
         * Computes the arctangent of a given value.
         * @param val A float value.
         * @return the arctangent of the given value in radians.
         */
        static inline Float ATangent(Float val);

        /**
         * Computes the arctangent 2 of the given values.
         * @param y Opposite side.
         * @param x Adjacent side.
         * @return the arctangent 2 of the given values in radians.
         */
        static inline Float ATangent2(Float y, Float x);

        /**
         * Computes a base raised to the power of a exponent.
         * @param base     The base.
         * @param exponent The exponent.
         * @return         The base raised to the power of the exponent.
         */
        static inline Float Power(Float base, Float exponent);

        /**
         * Computes the base 10 logarithm of the value given.
         * @param value    The value to compute the base 2 logarithm from.
         * @return         The base 10 logarithm of the value given.
         */
        static inline Float Log10(Float value);

        /**
         * Computes the absolute value of the value given.
         * @param value    The value to compute the absolute value from.
         * @return         The absolute value of the value given.
         */
        static inline UInt32 Absolute(Int32 value);

        /**
        * Computes the absolute value of the value given.
        * @param value    The value to compute the absolute value from.
        * @return         The absolute value of the value given.
        */
        static inline Float Absolute(Float value);

        /**
        * Computes the largest integer smaller than the value given.
        * @param value    The value to compute the floor value from.
        * @return         The floor value of the value given.
        */
        static inline Float Floor(Float value);

        /**
         * Computes the largest integer smaller than the value given.
         * @param value    The value to compute the floor value from.
         * @return         The floor value of the value given.
         */
        static inline Double Floor(Double value);

        /**
        * Computes the smallest integer larger than the value given.
        * @param value    The value to compute the floor value from.
        * @return         The floor value of the value given.
        */
        static inline Float Ceil(Float value);

        /**
        * Computes the value in radians corresponding to a value in degrees.
        * @param degree   The value in degrees.
        * @return         The value in radians.
        */
        static inline Float DegreeToRadian(Float degree);

        /**
        * Computes the value in degrees corresponding to a value in radians.
        * @param radian   The value in radians.
        * @return         The value in degrees.
        */
        static inline Float RadianToDegree(Float radian);

        /**
        * Compares two FeatStd::Float values.
        * @param a         The first value.
        * @param b         The second value.
        * @param relativeEpsilon, absoluteEpsilon   The accepted inaccuracy on comparison.
        *                                           If 99.999% accuracy is needed then pass a epsilon value of 0.00001.
        * @return          true if the given FeatStd::Float values are equal in respect to the given epsilon, otherwise false.
        */
        static bool FloatAlmostEqual(Float a, Float b, Float relativeEpsilon = EpsilonFloat(), Float absoluteEpsilon = EpsilonFloat3D());

        /** Returns the largest floating point number
            @return largest floating point number */
        static inline Float MaxFloat();

        /** Returns the smallest positive floating point number
            @return smallest floating point number */
        static inline Float MinFloat();

        /** Returns the smallest positive number such that 1.0+number !=1.0
            @return Smallest positive number such that 1.0+number !=1.0 */
        static inline Float EpsilonFloat();

        /** Returns the smallest positive number acceptable as a relative error in math operations
            @return Smallest positive number acceptable as a relative error in math operations */
        static inline Float EpsilonFloat3D();

        /**
        * Returns the remainder of x divided by y.
        * @param x  The division numerator.
        * @param y  The division denominator.
        * @return The remainder of x divided by y.
        */
        static inline Double FMod(Double x, Double y);

        /**
        * Check a Float for being infinite (INF).
        * @param value  The value to check.
        * @return true, if the value is infinite.
        */
        static inline bool IsInfinite(Float value);

        /**
        * Check a Float for being not a number or undefined (NAN, IND).
        * @param value  The value to check.
        * @return true, if the value is undefined / not a number.
        */
        static inline bool IsNan(Float value);

        /*
        * Check a Float for being valid, which means not infinite nor undefined.
        * @param value  The value to check.
        * @return true, if the value is a valid float.
        */
        static inline bool IsValid(Float value);
    };

    /** @}*/ //end of GenericOS

    inline Float GenericMath::SquareRoot(Float f) {
        return static_cast<Float>(sqrtf(static_cast<Float>(f)));
    }

    inline Double GenericMath::SquareRoot(Double d) {
        return static_cast<Double>(sqrt(static_cast<Double>(d)));
    }

    inline Float GenericMath::Sine(Float val) {
        return static_cast<Float>(sinf(static_cast<Float>(val)));
    }

    inline Float GenericMath::ASine(Float val) {
        return static_cast<Float>(asinf(static_cast<Float>(val)));
    }

    inline Float GenericMath::Cosine(Float val) {
        return static_cast<Float>(cosf(static_cast<Float>(val)));
    }

    inline Float GenericMath::ACosine(Float val) {
        return static_cast<Float>(acosf(static_cast<Float>(val)));
    }

    inline Float GenericMath::Tangent(Float val) {
        return static_cast<Float>(tanf(static_cast<Float>(val)));
    }

    inline Float GenericMath::ATangent(Float val) {
        return static_cast<Float>(atanf(static_cast<Float>(val)));
    }

    inline Float GenericMath::ATangent2(Float y, Float x) {
        return static_cast<Float>(atan2f(static_cast<Float>(y), static_cast<Float>(x)));
    }

    inline Float GenericMath::Power(Float base, Float exponent) {
        return static_cast<Float>(pow(static_cast<Float>(base), static_cast<Float>(exponent)));
    }

    inline Float GenericMath::Log10(Float value) {
        return static_cast<Float>(log10(static_cast<Float>(value)));
    }

    inline Float GenericMath::MaxFloat() {
        return FLT_MAX;
    }

    inline Float GenericMath::MinFloat() {
        return FLT_MIN;
    }

    inline Float GenericMath::EpsilonFloat() {
        return FLT_EPSILON;
    }

    inline Float GenericMath::EpsilonFloat3D() {
        return 0.00001F;
    }

    inline UInt32 GenericMath::Absolute(Int32 value) {
        return static_cast<UInt32>(ABS(value));
    }

    inline Float GenericMath::Absolute(Float value) {
        return static_cast<Float>(fabs(static_cast<Float>(value)));
    }

    inline Float GenericMath::Floor(Float value) {
        return static_cast<Float>(floor(static_cast<Float>(value)));
    }

    inline Double GenericMath::Floor(Double value) {
        return static_cast<Double>(floor(static_cast<Double>(value)));
    }

    inline Float GenericMath::Ceil(Float value) {
        return static_cast<Float>(ceil(static_cast<Float>(value)));
    }

    inline Float GenericMath::DegreeToRadian(Float degree)
    {
        return degree * 0.0174532925199432957692369076848F;
    }

    inline Float GenericMath::RadianToDegree(Float radian)
    {
        return radian * 57.295779513082320876798154814105F;
    }

    inline Double GenericMath::FMod(Double x, Double y) {
        return static_cast<Double>(fmod(static_cast<Double>(x), static_cast<Double>(y)));
    }

    inline bool GenericMath::IsInfinite(Float value) {
#if __cplusplus >= 201103L
        return std::isinf(value);
#else
        Float max_value = Internal::Limits<Float>::Max();
        Float min_value = -max_value;
        return !((min_value <= value) && (value <= max_value));
#endif
    }

    inline bool GenericMath::IsNan(Float value) {
#if __cplusplus >= 201103L
        return std::isnan(value);
#else
        FEATSTD_LINT_NEXT_EXPRESSION(777, "Testing floats for equality [MISRA C++ Rule 6-2-2]: Intended in this case.")
        return value != value;
#endif
    }

    inline bool GenericMath::IsValid(Float value) {
        return ((!GenericMath::IsInfinite(value)) && (!GenericMath::IsNan(value)));
    }
}}}

namespace FeatStd { namespace Internal { namespace Impl {
    typedef FeatStd::Internal::Generic::GenericMath Math;
}}}

#endif
