//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_Generic_GenericFile_h)
#define FeatStd_Platform_Generic_GenericFile_h

#include <FeatStd/Platform/Base.h>

#include <stdio.h>

#if defined(FEATSTD_64BIT_PLATFORM) && defined(_MSC_VER)
    #define SEEK _fseeki64
    #define FTELL _ftelli64 
#else
    #define SEEK fseek
    #define FTELL ftell 
#endif

namespace FeatStd { namespace Internal { namespace Generic {
/// @addtogroup GenericOS
///  @{
/**
* @brief Typedef for the file handle. Implemented as a pointer to a FILE struct for now
*/
typedef FILE* FileHandle;

/**
* @brief Incapsulates file handling routines
*/
struct GenericFile
{
    /**
        *  Different seek marks
        */
    enum SeekMark {
        Begin = SEEK_SET,
        Current = SEEK_CUR,
        End = SEEK_END
    };

    /**
        *  The mode (ReadBinary, WriteBinary)
        */
    enum Mode {
        ReadBinary = 1,
        WriteBinary = 2,
        AppendBinary = 3
    };

    /**
        *  @brief Opens the file for binary reading.
        *  To check for open status, use IsFileOpened()
        *  @param filePath The path of the file.
        *  @param mode The mode, default value is ReadBinary
        *  @return The file handle to the opened file.
        */
    static FileHandle Open(const Char* filePath, Mode mode=ReadBinary) {
        switch (mode)
        {
            case ReadBinary:
                return fopen(filePath, "rb");
            case WriteBinary:
                return fopen(filePath, "wb");
            case AppendBinary:
                return fopen(filePath, "ab");
            default:
                return 0;
        }
    }

    /**
        *  @brief To be used after Open() in order to check if the file was successfuly opened
        *  @param handle The file handle that is checked
        *  @return True if file is opened, otherwise false.
        */
    static bool IsFileOpened(FileHandle handle) {
        return handle != 0;
    }

    /**
        *  @brief Opens the file for binary reading
        *  @param file      The file handle of the file
        *  @param buffer    The buffer
        *  @param blockSize The block size
        *  @param count     The count
        *  @return The FeatStd::Int32 value.
        */
    static SizeType Read(FileHandle file, void* buffer,
                        SizeType blockSize, SizeType count) {
        return fread(buffer, blockSize, count, file);
    }

    /**
        *  @brief Writes to the file.
        *  @param file      The file handle of the file
        *  @param buffer    The buffer
        *  @param blockSize The block size
        *  @param count     The count
        *  @return The FeatStd::Int32 value.
        */
    static SizeType Write(FileHandle file, const void* buffer,
                        SizeType blockSize, SizeType count) {
        return fwrite(buffer, blockSize, count, file);
    }

    /**
        *  @brief returns current file pointer position
        *  @param hdl The file handle
        *  @return The current file pointer position.
        */
    static inline OffsetType Tell(FileHandle hdl) {
        return FTELL(hdl);
    }

    /**
        *  @brief Seeks the file by @see offset bytes
        *  @param file   The file handle.
        *  @param offset The offset.
        *  @param mark   The seek mark.
        *  @return True if successful, false otherwise.
        */
    static bool Seek(FileHandle file, OffsetType offset, SeekMark mark) {
        return (SEEK(file, offset, static_cast<int>(mark)) == 0);
    }

    /**
        *  @brief Closes the file associated to the handle and sets the parameter to the null handle
        *  @param handle The file handle that is closed.
        *  @return True if successful, false otherwise
        */
    static bool Close(FileHandle& handle) {
        fclose(handle);
        handle = 0;
        return true;
    }
    };
/// @}
}}}

namespace FeatStd { namespace Internal { namespace Impl {
    typedef FeatStd::Internal::Generic::GenericFile File;
    typedef FeatStd::Internal::Generic::FileHandle FileHandle;
}}}
#endif
