//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_CriticalSectionLocker_h)
#define FeatStd_Platform_CriticalSectionLocker_h

#include <FeatStd/Platform/CriticalSection.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** A critical section locker encapsulates the obtaining and releasing
        of a critical section into an object.
        Because the obtaining is done in the constructor and the releasing
        in the destructor, it is assured that there is no obtain without its
        corresponding release. */
    FEATSTD_LINT_CTORDTOR_EFFECT(FeatStd::Internal::CriticalSectionLocker)
    class CriticalSectionLocker {
    public:
        /** Waits for the ownership of the critical section. The state of
            the critical section is signaled if no thread owns it.
            The method returns when the calling thread is granted
            ownership. */
        CriticalSectionLocker(CriticalSection * cs) :
            mCs(cs),
            mObtained(false)
        {
//            FEATSTD_DEBUG_ASSERT(0 != mCs);
            Obtain();
        }

        /** copy ctor
            Obtains the critical section referenced by locker. */
        CriticalSectionLocker(const CriticalSectionLocker & locker) :
            mCs(locker.mCs),
            mObtained(false)
        {
//            FEATSTD_DEBUG_ASSERT(0 != locker.mCs);
            Obtain();
        }

        /** Releases ownership of the critical section. */
        ~CriticalSectionLocker()
        {
//            FEATSTD_DEBUG_ASSERT(0 != mCs);
            Release();
            mCs = 0;
        }

        /** assignment operator
            Releases the currently held critical section and obtains the one referenced by locker */
        CriticalSectionLocker & operator=(const CriticalSectionLocker & locker)
        {
            if (mCs != locker.mCs) {
                Release();
                mCs = locker.mCs;
                Obtain();
            }
            return *this;
        }

        /** If not already obtained the critical section will be obtained. */
        void Obtain()
        {
            if ((0 != mCs) && !mObtained) {
                mCs->Obtain();
                mObtained = true;
            }
        }

        /** If not already released the critical section will be released. */
        void Release()
        {
            if ((0 != mCs) && mObtained) {
                mCs->Release();
                mObtained = false;
            }
        }

    private:
        CriticalSection* mCs;
        bool mObtained;
    };
/// @}
}}
#endif
