//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_CriticalSection_h)
#define FeatStd_Platform_CriticalSection_h

#include <FeatStd/Platform/Base.h>

#include FEATSTD_PLATFORM_INCLUDE(CriticalSection)

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** A critical section is a light weight synchronization mechanism to
        protect a shared resource from being accessed and manipulated from
        different thread contexts.

        When two or more threads need to access a shared resource at the same
        time, synchronization is a coordination process to ensure that the
        resource is used by only one thread at a time. A critical section is
        a synchronization primitive that allows exclusive access to the shared
        resource to only one thread. If the critical section lock of an object
        is acquired by a thread, the second thread that wants to acquire the
        lock is suspended until the lock is released by the first thread.

        Critical sections are per process for synchronization between threads
        in one process. Critical section objects can be more efficient, as they
        don't require a transition to kernel mode if there's no contention.

        Recursive critical section obtain calls within one thread context will be
        granted. Means if the current thread already owns the critical section, and
        an obtain is called the critical section will be entered.

        @note The critical section should be implemented as non clone able. */
    class CriticalSection : private Impl::CriticalSection {
        typedef Impl::CriticalSection Base;
    public:
        /** Waits for the ownership of the critical section. The state of
            the critical section is signaled if no thread owns it.
            The method returns when the calling thread is granted
            ownership. */
        void Obtain();

        /** Releases ownership of the critical section. */
        void Release();

        /** Attempts to enter the critical section without blocking.
            If the call is successful, the calling thread takes ownership
            of the critical section.
            @return     <em>true</em>   If the critical section is successfully entered or
                                        the current thread already owns the critical section.
                        <em>false</em>  If another thread already owns the critical section.*/
        bool Try();
   };

    /*  --------------------------------------------------------------------
        Obtain
        -------------------------------------------------------------------- */
    inline void CriticalSection::Obtain()
    {
        Base::Obtain();
    }

    /*  --------------------------------------------------------------------
        Release
        -------------------------------------------------------------------- */
    inline void CriticalSection::Release()
    {
        Base::Release();
    }

    /*  --------------------------------------------------------------------
        Try
        -------------------------------------------------------------------- */
    inline bool CriticalSection::Try()
    {
        return Base::Try();
    }
/// @}
}}
#endif
