//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_SharedPointer_h)
#define FeatStd_MemoryManagement_SharedPointer_h

#include <FeatStd/Base.h>

#include <FeatStd/MemoryManagement/Heap.h>
#include <FeatStd/Platform/AtomicOp.h>

/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{
namespace FeatStd { namespace MemoryManagement {
/**
 * @brief  An intrusive reference-counting smart pointer.
 *         If you want your class to be managed by SharedPointer you need to implement two methods:
 *         Retain(); // usually increments a reference count by one
 *         Release(); // usually decrements a reference count by one and destroys the object if the reference count reaches zero.
 */
template<typename T>
class SharedPointer
{
    public:
        /**
         *  Constructor.
         */
        SharedPointer();

        /**
         *  Destructor.
         */
        ~SharedPointer();

        /**
         *  This Constructor takes the ownership of the object.
         *  Eventually the object will be deleted from a SharedPointer instance.
         *  @param pointerToObject Pointer to object that will be maintained by this shared pointer.
         */
        explicit SharedPointer(T* pointerToObject);

        /**
         *  Copy constructor.
         *  Share ownership.
         *  @param rhs SharedPointer to copy.
         */
        SharedPointer(const SharedPointer<T>& rhs);

        /**
         *  Release the instance and reset the pointer to NULL.
         */
        void Release();

        /**
         *  Returns the passed object.
         *  @return The passed object.
         */
        T& GetSharedInstance() const { return *m_pointerToObject; }

        /**
         *  Retrieves a pointer to the shared instance.
         *  @return A pointer to the shared instance
         */
        T* GetPointerToSharedInstance() const { return m_pointerToObject; }

        /**
         *  Retrieves whether the pointer points to 0 or not.
         *  @return True, if pointer points to 0, false otherwise.
         */
        bool PointsToNull() const { return m_pointerToObject == 0; }

        /**
         *  Share ownership.
         *  @param rhs The right hand SharedPointer that is assigned.
         *  @return The result of the operation.
         */
        SharedPointer<T>& operator = (const SharedPointer<T>& rhs);

        /**
         *  Copy constructor (from different, but cast-able type).
         *  Share ownership.
         */
        template <typename T_CastFrom> friend class SharedPointer;
        template <typename T_CastFrom>
        inline SharedPointer(const SharedPointer<T_CastFrom>& rhs);

        /**
         *  Copy from different, but cast-able type.
         *  @param rhs The right hand shared pointer
         *  @return The SharedPointer
         */
        template <typename T_CastFrom>
        SharedPointer<T>& operator = (const SharedPointer<T_CastFrom>& rhs);

        void Set(T* pointer);

        /**
         *  Compare the underlying pointers for equality.
         *  @param rhs The right hand shared pointer
         *  @return True if equal, false if not.
         */
        inline bool operator == (const SharedPointer<T>& rhs) const;

        //bool operator != (const SharedPointer<T>& rhs) const;
        /**
         *  Compare the underlying pointers for inequality.
         *  @param rhs The right hand shared pointer
         *  @return True if unequal, false if equal
         */
        template <typename T_CastFrom>
        inline bool operator != (const SharedPointer<T_CastFrom>& rhs) const;

        /**
         *  Compare the pointers for equality.
         *  @param ptr The second pointer
         *  @return True if equal, false if not.
         */
        inline bool operator == (T* ptr) const;

        /**
         *  Compare the pointers for inequality.
         *  @param ptr The second pointer
         *  @return True if inequal, false if equal.
         */
        inline bool operator != (T* ptr) const;

        /**
         *  Dereferencing.
         *  @return The dereferenced pointer.
         */
        inline T& operator*();

        /**
         *  Dereferencing.
         *  @return The dereferenced pointer.
         */
        inline const T& operator*() const;

        /**
         *  Dereferencing.
         *  Use the methods of the underlying instance like so:
         *  @code sharedPointer->MemberCall() @endcode
         *  @return The dereferenced pointer.
         */
        inline T* operator->() const;

        /**
         *  Dereferencing.
         *  Use the methods of the underlying instance like so:
         *  @code sharedPointer->MemberCall() @endcode
         *  @return The dereferenced pointer.
         */
        inline T* operator->();

    private:
        inline void RetainObject() const;
        inline void ReleaseObject() const;

        T* m_pointerToObject;
};

// inline implementations

template<typename T>
SharedPointer<T>::SharedPointer() : m_pointerToObject(0)
{
}

template<typename T>
SharedPointer<T>::SharedPointer(T* pointerToObject) : m_pointerToObject(pointerToObject)
{
    RetainObject();
}

template<typename T>
SharedPointer<T>::SharedPointer(const SharedPointer<T>& rhs) : m_pointerToObject(rhs.m_pointerToObject)
{
    RetainObject();
}

template<typename T>
template <typename T_CastFrom>
inline SharedPointer<T>::SharedPointer(const SharedPointer<T_CastFrom>& rhs)  : m_pointerToObject(rhs.m_pointerToObject)
{
    RetainObject();
}

template<typename T>
template <typename T_CastFrom>
SharedPointer<T>& SharedPointer<T>::operator = (const SharedPointer<T_CastFrom>& rhs)
{
    if(rhs != *this)
    {
        rhs.RetainObject();
        ReleaseObject();
        m_pointerToObject = rhs.m_pointerToObject;
    }

    return *this;
}

template<typename T>
void SharedPointer<T>::Set(T* object)
{
    if (m_pointerToObject != object) {
        if (object != 0) {
            object->Retain();
        }
        ReleaseObject();
        m_pointerToObject = object;
    }
}

template<typename T>
template <typename T_CastFrom>
bool SharedPointer<T>::operator != (const SharedPointer<T_CastFrom>& rhs) const
{
    return m_pointerToObject != rhs.m_pointerToObject;
}

template<typename T >
SharedPointer<T>& SharedPointer<T>::operator=(const SharedPointer<T>& rhs)
{
    if(&rhs != this)
    {
        rhs.RetainObject();
        ReleaseObject();
        m_pointerToObject = rhs.m_pointerToObject;
    }

    return *this;
}

template<typename T>
SharedPointer<T>::~SharedPointer()
{
    ReleaseObject();
    m_pointerToObject = 0;
}

template<typename T>
bool SharedPointer<T>::operator == (const SharedPointer<T>& rhs) const
{
    return m_pointerToObject == rhs.m_pointerToObject;
}

//template<typename T>
//bool SharedPointer<T>::operator != (const SharedPointer<T>& rhs) const
//{
//    return !operator==(rhs);
//}

template<typename T>
bool SharedPointer<T>::operator == (T* ptr) const
{
    return m_pointerToObject == ptr;
}

template<typename T>
bool SharedPointer<T>::operator != (T* ptr) const
{
    return !operator==(ptr);
}

template<typename T> inline
T& SharedPointer<T>::operator*()
{
    return *m_pointerToObject;
}

template<typename T> inline
const T& SharedPointer<T>::operator * () const
{
    return *m_pointerToObject;
}

template<typename T> inline
T* SharedPointer<T>::operator->() const
{
    return m_pointerToObject;
}

template<typename T> inline
T* SharedPointer<T>::operator->()
{
    return m_pointerToObject;
}

template<typename T>
void SharedPointer<T>::Release()
{
    //Forget any previous instance:
    //This will not affect any other SharedPointer that happens to contain the same underlying instance.
    if(m_pointerToObject != 0)
    {
        ReleaseObject();
        m_pointerToObject = 0;
    }
}

template <typename T> inline
void SharedPointer<T>::RetainObject() const
{
    if(m_pointerToObject != 0) {
        m_pointerToObject->Retain();
    }
}

template <typename T> inline
void SharedPointer<T>::ReleaseObject() const
{
    if(m_pointerToObject != 0) {
        m_pointerToObject->Release();
    }
}

namespace Internal {

/**
  * Function used to determine whether a shared pointer is the only
  * reference to an object.
  * @param sharedPointer The shared pointer that is tested.
  * @return True if this is the only reference. False otherwise.
  */
template <typename T>
inline bool IsSoleReference(const SharedPointer<T>& sharedPointer)
{
    if (sharedPointer != 0) {
        return *sharedPointer->m_retainCount.m_value == 1;
    }
    return true;
}

} // namespace Internal

} // namespace MemoryManagement
} // namespace FeatStd

/*  ----------------------------------------------------------------------------
    Public macros
    ---------------------------------------------------------------------------- */
/** FEATSTD_TYPEDEF_SHARED_POINTER suppresses Lint warning 1516 when the declaration
"typedef <TypeOfBaseClass> SharedPointer" is used in a derived class. The SharedPointer typedef
is available in all classes on purpose to allow usage in a consistent way.
*/
#define FEATSTD_TYPEDEF_SHARED_POINTER( CLASS ) \
    FEATSTD_LINT_SYMBOL(1516, *::SharedPointer, "Typedef to SharedPointer is also available in derived class for consistent usage.") \
    typedef FeatStd::MemoryManagement::SharedPointer<CLASS> SharedPointer

/** FEATSTD_SHARED_POINTER_CREATE_DECLARATION suppresses Lint warning 1511 when a static Create Method that returns a SharedPointer
    is also present in a derived class. The Create method is named the same in all classes on purpose to allow usage in a consistent way.
*/
#define FEATSTD_SHARED_POINTER_CREATE_DECLARATION() \
    FEATSTD_LINT_SYMBOL(1511, *::Create, "Create method that returns SharedPointer is also available in derived class for consistent usage.") \
    static SharedPointer Create()

/**
 *  Enables SharedPointer memory management for the object using this macro.
 *  Release and Retain should be private, so use this macro in the private section of your class.
 */
#ifdef FEATSTD_THREADSAFETY_ENABLED
#define FEATSTD_SHARED_POINTER_DECLARATION_WITH_DISPOSE_CALL(disposeCall)                               \
/* Make class manageable by SharedPointer
 ************************************************************************************************
 Shared object inline implementation for use with SharedPointer
 ************************************************************************************************/      \
/**
 *  Increments the reference count by one.
 */                                                                                                     \
inline void Retain() {                                                                                  \
    (void) FeatStd::Internal::AtomicOp::Inc(m_retainCount.m_value);                                     \
}                                                                                                       \
/**
 *  Decrements the reference count by one.
 *  If the reference count reaches zero the object will be deleted.
 */                                                                                                     \
                                                                                                        \
inline void Release() {                                                                                 \
    if (FeatStd::Internal::AtomicOp::Dec(m_retainCount.m_value) == 0) {                                 \
        disposeCall                                                                                     \
    }                                                                                                   \
}                                                                                                       \
                                                                                                        \
/**
 * Define and initialize the reference counter.
 */                                                                                                     \
class RetainCount {                                                                                     \
public:                                                                                                 \
    RetainCount() {                                                                                     \
        *m_value = 0;                                                                                   \
    }                                                                                                   \
                                                                                                        \
    FeatStd::Internal::AtomicOp::Atomic m_value;                                                        \
                                                                                                        \
private:                                                                                                \
    FEATSTD_MAKE_CLASS_UNCOPYABLE(RetainCount);                                                         \
} m_retainCount;                                                                                        \
                                                                                                        \
/* Retain and Release should be private, but SharedPointer needs access to it */                        \
template <typename T> friend class FeatStd::MemoryManagement::SharedPointer;                            \
template <typename T> friend bool FeatStd::MemoryManagement::Internal::IsSoleReference(                 \
    const FeatStd::MemoryManagement::SharedPointer<T>& sharedPointer)                                   \
/***************************************************************************************************/
#else
#define FEATSTD_SHARED_POINTER_DECLARATION_WITH_DISPOSE_CALL(disposeCall)                               \
/* Make class manageable by SharedPointer
 ************************************************************************************************
 Shared object inline implementation for use with SharedPointer
 ************************************************************************************************/      \
/**
 *  Increments the reference count by one.
 */                                                                                                     \
inline void Retain() {                                                                                  \
    ++(*m_retainCount.m_value);                                                                         \
}                                                                                                       \
/**
 *  Decrements the reference count by one.
 *  If the reference count reaches zero the object will be deleted.
 */                                                                                                     \
                                                                                                        \
inline void Release() {                                                                                 \
    if ((--(*m_retainCount.m_value)) == 0) {                                                            \
        disposeCall                                                                                     \
    }                                                                                                   \
}                                                                                                       \
                                                                                                        \
/**
 * Define and initialize the reference counter.
 */                                                                                                     \
class RetainCount {                                                                                     \
public:                                                                                                 \
    RetainCount() {                                                                                     \
        *m_value = 0;                                                                                   \
    }                                                                                                   \
                                                                                                        \
    class RetainCountValue {                                                                            \
    public:                                                                                             \
        FeatStd::UInt32& operator*() {                                                                           \
            return m_value;                                                                             \
        }                                                                                               \
    private:                                                                                            \
     FeatStd::UInt32 m_value;                                                                                 \
    } m_value;                                                                                          \
                                                                                                        \
private:                                                                                                \
    FEATSTD_MAKE_CLASS_UNCOPYABLE(RetainCount);                                                         \
} m_retainCount;                                                                                        \
                                                                                                        \
/* Retain and Release should be private, but SharedPointer needs access to it */                        \
template <typename T> friend class FeatStd::MemoryManagement::SharedPointer;                            \
template <typename T> friend bool FeatStd::MemoryManagement::Internal::IsSoleReference(                 \
    const FeatStd::MemoryManagement::SharedPointer<T>& sharedPointer)                                   \
/***************************************************************************************************/

#endif

#define FEATSTD_SHARED_POINTER_DECLARATION()                                                            \
    FEATSTD_SHARED_POINTER_DECLARATION_WITH_DISPOSE_CALL(FEATSTD_DELETE(this);)

/// @}
#endif
