//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_Operators_h)
#define FeatStd_MemoryManagement_Operators_h

#include <FeatStd/Base.h>

#include <FeatStd/Util/Traits.h>
#include <FeatStd/MemoryManagement/PlatformHeap.h>
#include <FeatStd/MemoryManagement/PlacementNew.h>
#include <FeatStd/MemoryManagement/MemoryUtil.h>
#include <FeatStd/Platform/Memory.h>

#if defined(FEATSTD_MEMORYPOOL_ENABLED)
#   include <FeatStd/MemoryPool/MemoryPoolDebug.h>
#endif

namespace FeatStd { namespace MemoryManagement { namespace Internal {
    /// @addtogroup FEATSTD_MEMORYMANAGEMENT
    /// @{

    // =========================================================================

    /**
        *  Helper class to create an array of Non-POD objects
        */
    template<typename T, bool podType = false> struct ArrayCreator {

        /** construct array items 
            @param p pointer to buffer
            @param nItems number of items in the array
            @return pointer to constructed items (same to p) */
        static T* ConstructArray(void *p, SizeType nItems) {
            T *obj = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                Construct(obj + i);
            }
            return obj;
        }

        /**
            *  Creates (i.e. allocates & constructs) an array of objects of type T
            *  @param nItems number of items in the array
            *  @return pointer to the newly allocated array with constructed objects
            */
        static T* Create(SizeType nItems) {
            T *obj = reinterpret_cast<T*>(PlatformHeap::AllocArray(ArrayByteSize<T>(1), nItems));
            if (obj != 0) {
                (void) ConstructArray(obj, nItems);
            }
            return obj;
        }

        /** construct array items 
            @param p pointer to buffer
            @param nItems number of items in the array
            @param a0 inital value of array items
            @return pointer to constructed items (same to p) */
        template<typename A0> 
        static T* ConstructArray(void *p, SizeType nItems, const A0 &a0) {
            T *obj = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                ConstructObject<T>(obj + i, a0);
            }
            return obj;
        }

        template<typename A0, typename A1> 
        static T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1) {
            T *obj = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                ConstructObject<T>(obj + i, a0, a1);
            }
            return obj;
        }

        template<typename A0, typename A1, typename A2> 
        static T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1, const A2 &a2) {
            T *obj = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                ConstructObject<T>(obj + i, a0, a1, a2);
            }
            return obj;
        }

        template<typename A0, typename A1, typename A2, typename A3> 
        static T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1, const A2 &a2, const A3 &a3) {
            T *obj = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                ConstructObject<T>(obj + i, a0, a1, a2, a3);
            }
            return obj;
        }

        template<typename A0, typename A1, typename A2, typename A3, typename A4> 
        static T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1, const A2 &a2, const A3 &a3, const A4 &a4) {
            T *obj = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                ConstructObject<T>(obj + i, a0, a1, a2, a3, a4);
            }
            return obj;
        }

        /**
            *  Creates (i.e. allocates & constructs) an array of objects of type T
            *  @param nItems number of items in the array
            *  @param val array elements will be pre-initialized with val
            *  @return pointer to the newly allocated array with constructed objects
            */
        static T* Create(SizeType nItems, const T& val) {
            T *obj = reinterpret_cast<T*>(PlatformHeap::AllocArray(ArrayByteSize<T>(1), nItems));
            if (obj != 0) {
                (void) ConstructArray(obj, nItems, val);
            }
            return obj;
        }

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
        static T* Create(FEATSTD_MEMORYPOOL_DECL_CONTEXT(UInt32 nItems, context), UInt8 attrib) {
            T *obj = reinterpret_cast<T*>(PlatformHeap::AllocArray(ArrayByteSize<T>(1), FEATSTD_MEMORYPOOL_PASS_CONTEXT(nItems, context), attrib));
            if (obj != 0) {
                (void) ConstructArray(obj, nItems);
            }
            return obj;
        }

        static T* Create(UInt32 nItems, FEATSTD_MEMORYPOOL_DECL_CONTEXT(const T& val, context), UInt8 attrib) {
            T *obj = reinterpret_cast<T*>(PlatformHeap::AllocArray(ArrayByteSize<T>(1), FEATSTD_MEMORYPOOL_PASS_CONTEXT(nItems, context), attrib));
            if (obj != 0) {
                (void) ConstructArray(obj, nItems, val);
            }
            return obj;
        }
#endif
    };

    // =========================================================================

    /**
        *  Helper class to create an array of POD objects
        */
    template<typename T> struct ArrayCreator<T, true> {

        static inline T* ConstructArray(void *p, SizeType) {
            return static_cast<T*>(p);
        }

        /**
            *  Creates (i.e. allocates) an array of objects of type T where T is a POD type.
            *  @param nItems number of items in the array
            *  @return pointer to the newly allocated array
            */
        static inline T* Create(SizeType nItems) {
            // POD arrays are allocated without the overhead of non-POD arrays
            return reinterpret_cast<T*>(PlatformHeap::Alloc(ArrayByteSize<T>(nItems)));
        }

        template<typename A0> 
        static inline T* ConstructArray(void *p, SizeType nItems, const A0 &a0) {
            T *t = static_cast<T*>(p);
            for (SizeType i = 0; i < nItems; ++i) {
                t[i] = a0;
            }
            return t;
        }

        /**
            *  Creates (i.e. allocates) an array of objects of type T where T is a POD type.
            *  @param nItems number of items in the array
            *  @param val array items will be pre-initialized with val
            *  @return pointer to the newly allocated array
            */
        static inline T* Create(SizeType nItems, const T &val) {
            T *t = Create(nItems);
            if (t != 0) {
                (void) ConstructArray(t, nItems, val);
            }
            return t;
        }

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
        static inline T* Create(FEATSTD_MEMORYPOOL_DECL_CONTEXT(UInt32 nItems, context), UInt8 attrib) {
            return reinterpret_cast<T*>(PlatformHeap::Alloc(FEATSTD_MEMORYPOOL_PASS_CONTEXT(ArrayByteSize<T>(nItems), context), attrib));
        }

        static T* Create(UInt32 nItems, FEATSTD_MEMORYPOOL_DECL_CONTEXT(const T& val, context), UInt8 attrib) {
            T *t = Create(FEATSTD_MEMORYPOOL_PASS_CONTEXT(nItems, context), attrib);
            if (t != 0) {
                (void) ConstructArray(t, nItems, val);
            }
            return t;
        }
#endif
    };

    // =========================================================================

    template<typename T, bool isPodType = false> struct ArrayDestroyer {

        static void DestructArray(T *p, SizeType nItems) {
            while (nItems > 0) {
                --nItems;
                (p + nItems)->~T();
            }
        }

        static void Destroy(T *object) {
            SizeType nItems;
            SizeType itemSize;
            typedef typename FeatStd::Internal::PureType<T>::Type CleanType;
            PlatformHeap::GetArrayInfo(const_cast<CleanType*>(object), &itemSize, &nItems);
            Destruct(object, itemSize, nItems);
            PlatformHeap::FreeArray(const_cast<CleanType*>(object));
        }
    };

    // =========================================================================

    template<typename T> struct ArrayDestroyer<T, true> {
        inline static void DestructArray(T *, SizeType) {
        }

        static void inline Destroy(T *object) {
            PlatformHeap::Free(const_cast<void*>(reinterpret_cast<const void*>(object)));
        }
    };

}}} // namespace Internal



namespace FeatStd { namespace MemoryManagement { 
    /**
        *  Destructs & frees the given object in the specified heap
        *  @param object the dynamically allocated object to be destroyed
        */
    template<typename T> static inline void Destroy(T *object) {
        if (object != 0) {
            Destruct(object);
            typedef typename FeatStd::Internal::PureType<T>::Type CleanType;
            PlatformHeap::Free(const_cast<CleanType*>(object));
        }
    }

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
#if defined(FEATSTD_MEMORYPOOL_LEAK_DETECTION_ENABLED)
    template<typename T> inline T* CreateArray(FEATSTD_MEMORYPOOL_DECL_CONTEXT(UInt32 nItems, context)) {
        using FeatStd::TypeTraits::Internal::TypeTrait;
        return Internal::ArrayCreator<T, TypeTrait<T>::IsPodType>::Create(FEATSTD_MEMORYPOOL_PASS_CONTEXT(nItems, context), UInt8(MemAttrib::Transient));
    }
#endif

    template<typename T> inline T* CreatePermanentArray(FEATSTD_MEMORYPOOL_DECL_CONTEXT(UInt32 nItems, context)) {
        using FeatStd::TypeTraits::Internal::TypeTrait;
        return Internal::ArrayCreator<T, TypeTrait<T>::IsPodType>::Create(FEATSTD_MEMORYPOOL_PASS_CONTEXT(nItems, context), UInt8(MemAttrib::Permanent));
    }

    template<typename T> inline T* CreatePresetArray(UInt32 nItems, FEATSTD_MEMORYPOOL_DECL_CONTEXT(const T &preset, context), UInt8 attrib = MemAttrib::Transient) {
        using FeatStd::TypeTraits::Internal::TypeTrait;
        return Internal::ArrayCreator<T, TypeTrait<T>::IsPodType>::Create(nItems, FEATSTD_MEMORYPOOL_PASS_CONTEXT(preset, context), attrib);
    }

#endif

    /**
        *  Creates (i.e. allocates and constructs) an object array of Type T with given number of items
        *  @param nItems number of items in the array
        *  @return array of nItem objects
        */
    template<typename T> inline T* CreateArray(SizeType nItems) {
        using FeatStd::TypeTraits::Internal::TypeTrait;
        return Internal::ArrayCreator<T, TypeTrait<T>::IsPodType>::Create(nItems);
    }

    /**
        *  Creates (i.e. allocates and constructs) an object array of Type T with given number of items
        *  @param nItems number of items in the array
        *  @param preset array elements will be initialized to preset
        *  @return array of nItem objects
        */
    template<typename T> inline T* CreateArray(SizeType nItems, const T &preset) {
        using FeatStd::TypeTraits::Internal::TypeTrait;
        return Internal::ArrayCreator<T, TypeTrait<T>::IsPodType>::Create(nItems, preset);
    }

    /**
        *  Destructs & frees the given object array in the specified heap
        *  @param object the dynamically allocated object array to be destroyed
        */
    template<typename T> static inline void DestroyArray(T *object) {
        if (object != 0) {
            using FeatStd::TypeTraits::Internal::TypeTrait;
            Internal::ArrayDestroyer<T, TypeTrait<T>::IsPodType>::Destroy(object);
        }
    }


    /// @{
    /** construct array in given buffer p 
        @tparam T array item type
        @param p the pointer to the buffer 
        @param nItems the number of items in the array 
        @return pointer to constructed array (equal to p) */
    template<typename T> 
    inline T* ConstructArray(void *p, SizeType nItems) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        return (p != 0) ? Internal::ArrayCreator<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArray(p, nItems) : 0;
    }

    /** construct array in given buffer p 
        @tparam T array item type
        @param p the pointer to the buffer 
        @param nItems the number of items in the array 
        @param a0 argument to ctor of each array item
        @return pointer to constructed array (equal to p) */
    template<typename T, typename A0> 
    inline T* ConstructArray(void *p, SizeType nItems, const A0 &a0) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        return (p != 0) ? Internal::ArrayCreator<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArray(p, nItems, a0) : 0;
    }

    /** construct array in given buffer p 
        @tparam T array item type
        @param p the pointer to the buffer 
        @param nItems the number of items in the array 
        @param a0 argument to ctor of each array item
        @param a1 argument to ctor of each array item
        @return pointer to constructed array (equal to p) */
    template<typename T, typename A0, typename A1> 
    inline T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        return (p != 0) ? Internal::ArrayCreator<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArray(p, nItems, a0, a1) : 0;
    }

    /** construct array in given buffer p 
        @tparam T array item type
        @param p the pointer to the buffer 
        @param nItems the number of items in the array 
        @param a0 argument to ctor of each array item
        @param a1 argument to ctor of each array item
        @param a2 argument to ctor of each array item
        @return pointer to constructed array (equal to p) */
    template<typename T, typename A0, typename A1, typename A2> 
    inline T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1, const A2 &a2) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        return (p != 0) ? Internal::ArrayCreator<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArray(p, nItems, a0, a1, a2) : 0;
    }

    /** construct array in given buffer p 
        @tparam T array item type
        @param p the pointer to the buffer 
        @param nItems the number of items in the array 
        @param a0 argument to ctor of each array item
        @param a1 argument to ctor of each array item
        @param a2 argument to ctor of each array item
        @param a3 argument to ctor of each array item
        @return pointer to constructed array (equal to p) */
    template<typename T, typename A0, typename A1, typename A2, typename A3> 
    inline T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1, const A2 &a2, const A3 &a3) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        return (p != 0) ? Internal::ArrayCreator<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArray(p, nItems, a0, a1, a2, a3) : 0;
    }

    /** construct array in given buffer p 
        @tparam T array item type
        @param p the pointer to the buffer 
        @param nItems the number of items in the array 
        @param a0 argument to ctor of each array item
        @param a1 argument to ctor of each array item
        @param a2 argument to ctor of each array item
        @param a3 argument to ctor of each array item
        @param a4 argument to ctor of each array item
        @return pointer to constructed array (equal to p) */
    template<typename T, typename A0, typename A1, typename A2, typename A3, typename A4> 
    inline T* ConstructArray(void *p, SizeType nItems, const A0 &a0, const A1 &a1, const A2 &a2, const A3 &a3, const A4 &a4) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        return (p != 0) ? Internal::ArrayCreator<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::ConstructArray(p, nItems, a0, a1, a2, a3, a4) : 0;
    }
    /// @}

    /** Destruct array items in the given array 
        note - array items only get destructed, no memory will be freed
        @tparam T the array item type
        @param p the pointer to the array
        @param nItems the number of items to destruct */
    template<typename T> inline void DestructArray(T *p, SizeType nItems) {
        FEATSTD_DEBUG_ASSERT(p != 0);
        if (p != 0 && nItems > 0) {
            Internal::ArrayDestroyer<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::DestructArray(p, nItems);
        }
    }

    /// @}
}}
#endif
