//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "EventSource.h"

#include <FeatStd/Event/EventListener.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Util/StaticObject.h>

namespace FeatStd {

EventSource::EventSource()
    :
    m_notificationDepth(0)
{
}

bool EventSource::AddEventListener(EventListener* eventListener)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
    eventListener->Obtain();
#endif

    bool isSuccessful = true;
    if (IsListenerRemovalAllowed()) {
        isSuccessful = m_eventListeners.Append(eventListener);
    }
    else {
        isSuccessful = m_pendingEventListeners.Append(eventListener);
    }

    return isSuccessful;
}

bool EventSource::RemoveEventListener(EventListener* eventListener, bool waitForListenerRelease)
{
    bool isSuccessful = false;

#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

    if (IsListenerRemovalAllowed()) {
        isSuccessful = m_eventListeners.Remove(eventListener);
    }
    else {
        for (EventListeners::Iterator it = m_eventListeners.Begin(); it != m_eventListeners.End(); ++it) {
            if (eventListener == *it) {
                *it = 0;
                isSuccessful = true;
            }
        }

        isSuccessful = m_pendingEventListeners.Remove(eventListener) && isSuccessful;
    }

#ifdef FEATSTD_THREADSAFETY_ENABLED
    eventListener->Release();
    lock.Release();
    if (waitForListenerRelease) {
        eventListener->WaitForRelease();
    }
#else
    FEATSTD_UNUSED(waitForListenerRelease);
#endif

    return isSuccessful;
}

void EventSource::DispatchEvent(const Event& event)
{
    // EventSource::DispatchEvent is not allowed to lock the critical section during EventListener::OnEvent itself.
    //
    // Reason (problem 1):
    // Thread 1 triggers EventSource::DispatchEvent (locking sc1 => GetCriticalSection) and
    // locks an additional critical section (let's call it sc2) while in EventListener::OnEvent.
    // Before Thread 1 calls EventSource::DispatchEvent another thread (let's call it thread 2) locks already sc2.
    // Therefore, thread 1 has to wait until thread 2 releases sc2. But, if thread 2 now triggers a call of EventSource::DispatchEvent it
    // needs to lock sc1 which is already locked by thread 1. In this case we end in a deadlock situation.
    //
    // Solution:
    // By unlocking sc1 before calling EventListener::OnEvent and lock it again afterward we can easily prevent the deadlock situation and thread 2
    // can continue the notification, release afterward sc2 to allow thread 1 to continue its notification.
    //
    // Additional problem in case of destruction of a listener (problem 2):
    // Each listener has to be removed as listener before destruction.
    // So, if thread 1 is currently in EventListener::OnEvent and thread 2 decides to destruct this listener then the EventListener call is not blocked.
    // Thread 2 will return before thread 1 is finished with the call of EventListener::OnEvent and destroy the listener.
    //
    // Solution:
    // The EventListener will be informed to keep the instance alive within the scope of the EventListener method calls Obtain and Release.
    // The EventListener method WaitForRelease has to be called before destroying the EventListener instance. For convenience the
    // EventListener WaitForRelease method is called in EventSource::EventListener if waitForListenerRelease is true.
    // Since the EventListener has to be removed from the EventSource before its destruction the thread safety is obtained without further changes
    // than the implementation of the methods EventListener::Obtain, EventListener::Release and EventListener::WaitForRelease
    // in a thread safe way using atomic ops.
    //
    // Remaining deadlock scenario (unpreventable by framework => has to be prevented in application code):
    // We consider problem 1 and add the removal of a listener to that scenario while thread 2 is calling EventListener::OnEvent.
    // In that case thread 1 will wait for cs2 in EventListener::OnEvent and has sc3 locked. Thread 2 has locked cs2 and waits for cs3.
    // This scenario cannot be prevented by the framework because if thread 2 is not waiting for cs3 it will cause a crash of thread 1.
    // Therefore such a scenario has to be prevented in the application code. To do so the applications is not allowed to remove the listener while still
    // holding the lock of another critical section that may be locked by another listener.

#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    NotificationLoopBegin();
    for (EventListeners::Iterator it = m_eventListeners.Begin(); it != m_eventListeners.End(); ++it) {
        EventListener* eventListener = *it;
        if (0 != eventListener) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            eventListener->Obtain();
            lock.Release();
#endif
            const EventResult::Enum eventResult = eventListener->OnEvent(event);
#ifdef FEATSTD_THREADSAFETY_ENABLED
            lock.Obtain();
            eventListener->Release();
#endif

            if (EventResult::Stop == eventResult) {
                break;
            }
        }
    }

    NotificationLoopEnd();
}

#ifdef FEATSTD_THREADSAFETY_ENABLED
static Internal::CriticalSection& EventSourceCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(Internal::CriticalSection, s_eventSourceCriticalSection);
    return s_eventSourceCriticalSection;
}
static Internal::CriticalSection& s_forceInitEventSourceCriticalSection = EventSourceCriticalSection();

Internal::CriticalSection& EventSource::GetCriticalSection() const
{
    return EventSourceCriticalSection();
}
#endif

void EventSource::NotificationLoopEnd()
{
    --m_notificationDepth;
    if (IsListenerRemovalAllowed()) {
        static_cast<void>(m_eventListeners.Remove(0, true)); // remove listeners which were removed during notification
        // add listeners which were added during notification
        for (EventListeners::Iterator it = m_pendingEventListeners.Begin(); it != m_pendingEventListeners.End(); ++it) {
            static_cast<void>(m_eventListeners.Append(*it));
        }
        m_pendingEventListeners.Clear();
    }
}

}
