//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_EVENTLISTENER_H)
#define FEATSTD_EVENTLISTENER_H

#include <FeatStd/Config.h>
#include <FeatStd/Event/EventResult.h>
#include <FeatStd/Util/ConvenienceMacros.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED 
#include <FeatStd/Platform/Thread.h>
#include <FeatStd/Platform/AtomicOp.h>
#endif

namespace FeatStd {

    class Event;
    /// @addtogroup FEATSTD_EVENT
    /// @{

    /**
    * @brief Event Listeners are used to dispatch events via an EventSource.
    *
    * Events are dispatched to registered Event Listeners.
    * \see EventSource::AddEventListener
    */
    class EventListener {
    public:
        /**
         *  Destructor
         */
        virtual ~EventListener() {};

        /**
         *  Notifies the event listener about the given event.
         *  @param  event  The event to notify the event listener about.
         *  @return        An EventResult for the caller to process.
         */
        virtual EventResult::Enum OnEvent(const Event& event) = 0;

#ifdef FEATSTD_THREADSAFETY_ENABLED
        /**
         *  The Obtain method will be called in order to prevent destruction before removing the event listener
         *  and is also able to prevent removing the event listener from its event source whilst an OnEvent is ongoing.
         *  The macro FEATSTD_EVENTLISTENER_STANDARD_THREADSAFETY provides the basic functionality which has to be available.
         *
         *  IMPORTANT NOTE:
         *  During OnEvent the EventSource holds no lock of any critical section to prevent deadlock situations.
         *  The EventListener implementation is responsible to prevent the destruction of the EventListener instance
         *  while OnEvent is called. 
         */
        virtual void Obtain() const {}

        /**
         *  The Obtain method will be called before OnEvent in a thread safe way.
         */
        virtual void Release() const {}

        /**
         *  This method has to wait for the last call of Release:
         *
         *  IMPORTANT NOTE:
         *  WaitForRelease must be called in the destructor of any EventListener derived class to ensure thread safety.
         *  If an obtain counter is used to prevent destruction of the EventListener instance during an OnEvent call
         *  then the WaitForRelease is only allowed to return as soon as the last Release call has been performed.
         */
        virtual void WaitForRelease() const {};
#endif

    protected:
        FEATSTD_MAKE_CLASS_STATIC(EventListener);
    };
/// @}
}
#ifdef FEATSTD_THREADSAFETY_ENABLED 
/**
* The macro FEATSTD_EVENTLISTENER_STANDARD_THREADSAFETY provides the basic removal during OnEvent thread safety.
* It can be used for a default implementation if necessary. 
* It uses FeatStd::Internal::Thread and FeatStd::Internal::AtomicOp as classes and therefore needs includes for both.
* The visibility scope after the block is not guaranteed to be a specified one. It is advised to set the visibility scope after
* calling this macro or call the macro at the end of an EventListener implementation.
*/
#define FEATSTD_EVENTLISTENER_STANDARD_THREADSAFETY \
public: \
    virtual void Obtain() const override { static_cast<void>(FeatStd::Internal::AtomicOp::Inc(m_obtainCounter)); } \
    virtual void Release() const override { static_cast<void>(FeatStd::Internal::AtomicOp::Dec(m_obtainCounter)); } \
    virtual void WaitForRelease() const override\
    { \
        while (!FeatStd::Internal::AtomicOp::TestAndSet(m_obtainCounter, 0, 0)) { \
            FeatStd::Internal::Thread::Sleep(5); \
        } \
    } \
private: \
    mutable FeatStd::Internal::AtomicOp::Atomic m_obtainCounter
#else
#define FEATSTD_EVENTLISTENER_STANDARD_THREADSAFETY 

#endif 
 

#endif
