//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if (!defined(FeatStd_Diagnostics_SystemMemoryStatistic_h)) && (!defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS))
#define FeatStd_Diagnostics_SystemMemoryStatistic_h

#include <FeatStd/Base.h>
#include <FeatStd/Util/StaticObject.h>

FEATSTD_LINT_FILE(967, SystemMemoryStatistic.h, "false positive - file has a include guard")

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace FeatStd { namespace MemoryManagement {
    class PlatformHeap;
}}

namespace FeatStd { namespace Diagnostics {
#ifdef FEATSTD_THREADSAFETY_ENABLED
    namespace Internal {

        extern FeatStd::Internal::CriticalSection& GetAllocationInfoCs();
    }
#endif

/// @addtogroup FEATSTD_DIAGNOSTICS
/// @{
    /**
     * SystemMemoryStatistic provides live statistics regarding system memory usage.
     *
     * To enable statistics FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED must be defined. If so, the class can
     *  provide the following information:
     * - number of active memory allocations.
     * - total size of allocated memory.
     * - maximum size of one allocated memory item.
     *
     * Assuming correct hints were provided before each memory allocation, the next statistics
     *  can be also retrieved:
     * - same as above, but for each unique requested data type (e.g. Bitmaps, VertexBuffers etc).
     * - the file name and line number where the memory was allocated.
     *
     * Note: The statistics do not include memory allocated by the SystemMemoryStatistic class itself
     * (around 16 bytes for each not yet freed allocated memory).
     */
    class SystemMemoryStatistic
    {
        friend class FeatStd::MemoryManagement::PlatformHeap;

        public:
            struct AllocationInfo {
                SizeType m_size;
#if defined(FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED)
                const Char* m_fileName;
                Int32 m_line;
#endif
                void* m_userData;
            };
            /**
             *  Set hints regarding next memory allocation.
             *
             *  Hints are automatically provided by FEATSTD_NEW(_ARRAY(_T)) macros and by
             *   Heap container allocators. If the application uses directly PlatformHeap
             *   to allocate memory, it could provide hints.
             *
             *  Automatic generation of file name and line number hints can be enabled with
             *   definition of FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED. Note: This leads
             *   to increased memory consumption.
             *
             *  Template typename specifies the data type of the next memory allocation.
             *  @param fileName Name of the file where the allocation occurred.
             *  @param line Line number where the allocation occurred.
             *  @param userData A pointer to the userData, default = 0.
             */
            template<typename T>
            static void HintNextAllocation(const Char* fileName, Int32 line, void* userData = 0);

            /**
             * Get number of memory allocations for given data type.
             *
             * @return Number of memory allocations for given data type.
             */
            template<typename T>
            static SizeType GetAllocationCount() { return GetAllocationInfoCount(GetAllocationInfoList<T>()); }

            /**
             * Get total number of memory allocations.
             *
             * @return Total number of memory allocations.
             */
            static SizeType GetAllocationCount();

            /**
             * Get statistic for one memory allocation of the given type. Use GetAllocationCount<T>() to
             * obtain the upper bound of valid allocation statistic indices.
             *
             * @param index The index of the memory allocation.
             * @return Pointer to a AllocationInfo structure.
             */
            template<typename T>
            static const AllocationInfo* GetAllocationInfo(SizeType index) { return GetAllocationInfo(GetAllocationInfoList<T>(), index); }

            /**
             * Get statistic for one memory allocation. Use GetAllocationCount() to
             * obtain the upper bound of valid allocation statistic indices.
             *
             * @param index The index of the memory allocation.
             * @return Pointer to a AllocationInfo structure.
             */
            static const AllocationInfo* GetAllocationInfo(SizeType index);

            /**
             * Get current memory usage in bytes for given data type.
             *
             * @return Current memory usage in bytes for given data type.
             */
            template<typename T>
            static SizeType GetCurrentMemoryUsage() { return GetUsage(GetAllocationInfoList<T>()); }

            /**
             * Get current total memory usage in bytes.
             *
             * @return Current total memory usage in bytes.
             */
            static SizeType GetCurrentMemoryUsage();

            /**
             * Get peak memory usage in bytes for given data type.
             *
             * @return Peak memory usage in bytes for given data type.
             */
            template<typename T>
            static SizeType GetPeakMemoryUsage() { return GetUsagePeak(GetAllocationInfoList<T>()); }

            /**
             * Get peak total memory usage in bytes.
             *
             * @return Peak total memory usage in bytes.
             */
            static SizeType GetPeakMemoryUsage();

            /**
             * Get maximum size of one allocated item for given data type.
             *
             * @return Maximum size of one allocated item for given data type.
             */
            template<typename T>
            static SizeType GetMaximumItemSize() { return GetMaxItemUsage(GetAllocationInfoList<T>()); }

            /**
             * Get maximum size of one allocated item.
             *
             * @return Maximum size of one allocated item.
             */
            static SizeType GetMaximumItemSize();

            /**
             * Reset current statistics.
             */
            static void Reset();

            typedef void (*OnBeforeDestroyStatisticsCallback)(void*);

            /**
             * Set statistics callback function.
             *
             * The callback function will be called before exiting the application when the statistics are destroyed.
             *  This callback is the last chance to get system memory statistics.
             *
             *  @param cb           The notification callback function.
             *  @param userData     The user data that is sent as parameter to the callback function.
             */
            static void SetOnBeforeDestroyStatisticsCallback(OnBeforeDestroyStatisticsCallback cb, void* userData) { CallbackFunc() = cb; CallbackData() = userData; }

            /**
             *  Retrieve statistics callback function.
             *
             *  @return The notification callback function.
             */
            static OnBeforeDestroyStatisticsCallback GetOnBeforeDestroyStatisticsCallback() { return CallbackFunc(); }

        private:
            class AllocationInfoList;
            class AllocationInfoListList;
            friend class ForceStaticInitHelper;

            struct AllocationInfoInternal{
                AllocationInfo m_info;
                AllocationInfoList* m_list;
            };

            //methods used by PlatformHeap to notice memory allocations/frees
            static SizeType GetAllocationInfoSize();

            static void* OnMemoryAllocated(void* location, SizeType size);
            static void* OnMemoryFreed(void* location);
            static void* GetAllocationInfoLocation(void* location);

            //Methods that create static objects before they are used for the first time.
            template <typename T>
            static AllocationInfoList* GetAllocationInfoList();
            static AllocationInfoListList& GlobalAllocationInfoListList();
            static AllocationInfoInternal& NextAllocationInfo();
            static bool& IgnoreAllocations();
            static OnBeforeDestroyStatisticsCallback& CallbackFunc();
            static void*& CallbackData();

            //Helper methods to create and query an AllocationMetaInfoList
            static AllocationInfoList* CreateAllocationInfoList();
            static SizeType GetAllocationInfoCount(const AllocationInfoList* list);
            static const AllocationInfo* GetAllocationInfo(const AllocationInfoList* list, SizeType index);
            static SizeType GetUsage(const AllocationInfoList* list);
            static SizeType GetUsagePeak(const AllocationInfoList* list);
            static SizeType GetMaxItemUsage(const AllocationInfoList* list);

            FEATSTD_MAKE_CLASS_STATIC(SystemMemoryStatistic);
    };

    template<typename T>
    void SystemMemoryStatistic::HintNextAllocation(const Char* fileName, Int32 line, void* userData /* = 0 */)
    {
#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
        if (IgnoreAllocations()) {
            return;
        }
        AllocationInfoInternal& info = NextAllocationInfo();
        info.m_list = GetAllocationInfoList<T>();
#if defined(FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED)
        info.m_info.m_fileName = fileName;
        info.m_info.m_line = line;
#else
        FEATSTD_UNUSED(fileName);
        FEATSTD_UNUSED(line);
#endif
        info.m_info.m_userData = userData;
#else
        FEATSTD_UNUSED(fileName);
        FEATSTD_UNUSED(line);
        FEATSTD_UNUSED(userData);
#endif
    }

    template <typename T>
    SystemMemoryStatistic::AllocationInfoList* SystemMemoryStatistic::GetAllocationInfoList()
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FEATSTD_SYNCED_STATIC_OBJECT(SystemMemoryStatistic::AllocationInfoList*, memoryAllocationInfoList, CreateAllocationInfoList());
#else
        FEATSTD_UNSYNCED_STATIC_OBJECT(SystemMemoryStatistic::AllocationInfoList*, memoryAllocationInfoList, CreateAllocationInfoList());
#endif
        return memoryAllocationInfoList;
    }
/// @}
}}
#endif
