//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SystemMemoryStatistic.h"

#include <FeatStd/Container/Vector.h>
#include <FeatStd/MemoryManagement/Heap.h>

namespace FeatStd { namespace Diagnostics {

#ifdef FEATSTD_THREADSAFETY_ENABLED
        namespace Internal {

            FeatStd::Internal::CriticalSection& GetAllocationInfoCs()
            {
                FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, cs);
                return cs;
            }
            static FeatStd::Internal::CriticalSection& s_allocationInfoCsForceInit = GetAllocationInfoCs();
        }
#endif

    using FeatStd::Internal::Vector;

    struct Math
    {
        template<typename OperandType>
        static const OperandType& Maximum(const OperandType& a, const OperandType& b)
        {
            return (a > b) ? a : b;
        }
    };

    /**
     * Collection of AllocationMetaInfoList objects.
     *
     * One AllocationMetaInfoList object is generated for each data type that is kept for statistics.
     */
    class SystemMemoryStatistic::AllocationInfoListList {
    public:
        AllocationInfoListList() {};
        ~AllocationInfoListList();
        Vector<SystemMemoryStatistic::AllocationInfoList*> m_list;
    };

    /**
     * List of AllocationMetaInfoInternal structure pointers.
     *
     * One AllocationMetaInfoInternal structure exists for each allocated memory. The class provides
     *  statistics for the AllocationMetaInfoInternal list like: number of items, maximum item size
     *  and total size.
     */
    class SystemMemoryStatistic::AllocationInfoList {
    public:
        AllocationInfoList();
        ~AllocationInfoList() {}

        typedef Vector<SystemMemoryStatistic::AllocationInfoInternal*> InternalList;

        void Add(SystemMemoryStatistic::AllocationInfoInternal* item);
        void Remove(const SystemMemoryStatistic::AllocationInfoInternal* item);
        SizeType GetCount() const { return m_internal.Size(); }
        const SystemMemoryStatistic::AllocationInfoInternal* Get(SizeType index) const { return m_internal[index]; }
        SizeType GetTotalUsage() const { return m_totalUsageCache; }
        SizeType GetUsagePeak() const { return m_usagePeak; }
        SizeType GetMaxItemUsage() const { return m_maxItemUsageCache; }
        void Reset();
    private:
        SizeType m_maxItemUsageCache;
        SizeType m_usagePeak;
        SizeType m_totalUsageCache;
        InternalList m_internal;
    };

    SystemMemoryStatistic::AllocationInfoListList::~AllocationInfoListList()
    {
        if (CallbackFunc() != 0) {
            CallbackFunc()(CallbackData());
        }

        //Assure that after the (one and only static) list is destructed, no more allocation statistics are kept.
        SystemMemoryStatistic::IgnoreAllocations() = true;

        //Delete all AllocationMetaInfoList objects before exit.
        for (SizeType index = 0; index < m_list.Size(); ++index) {
            FEATSTD_DELETE(m_list[index]);
        }
        m_list.Clear();
    }

    SystemMemoryStatistic::AllocationInfoList::AllocationInfoList():
        m_maxItemUsageCache(0),
        m_usagePeak(0),
        m_totalUsageCache(0)
    {
        //Add the new list to the global collection of lists.
        (void) GlobalAllocationInfoListList().m_list.Add(this);
    }

    void SystemMemoryStatistic::AllocationInfoList::Reset()
    {
        m_internal.Clear();
        m_maxItemUsageCache = 0;
        m_usagePeak = 0;
        m_totalUsageCache = 0;
    }

void* SystemMemoryStatistic::OnMemoryAllocated(void* location, SizeType size)
{
#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
    FEATSTD_LINT_NEXT_EXPRESSION(925, "pointer to pointer cast necessary to keep memory interface transparent to upper software layers.")
    AllocationInfoInternal* metaInfo = reinterpret_cast<AllocationInfoInternal*>(location);
    if (!IgnoreAllocations()) {
        IgnoreAllocations() = true;
        metaInfo->m_info.m_size = size;
#if defined(FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED)
        metaInfo->m_info.m_fileName = NextAllocationInfo().m_info.m_fileName;
        metaInfo->m_info.m_line = NextAllocationInfo().m_info.m_line;
#endif
        metaInfo->m_info.m_userData = NextAllocationInfo().m_info.m_userData;
        metaInfo->m_list = NextAllocationInfo().m_list;
        metaInfo->m_list->Add(metaInfo);
        IgnoreAllocations() = false;

        //Use AllocationMetaInfoInternal for undefined types.
        HintNextAllocation<AllocationInfoInternal>(0, 0);
    }

    return metaInfo + 1;
#else
    FEATSTD_UNUSED(size);
    return location;
#endif
}

void* SystemMemoryStatistic::OnMemoryFreed(void* location)
{
#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif

    FEATSTD_LINT_NEXT_EXPRESSION(925, "pointer to pointer cast necessary to keep memory interface transparent to upper software layers.")
    AllocationInfoInternal* metaInfo = reinterpret_cast<AllocationInfoInternal*>(location)-1;

    if (!IgnoreAllocations()) {
        IgnoreAllocations() = true;
        metaInfo->m_list->Remove(metaInfo);
        IgnoreAllocations() = false;
    }

    return metaInfo;
#else
    return location;
#endif
}

void* SystemMemoryStatistic::GetAllocationInfoLocation(void* location)
{
#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
    FEATSTD_LINT_NEXT_EXPRESSION(925, "pointer to pointer cast necessary to keep memory interface transparent to upper software layers.")
    return reinterpret_cast<AllocationInfoInternal*>(location) - 1;
#else
    return location;
#endif
}

SizeType SystemMemoryStatistic::GetAllocationInfoSize()
{
#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
    return sizeof(AllocationInfoInternal);
#else
    return 0;
#endif
}

// Helper class used to force thread safe initialization of private static types.
class ForceStaticInitHelper {
public:
    static SystemMemoryStatistic::AllocationInfoInternal& GetNextAllocationInfo() {

        SystemMemoryStatistic::AllocationInfoList* allocationInfoList = SystemMemoryStatistic::GetAllocationInfoList<SystemMemoryStatistic::AllocationInfoInternal>();
#if defined(FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED)
        static SystemMemoryStatistic::AllocationInfoInternal metaInfo = { { 0, 0, 0, 0 }, allocationInfoList };
#else
        static SystemMemoryStatistic::AllocationInfoInternal metaInfo = { { 0, 0 }, allocationInfoList };
#endif
        return metaInfo;
    }

    static SystemMemoryStatistic::AllocationInfoListList& GlobalAllocationInfoListList()
    {
        static SystemMemoryStatistic::AllocationInfoListList listList;
        return listList;
    }

    static SystemMemoryStatistic::AllocationInfoInternal& s_allocationInfoForceInit;
    static SystemMemoryStatistic::AllocationInfoListList& s_globalAllocationInfoListForceInit;
};
SystemMemoryStatistic::AllocationInfoInternal& ForceStaticInitHelper::s_allocationInfoForceInit = ForceStaticInitHelper::GetNextAllocationInfo();
SystemMemoryStatistic::AllocationInfoListList& ForceStaticInitHelper::s_globalAllocationInfoListForceInit = ForceStaticInitHelper::GlobalAllocationInfoListList();


SystemMemoryStatistic::AllocationInfoInternal& SystemMemoryStatistic::NextAllocationInfo()
{
    return ForceStaticInitHelper::GetNextAllocationInfo();
}

// Helper for threadsafe private static init.
static bool& GetIgnoreAllocations()
{
    static bool ignoreAllocs = false;
    return ignoreAllocs;
}
static bool& s_ignoreAllocationsForceInit = GetIgnoreAllocations();

bool& SystemMemoryStatistic::IgnoreAllocations()
{
    return GetIgnoreAllocations();
}

// Helper for threadsafe private static init.
static SystemMemoryStatistic::OnBeforeDestroyStatisticsCallback& GetCallbackFunc()
{
    static SystemMemoryStatistic::OnBeforeDestroyStatisticsCallback callbackFunc = 0;
    return callbackFunc;
}
static SystemMemoryStatistic::OnBeforeDestroyStatisticsCallback& s_callbackFuncForceInit = GetCallbackFunc();

SystemMemoryStatistic::OnBeforeDestroyStatisticsCallback& SystemMemoryStatistic::CallbackFunc()
{
    return GetCallbackFunc();
}

// Helper for threadsafe private static init.
static void*& GetCallbackData() {
    static void* callbackData = 0;
    return callbackData;
}
static void*& s_callbackDataForceInit = GetCallbackData();

void*& SystemMemoryStatistic::CallbackData()
{
    return GetCallbackData();
}

SystemMemoryStatistic::AllocationInfoListList& SystemMemoryStatistic::GlobalAllocationInfoListList()
{
    return ForceStaticInitHelper::GlobalAllocationInfoListList();
}

SizeType SystemMemoryStatistic::GetAllocationCount()
{
    SizeType count = 0;

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
    for (SizeType index = 0; index < GlobalAllocationInfoListList().m_list.Size(); ++index) {
        count += GlobalAllocationInfoListList().m_list[index]->GetCount();
    }

    return count;
}

SizeType SystemMemoryStatistic::GetCurrentMemoryUsage()
{
    SizeType total = 0;

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
    for (SizeType index = 0; index < GlobalAllocationInfoListList().m_list.Size(); ++index) {
        total += GlobalAllocationInfoListList().m_list[index]->GetTotalUsage();
    }

    return total;
}

SizeType SystemMemoryStatistic::GetMaximumItemSize()
{
    SizeType max = 0;

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
    for (SizeType index = 0; index < GlobalAllocationInfoListList().m_list.Size(); ++index) {
        max = Math::Maximum(GlobalAllocationInfoListList().m_list[index]->GetMaxItemUsage(), max);
    }

    return max;
}

SizeType SystemMemoryStatistic::GetPeakMemoryUsage()
{
    SizeType total = 0;

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
    for (SizeType index = 0; index < GlobalAllocationInfoListList().m_list.Size(); ++index) {
        total += GlobalAllocationInfoListList().m_list[index]->GetUsagePeak();
    }

    return total;
}

void SystemMemoryStatistic::Reset()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
#endif
    for (SizeType index = 0; index < GlobalAllocationInfoListList().m_list.Size(); ++index) {
        GlobalAllocationInfoListList().m_list[index]->Reset();
    }
}

const SystemMemoryStatistic::AllocationInfo* SystemMemoryStatistic::GetAllocationInfo(SizeType index)
{
    //Threadsafety: No lock needed as GlobalAllocationInfoList will never decrease in size during execution.
    for (SizeType listIndex = 0; listIndex < GlobalAllocationInfoListList().m_list.Size(); ++listIndex) {
        AllocationInfoList* list = GlobalAllocationInfoListList().m_list[listIndex];
        if (index < GetAllocationInfoCount(list)) {
            return GetAllocationInfo(list, index);
        }
        else {
            index -= GetAllocationInfoCount(list);
        }
    }

    return 0;
}

SystemMemoryStatistic::AllocationInfoList* SystemMemoryStatistic::CreateAllocationInfoList()
{
    bool ignoreAllocs = IgnoreAllocations();
    IgnoreAllocations() = true;
    SystemMemoryStatistic::AllocationInfoList* result = FEATSTD_NEW(AllocationInfoList);
    IgnoreAllocations() = ignoreAllocs;
    return result;
}

SizeType SystemMemoryStatistic::GetAllocationInfoCount(const AllocationInfoList* list)
{
    return IgnoreAllocations() ? 0 : list->GetCount();
}

const SystemMemoryStatistic::AllocationInfo* SystemMemoryStatistic::GetAllocationInfo(const AllocationInfoList* list, SizeType index)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    //Ensure list size does not change while we are accessing by index.
    //For single-threaded applications the user is still responsible to ensure the index is within bounds.
    //For multi-threaded applications the value returned by GetAllocationCount can change however so it needs to be checked within a critical section.
    FeatStd::Internal::CriticalSectionLocker lLock(&Internal::GetAllocationInfoCs());
    if (index >= list->GetCount()) {
        return 0;
    }
#endif
    return IgnoreAllocations() ? 0 : &(list->Get(index)->m_info);
}

SizeType SystemMemoryStatistic::GetUsage(const AllocationInfoList* list)
{
    return IgnoreAllocations() ? 0 : list->GetTotalUsage();
}

SizeType SystemMemoryStatistic::GetMaxItemUsage(const AllocationInfoList* list)
{
    return IgnoreAllocations() ? 0 : list->GetMaxItemUsage();
}

SizeType SystemMemoryStatistic::GetUsagePeak(const AllocationInfoList* list)
{
    return IgnoreAllocations() ? 0 : list->GetUsagePeak();
}

void SystemMemoryStatistic::AllocationInfoList::Add(AllocationInfoInternal* item)
{
    (void) m_internal.Add(item);
    m_maxItemUsageCache = Math::Maximum(m_maxItemUsageCache, item->m_info.m_size);
    m_totalUsageCache += item->m_info.m_size;
    m_usagePeak = Math::Maximum(m_usagePeak, m_totalUsageCache);
}

void SystemMemoryStatistic::AllocationInfoList::Remove(const AllocationInfoInternal* item)
{
    for (SizeType index = 0; index < m_internal.Size(); ++index) {
        if (m_internal[index] == item) {
            (void) m_internal.Remove(index);
            break;
        }
    };

    if (item->m_info.m_size == GetMaxItemUsage()) {
        m_maxItemUsageCache = 0;
        for (SizeType index = 0; index < m_internal.Size(); ++index) {
            m_maxItemUsageCache = Math::Maximum(m_maxItemUsageCache, m_internal[index]->m_info.m_size);
        };
    }

    m_totalUsageCache -= item->m_info.m_size;
}
}
}
