//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MemoryAppender.h"

#include "LogEvent.h"
#include "LocationInfo.h"

#include <FeatStd/Util/CharBuffer.h>
#include <FeatStd/Platform/Diagnostic.h>
#include <FeatStd/Platform/Memory.h>

namespace FeatStd {
    namespace Diagnostics {

        using FeatStd::Internal::CharBuffer;
        using FeatStd::Internal::Diagnostic;
        using FeatStd::Internal::Memory;

        /******************************************************************************
         *  Constructor
         ******************************************************************************/
        MemoryAppender::MemoryAppender(Char* location, SizeType size) :
            Base(),
            m_location(location),
            m_currentLocation(location),
            m_size(size)
        {
        }

        /******************************************************************************
         *  Destructor
         ******************************************************************************/
        MemoryAppender::~MemoryAppender()
        {
            m_location = 0;
            m_currentLocation = 0;
        }

        /******************************************************************************
         *  DoAppend
         *  
         *  @remark Uses inherited char buffer!
         *  @remark In case of DebuggerOut not supported or not IsDebuggerPresent(),
         *         2nd sprintf is wasted time :-((
         *
         *  Following format is used:
         *      "time [threadId] level realm message {file(line): function}"
         ******************************************************************************/
        void MemoryAppender::DoAppend(const LogEvent& logEvent)
        {
            static CharBuffer<FEATSTD_LOGGER_MAX_BUFFER_SIZE> lCharBuffer;
            lCharBuffer.Format("%07d.%03d [0x%p] %-5s %-13s %s {%s(%d): %s}\n",
                               (logEvent.mTimeStamp / 1000),
                               (logEvent.mTimeStamp % 1000),
                               logEvent.mThreadId,
                               AsCStr(logEvent.mLogLevel),
                               logEvent.mLogRealmName,
                               logEvent.mMessage,
                               logEvent.mLocation.mFileName,
                               logEvent.mLocation.mLineNumber,
                               logEvent.mLocation.mMethodName
                              );

            SizeType length = lCharBuffer.Length();
            FEATSTD_DEBUG_ASSERT(length < lCharBuffer.Capacity());
            FEATSTD_DEBUG_ASSERT(m_currentLocation - m_location + length < m_size);

            Memory::Copy(m_currentLocation, lCharBuffer.c_str(), length);
            m_currentLocation += length;
        }

    }
}
