//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Logger.h"

#include "LogEvent.h"

#include "ConsoleAppender.h"

#include <FeatStd/Util/CharBuffer.h>
#include <FeatStd/Platform/Time.h>
#include <FeatStd/Platform/String.h>
#include <FeatStd/Platform/VarArg.h>
#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Util/NumericUtil.h>
#include <FeatStd/Diagnostics/CodeChecks.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
    #include <FeatStd/Platform/Thread.h>
    #include <FeatStd/Platform/CriticalSectionLocker.h>
    using FeatStd::Internal::Thread;
#endif

namespace FeatStd { namespace Diagnostics { namespace Internal {

bool InitLogger()
{
    return true;
}

void ShutdownLogger()
{
}

}}}

namespace FeatStd { namespace Diagnostics {

using FeatStd::Internal::CharBuffer;
using Internal::LoggerList;
using Internal::GetLoggerList;

FeatStd::Internal::Ticks Logger::mLoggingStartTime(FeatStd::Internal::TicksType::Now);

LogLevel::Enum Logger::mDefaultLogLevel(LogLevel::Warning);

LogLevel::Enum Logger::GetDefaultLogLevel()
{
    return mDefaultLogLevel;
}

void Logger::SetDefaultLogLevel(LogLevel::Enum logLevel)
{
    mDefaultLogLevel = logLevel;
}

Appender* Logger::GetDefaultAppender()
{
    return &ConsoleAppender::GetInstance();
}

Logger::Logger(const Char* logRealmName) : mLogRealmName(logRealmName),
                                           mLogLevel(GetDefaultLogLevel())
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    FEATSTD_DEBUG_REENTRANCE_GUARD();

    FEATSTD_DEBUG_ASSERT(mLogRealmName != 0);

    if ((GetLoggerList().GetSize() == 0) && (GetAppenderList().GetSize() == 0)) {
        SetAppender(GetDefaultAppender());
        FEATSTD_DEBUG_ASSERT(GetAppenderList().GetSize() == 1);
    }

    FEATSTD_DEBUG_ASSERT(this != 0);
    FEATSTD_DEBUG_ASSERT(!GetLoggerList().Contains(this));
    FEATSTD_DEBUG_ASSERT(!ContainsLoggerWithRealmName(GetRealmName()));
    const SizeType lSizeOld(GetLoggerList().GetSize());
    FEATSTD_RELEASE_UNUSED(lSizeOld);
    GetLoggerList().Prepend(this);
    FEATSTD_DEBUG_ASSERT(GetLoggerList().GetSize() == (lSizeOld + 1));
    FEATSTD_DEBUG_ASSERT(GetLoggerList().Contains(this));
}

Logger::~Logger()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    FEATSTD_DEBUG_REENTRANCE_GUARD();

    FEATSTD_DEBUG_ASSERT(this != 0);
    FEATSTD_DEBUG_ASSERT(GetLoggerList().Contains(this));
    const SizeType lSizeOld(GetLoggerList().GetSize());
    FEATSTD_RELEASE_UNUSED(lSizeOld);
    const bool lRet = GetLoggerList().Remove(this);
    FEATSTD_RELEASE_UNUSED(lRet);
    FEATSTD_DEBUG_ASSERT(lRet);
    FEATSTD_DEBUG_ASSERT(GetLoggerList().GetSize() == (lSizeOld - 1));
    FEATSTD_DEBUG_ASSERT(!GetLoggerList().Contains(this));
}

LogLevel::Enum Logger::SetLevel(LogLevel::Enum logLevel)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    const LogLevel::Enum lOldLevel(mLogLevel);
    mLogLevel = logLevel;
    return lOldLevel;
}

UInt32 Logger::GetAppenderCount() const
{
    const UInt32 lCount(FeatStd::Internal::NumericConversion<UInt32>(GetAppenderList().GetSize()));
    return lCount;
}

bool Logger::ContainsAppender(const Appender& appender)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    // SingleLinkedList: Interface change Param should be const &!
    Appender*const lpAppender(&const_cast<Appender&>(appender));
    FEATSTD_DEBUG_ASSERT(lpAppender != 0);
    // SingleLinkedList: Interface change Contains should be const!
    AppenderList& lAL(const_cast<AppenderList&>(GetAppenderList()));
    for (AppenderList::Iterator it = lAL.Begin(); it != lAL.End(); ++it) {
        if (&(*it) == lpAppender) {
            return true;
        }
    }
    return false;
}

void Logger::SetAppender(Appender* appender)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    // modifies static AppenderList
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    GetAppenderList().Clear();
    if (appender != 0) {
        AppendAppender(appender);
    }
}

/// Multiple Append of same Appender is not checked!
void Logger::AppendAppender(Appender* appender)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    // modifies static AppenderList
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    FEATSTD_GUARD(appender != 0) {

        FEATSTD_DEBUG_ASSERT(!ContainsAppender(*appender));
    const SizeType lSizeOld(GetAppenderList().GetSize());
        FEATSTD_RELEASE_UNUSED(lSizeOld);
        GetAppenderList().Prepend(appender);
        FEATSTD_DEBUG_ASSERT(GetAppenderList().GetSize() == (lSizeOld + 1));
    }
}

void Logger::RemoveAppender(Appender* appender)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    // modifies static AppenderList
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    FEATSTD_DEBUG_ASSERT(appender != 0);

    (void) GetAppenderList().Remove(appender);
}

Logger::AppenderList& Logger::GetAppenderList()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(Logger::AppenderList, sAppenderList);
    return sAppenderList;
}
Logger::AppenderList& Logger::s_forceInitAppenderList = GetAppenderList();

bool Logger::ContainsLoggerWithRealmName(const Char* realmName)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(&Logger::GetCriticalSection());
#endif
    FEATSTD_DEBUG_ASSERT(realmName != 0);

    const LoggerList& lLL(GetLoggerList());
    for (LoggerList::ConstIterator it = lLL.Begin(); it != lLL.End(); ++it) {
        if (FeatStd::Internal::String::CompareStrings(it->GetRealmName(), realmName) == 0) {
            return true;
        }
    }
    return false;
}

UInt32 GetThreadId() {
#ifdef FEATSTD_THREADSAFETY_ENABLED
    return Thread::GetCurrentId();
#else 
    return 0;
#endif
}

/// mAssembledInfo and Appender.Append protected by critical section!
// Modifies static mAppenderList)
FEATSTD_LINT_NEXT_EXPRESSION(1916, "exceptional allowance for the ellipsis in error handling and logging. Logging is not part of production code.")
void Logger::ForcedLog(LogLevel::Enum logLevel, const LocationInfo& location, const Char* info, ...) const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    // Lock against concurrent calls from various threads
    FeatStd::Internal::CriticalSectionLocker lLock(&Logger::GetCriticalSection());
#endif
    FEATSTD_DEBUG_ASSERT(logLevel >= mLogLevel);
    FEATSTD_DEBUG_ASSERT(IsEnabledFor(logLevel));

    if (GetAppenderList().GetSize() == 0) {
        return; // Increase performance
    }

    FEATSTD_DEBUG_ASSERT(GetAppenderList().GetSize() > 0);
    const UInt32 lTimeStamp(mLoggingStartTime.GetExpiredTime());

    const UInt16 lAssembledInfoLength(FEATSTD_LOGGER_MAX_BUFFER_SIZE); /// Includes \0!
    static CharBuffer<lAssembledInfoLength> lAssembledInfo;
    FEATSTD_VALIST lArgs;
    FEATSTD_VASTART(lArgs, info);
    lAssembledInfo.Format(info, lArgs);

    FEATSTD_LINT_NEXT_EXPRESSION(1924, "5-2-4 this cast warning is caused by system includes")
    FEATSTD_VAEND(lArgs);
    FEATSTD_UNUSED(lArgs);  // MISRA

    FEATSTD_DEBUG_ASSERT(lAssembledInfo.Length() < lAssembledInfo.Capacity());

    // Check maximum used FEATSTD_DEBUG_ASSERT(mAssembledInfo.Length() < 110); // SampleApp
    // Check maximum used FEATSTD_DEBUG_ASSERT(mAssembledInfo.Length() < 85); // UnitTests

    const Char*const lLogRealmName(GetRealmName());
    LogEvent lLogEvent(lLogRealmName, logLevel, lTimeStamp, location, GetThreadId(), lAssembledInfo.c_str());

    AppenderList& lAL(GetAppenderList());
    FEATSTD_DEBUG_ASSERT(lAL.GetSize() > 0);
    for (AppenderList::Iterator it = lAL.Begin(); it != lAL.End(); ++it) {
        it->Append(lLogEvent);
    }
    
}

static FeatStd::Internal::CriticalSection& LogControlCriticalSection()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_logControlCriticalSection);
    return s_logControlCriticalSection;
}
static FeatStd::Internal::CriticalSection& s_forceInitLogControlCriticalSection = LogControlCriticalSection();

#ifdef FEATSTD_THREADSAFETY_ENABLED
FeatStd::Internal::CriticalSection& Logger::GetCriticalSection()
{
    return LogControlCriticalSection();
}
#endif

namespace Internal {
LoggerList& GetLoggerList()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(LoggerList, sLoggerList);
    return sLoggerList;
}

static LoggerList& s_forceInitLoggerList = GetLoggerList();

}
}}
