//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Diagnostics_LogLevel_h)
#define FeatStd_Diagnostics_LogLevel_h

#include <FeatStd/Base.h>
#ifndef FEATSTD_LOG_ENABLED
    #error Feature FEATSTD_LOG_ENABLED is not enabled. Do not include any FeatStd logging releated header files!
#endif
#include <FeatStd/Diagnostics/Debug.h>

#include <FeatStd/Platform/String.h>

namespace FeatStd { namespace Diagnostics {
namespace LogLevel {

FEATSTD_LINT_SYMBOL(849, FeatStd::Diagnostics::LogLevel::Enum::Debug, "it is intended that Debug has the same value as All")

/// @addtogroup FEATSTD_DIAGNOSTICS
/// @{
/**
 *  Level of a log message.
 *  When used in filters (Logger::SetLevel), enabling one level will enable the lower ones as well,
 *  e.g. Enabling Warning for one module will also enable Error and Fatal.
 */
//! [LogLevelEnum]
enum Enum
{
    All = 0,        ///< All
    Debug = All,    ///< Debug
    Info = 1,       ///< Info
    Warning = 2,    ///< Warning = Default
    Error = 3,      ///< Error
    Fatal = 4,      ///< Fatal
    Off = 5         ///< Off
};
//! [LogLevelEnum]

/**
 *  Returns unique string for every LogLevel.
 *  @param level The LogLevel, its unique string is returned.
 *  @return The unique string for every LogLevel.
 */
inline const Char* AsCStr(LogLevel::Enum level)
{
    const Char * retString = "Unknown";
    switch (level) {
        //case All: 
        case Debug:   
            retString = "DEBUG"; 
            break;
        case Info:    
            retString = "INFO";
            break;
        case Warning: 
            retString = "WARN";
            break;
        case Error:   
            retString = "ERROR";
            break;
        case Fatal:   
            retString = "FATAL";
            break;
        case Off: 
            retString = "OFF";
            break;
        default:
            FEATSTD_DEBUG_FAIL();
    }
    return retString;
}

/**
 *  Returns corresponding enum value.
 *  @param name The name, its corresponding enum is returned.
 *  @return The corresponding enum value.
 */
inline LogLevel::Enum AsEnumValue(const Char* name)
{
    if (name == 0) { // Robustness
        FEATSTD_DEBUG_ASSERT(name != 0);
        return LogLevel::All;
    }

    FEATSTD_DEBUG_ASSERT(name != 0);
    LogLevel::Enum lLevel;
    if ((FeatStd::Internal::String::CompareStrings(name, "ALL") == 0) ||
        (FeatStd::Internal::String::CompareStrings(name, "DEBUG") == 0)) {
        lLevel = LogLevel::Debug;
    }
    else if (FeatStd::Internal::String::CompareStrings(name, "INFO") == 0) {
        lLevel = LogLevel::Info;
    }
    else if (FeatStd::Internal::String::CompareStrings(name, "WARN") == 0) {
        lLevel = LogLevel::Warning;
    }
    else if (FeatStd::Internal::String::CompareStrings(name, "ERROR") == 0) {
        lLevel = LogLevel::Error;
    }
    else if (FeatStd::Internal::String::CompareStrings(name, "FATAL") == 0) {
        lLevel = LogLevel::Fatal;
    }
    else if (FeatStd::Internal::String::CompareStrings(name, "OFF") == 0) {
        lLevel = LogLevel::Off;
    }
    else {
        FEATSTD_DEBUG_FAIL();
        lLevel = LogLevel::All;
    }

    FEATSTD_DEBUG_ASSERT(lLevel >= All);
    FEATSTD_DEBUG_ASSERT(lLevel <= Off);
    return lLevel;
}
}
/// @}
}}
#endif
