//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Diagnostics_Log_h)
#define FeatStd_Diagnostics_Log_h

#include <FeatStd/Base.h>

#ifdef FEATSTD_LOG_ENABLED

#define FEATSTD_IN_DIAGNOSTICS_LOG
#include <FeatStd/Diagnostics/LogRealm.h>
#include <FeatStd/Diagnostics/LogControl.h>
#include <FeatStd/Diagnostics/LocationInfo.h>
#undef FEATSTD_IN_DIAGNOSTICS_LOG

/// @addtogroup FEATSTD_DIAGNOSTICS
/// @{
/**
 * Sets LogRealm to be used by FEATSTD_LOG_<LogLevel> output macros.
 * Usage: <br>
 *   Insert >FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::xxx);<
 *   either in class declaration or in implementation file.
 */
#define FEATSTD_LOG_SET_REALM(logRealm) \
    FEATSTD_LINT_SYMBOL(751, Candera::mLogRealm, "Preparation for Log output") \
    FEATSTD_LINT_SYMBOL(751, mLogRealm, "Preparation for Log output") \
    typedef logRealm mLogRealm

/**
* Sets the Log-Level for all realms
* Example: <br>
*   FEATSTD_LOG_SET_LOG_LEVEL(Info);                     //Set the log level of all Realms to Info-level.
*   FEATSTD_LOG_SET_LOG_LEVEL_FOR_REALM(Error,MyRealm);  //Set log level of a specific realm, to Error-level.
* @param LEVEL All / Debug / Info / Warning(default) / Error / Fatal / Off
*/
#define FEATSTD_LOG_SET_LOG_LEVEL(LEVEL)                  FeatStd::Diagnostics::LogControl::SetLogLevel(FeatStd::Diagnostics::LogLevel::LEVEL)

/**
* Sets the Log-Level for a specified realm.
* Example: <br>
*   FEATSTD_LOG_SET_LOG_LEVEL(Info);                     //Set the log level of all Realms to Info-level.
*   FEATSTD_LOG_SET_LOG_LEVEL_FOR_REALM(Error,MyRealm);  //Set log level of a specific realm, to Error-level.
* @param LEVEL All / Debug / Info / Warning(default) / Error / Fatal / Off
* @param REALM CanderaEngine2D / CanderaEngine3D / UserDefinedRealm / ...
*/
#define FEATSTD_LOG_SET_LOG_LEVEL_FOR_REALM(LEVEL,REALM)  FeatStd::Diagnostics::LogControl::SetLogLevel<REALM>(FeatStd::Diagnostics::LogLevel::LEVEL)

/**
 * Prints a log message using LogLevel Debug.
 * Example: <br>
 *   FEATSTD_LOG_DEBUG("Top left edge: %d, %d", rect.GetX(), rect.GetY());
 * @param args The actual message to print.
 */
#define FEATSTD_LOG_DEBUG(args, ...) \
    FEATSTD_LOG_REALM(mLogRealm, FeatStd::Diagnostics::LogLevel::Debug,   args, ## __VA_ARGS__)
#define FEATSTD_LOG_INFO(args, ...) \
    FEATSTD_LOG_REALM(mLogRealm, FeatStd::Diagnostics::LogLevel::Info,    args, ## __VA_ARGS__)
#define FEATSTD_LOG_WARN(args, ...) \
    FEATSTD_LOG_REALM(mLogRealm, FeatStd::Diagnostics::LogLevel::Warning, args, ## __VA_ARGS__)
#define FEATSTD_LOG_ERROR(args, ...) \
    FEATSTD_LOG_REALM(mLogRealm, FeatStd::Diagnostics::LogLevel::Error,   args, ## __VA_ARGS__)
#define FEATSTD_LOG_FATAL(args, ...) \
    FEATSTD_LOG_REALM(mLogRealm, FeatStd::Diagnostics::LogLevel::Fatal,   args, ## __VA_ARGS__)

/**
 * Prints a log message.
 * Example: <br>
 *   FEATSTD_LOG_REALM(FeatStd::Diagnostics::LogLevel::LogRealm::Application,
 *                     FeatStd::Diagnostics::LogLevel::Debug,
 *                     "Top left edge: %d, %d\n", rect.GetX(), rect.GetY());
 * @param logRealm Realm to be used
 * @param logLevel The level of the message
 * @param args     The actual message to print.
 */
FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_LOG_REALM)
#define FEATSTD_LOG_REALM(logRealm, logLevel, args, ...) \
    do { \
        FeatStd::Diagnostics::Logger& lLogger(FeatStd::Diagnostics::LogControl::GetLogger<logRealm>()); \
        FEATSTD_LOG(lLogger, logLevel, args, ## __VA_ARGS__); \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)

/**
 * Prints a log message.
 * Example: <br>
 *   FEATSTD_LOG(myLogger,
 *               FeatStd::Diagnostics::LogLevel::LogRealm::Application,
 *               FeatStd::Diagnostics::LogLevel::Debug,
 *               "Top left edge: %d, %d\n", rect.GetX(), rect.GetY());
 * @param logger   Logger instance to be used
 * @param logLevel The level of the message
 * @param args     The actual message to print.
 */
FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_LOG_REALM)
#define FEATSTD_LOG(logger, logLevel, args, ...) \
    do { \
        if (logger.IsEnabledFor(logLevel)) { \
            FeatStd::Diagnostics::LocationInfo  lInfo(FEATSTD__FILE__, __LINE__, FEATSTD_LOG_FUNC); \
            logger.ForcedLog(logLevel, lInfo, \
                             args, ## __VA_ARGS__); \
        } \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)
/// @}
#else
#if defined(__ghs__) || defined(__IAR_SYSTEMS_ICC__)
#define FEATSTD_LOG_SET_REALM(logRealm) \
    FEATSTD_LINT_SYMBOL(751, supressWarning, "Only dummy code to suppress compiler warning for extra semicolon when building without log") \
    typedef bool supressWarning
#else
#define FEATSTD_LOG_SET_REALM(logRealm) 
#endif

// --Map Log-related macros to empty when Logging is disabled--
#define FEATSTD_LOG_DEFINE_REALM(REALM)
#define FEATSTD_LOG_DECLARE_REALM(REALM)
#define FEATSTD_LOG_SET_LOG_LEVEL(LEVEL)
#define FEATSTD_LOG_SET_LOG_LEVEL_FOR_REALM(LEVEL,REALM)

#define FEATSTD_LOG_DEBUG(args, ...) _FEATSTD_LOG_DUMMY
#define FEATSTD_LOG_INFO(args, ...)  _FEATSTD_LOG_DUMMY
#define FEATSTD_LOG_WARN(args, ...)  _FEATSTD_LOG_DUMMY
#define FEATSTD_LOG_ERROR(args, ...) _FEATSTD_LOG_DUMMY
#define FEATSTD_LOG_FATAL(args, ...) _FEATSTD_LOG_DUMMY

#define FEATSTD_LOG_REALM(logRealm, logLevel, args, ...) _FEATSTD_LOG_DUMMY

FEATSTD_LINT_MACRO_WHILEFALSE(_FEATSTD_LOG_DUMMY)
#define _FEATSTD_LOG_DUMMY \
    do { \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)
#endif
#endif

