//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "FileAppender.h"

#include "LogEvent.h"

#include <FeatStd/Util/CharBuffer.h>
#include <FeatStd/Platform/String.h>

namespace FeatStd { namespace Diagnostics {

using FeatStd::Internal::CharBuffer;

FileAppender::FileAppender() : Base(), mFilePointer(0)
{
    FEATSTD_DEBUG_ASSERT(!IsFileOpened());
}

FileAppender::~FileAppender()
{
    if (IsFileOpened()) {
        bool lRet = Close();
        if (!lRet) {
            FEATSTD_LOG_ERROR("Close failed!");
        }
    }
    FEATSTD_DEBUG_ASSERT(!IsFileOpened());
}

bool FileAppender::IsFileOpened() const
{
    return File::IsFileOpened(mFilePointer);
}

bool FileAppender::Open(const Char* filename, bool append)
{
    FEATSTD_DEBUG_ASSERT(filename != 0);

    FEATSTD_DEBUG_ASSERT(!IsFileOpened()); //??? error

    FEATSTD_DEBUG_ASSERT(mFilePointer == 0);
    const File::Mode lMode(append ? File::AppendBinary :
                                    File::WriteBinary);
    mFilePointer = File::Open(filename, lMode);
    if (!IsFileOpened()) {
        //Signal error Log
        return false;
    }

    // Log message
    return true;
}

bool FileAppender::Close()
{
    FEATSTD_DEBUG_ASSERT(IsFileOpened()); //??? error

    FEATSTD_DEBUG_ASSERT(mFilePointer != 0);
    const bool lRet = File::Close(mFilePointer);
    FEATSTD_DEBUG_ASSERT(lRet);
    FEATSTD_DEBUG_ASSERT(mFilePointer == 0);

    FEATSTD_DEBUG_ASSERT(!IsFileOpened());
    return lRet;
}

void FileAppender::Flush()
{
    FEATSTD_DEBUG_ASSERT(IsFileOpened());
#if (!defined(FEATSTD_OS_INTEGRITY)) && (!defined(FEATSTD_OS_NOOS)) && (!defined(FEATSTD_OS_MICROSAR)) && (!defined(FEATSTD_OS_OSEK))
    Int lRet = fflush(mFilePointer);
    FEATSTD_DEBUG_ASSERT(lRet == 0);
    FEATSTD_RELEASE_UNUSED(lRet);
#else
    FEATSTD_LINT_NEXT_EXPRESSION(1762, "can only be made const in certain configuration")
#endif
}

/// Following format is used:
///      "time [threadId] level realm message {file(line): function}"
/// @remark Expects open file!
void FileAppender::DoAppend(const LogEvent& logEvent)
{
    FEATSTD_DEBUG_ASSERT(IsFileOpened());
    // max size of "%07d.%03d [0x%p] %-5s %-13s 
    // This line would only be valid when a log realm name is not longer than 32 character.
    // static const Int maxBufferSize = sizeof(Int)*2 /*%07d.%03d*/ + 4 /* [0x*/+ sizeof(void*) /*%p*/ + 2 /*] */ + 32 * 2 + 1 /*%-5s %-13s*/;
    // "time [threadId] level realm "
    static CharBuffer<FEATSTD_LOGGER_MAX_BUFFER_SIZE> lCharBuffer;
    FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(AsCStr(logEvent.mLogLevel)) <= 5);
    lCharBuffer.Format("%07d.%03d [0x%p] %-5s %-13s ",
                       (logEvent.mTimeStamp / 1000), (logEvent.mTimeStamp % 1000),
                       logEvent.mThreadId,
                       AsCStr(logEvent.mLogLevel),
                       logEvent.mLogRealmName);
    FEATSTD_DEBUG_ASSERT(lCharBuffer.Length() < lCharBuffer.Capacity());

    CheckedFileWrite(lCharBuffer.c_str());

    // "message"
    FEATSTD_DEBUG_ASSERT(logEvent.mMessage != 0);
    if (FeatStd::Internal::String::Length(logEvent.mMessage) > 0) {
        CheckedFileWrite(logEvent.mMessage);
    }

    // " {"
    CheckedFileWrite(" {");

    // "file"
    CheckedFileWrite(logEvent.mLocation.mFileName);

    // "(line): "
    FEATSTD_DEBUG_ASSERT(logEvent.mLocation.mLineNumber <= 999999);
    FEATSTD_DEBUG_ASSERT(lCharBuffer.Capacity() >= (4 + 6));
    lCharBuffer.Format("(%d): ", logEvent.mLocation.mLineNumber);
    FEATSTD_DEBUG_ASSERT(lCharBuffer.Length() < lCharBuffer.Capacity());

    CheckedFileWrite(lCharBuffer.c_str());

    // "function"
    CheckedFileWrite(logEvent.mLocation.mMethodName);

    // "}\n"
    CheckedFileWrite("}\n");

    // TODO Flush(); needed? Configurable?
}

void FileAppender::CheckedFileWrite(const Char* buffer)
{
    FEATSTD_DEBUG_ASSERT(buffer != 0);

    const UInt32 lBlockSize(1);
    const SizeType lLength(FeatStd::Internal::String::Length(buffer));
    FEATSTD_DEBUG_ASSERT(lLength > 0);

    SizeType lWrittenLength = File::Write(mFilePointer, buffer,
                                        lBlockSize, lLength);
    FEATSTD_DEBUG_ASSERT(lWrittenLength == lLength);
    FEATSTD_RELEASE_UNUSED(lWrittenLength);
}
}}
