//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Diagnostics_Debug_h)
#define FeatStd_Diagnostics_Debug_h

#include <FeatStd/Base.h>

#include <FeatStd/Platform/Diagnostic.h>
#include <FeatStd/Diagnostics/DebugControl.h>

/// @addtogroup FEATSTD_DIAGNOSTICS
/// @{
#if !defined(FEATSTD_OS_BREAK)
    #error Must be defined in platform
#endif

#if !defined(FEATSTD_DEBUG_BREAK)
    #if defined(FEATSTD_DEBUG)
        /**
         * Halts the program execution or triggers a breakpoint if supported for this platform.
         */
        #define FEATSTD_DEBUG_BREAK FEATSTD_OS_BREAK
    #else
        #define FEATSTD_DEBUG_BREAK()
    #endif
#endif

#if !defined(FEATSTD_DEBUG_ASSERT)
    #if defined(FEATSTD_DEBUG)
        /**
        * Evaluates the given condition and signals an assertion if the result is false.
        * Never put an condition with side effects inside FEATSTD_DEBUG_ASSERT as
        * without DEBUG defined the expression won't be executed.
        * @param condition The condition to evaluate.
        */
        FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_DEBUG_ASSERT)
        #define FEATSTD_DEBUG_ASSERT(condition) \
                do { \
                    if (!(condition)) { \
                        if (FeatStd::Diagnostics::DebugControl::Assert((#condition), \
                                                                       FEATSTD__FILE__, __LINE__)) { \
                            FEATSTD_OS_BREAK(); \
                                       } \
                                } \
                            FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
                        } while (false)
    #else
        #define FEATSTD_DEBUG_ASSERT(condition)
    #endif
#endif

#if !defined(FEATSTD_DEBUG_FAIL)
    #if defined(FEATSTD_DEBUG)
        /**
         * Signals an assertion.
         */
        FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_DEBUG_ASSERT)
        #define FEATSTD_DEBUG_FAIL() \
            do { \
                if (FeatStd::Diagnostics::DebugControl::Assert("DEBUG_FAIL", \
                                                                FEATSTD__FILE__, __LINE__)) { \
                    FEATSTD_OS_BREAK(); \
                } \
            FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
            } while (false)
    #else
        #define FEATSTD_DEBUG_FAIL()
    #endif
#endif

#if !defined(FEATSTD_DEBUG_REENTRANCE_GUARD)
    #if defined(FEATSTD_DEBUG)
        /**
         * Signals an assertion if the block containing the macro is entered a second time.
         * @remark sReentranceGuardCount handling isn't protected for perfomance reasons.
         */
        #define FEATSTD_DEBUG_REENTRANCE_GUARD() \
            static UInt32 sReentranceGuardCount(0); \
            FeatStd::Diagnostics::ReentranceGuard lReentranceGuard(sReentranceGuardCount)

    #else
        #define FEATSTD_DEBUG_REENTRANCE_GUARD()
    #endif
#endif

/** Compile Time Assertion Macro.
    If the compile time constant condition cond evaluates to false, a compilation error
    will be issued. The error message is compiler dependent, but will refer to an undefined
    struct "CtaOnlyTrue".

    Examples:
    @code
        FEATSTD_COMPILETIME_ASSERT(sizeof(Int) == 4);
        FEATSTD_COMPILETIME_ASSERT(FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType);
    @endcode */
#define FEATSTD_COMPILETIME_ASSERT(cond) \
    FEATSTD_LINT_NEXT_EXPRESSION(866, "Usage of sizeof is not unusual in terms how its used") \
    FEATSTD_LINT_NEXT_EXPRESSION(778, "most CTA evaluate to constant expressions - this is their main intent.") \
    ((void)sizeof(FeatStd::Diagnostics::Internal::CtaOnlyTrue<(cond)>)) 

/** Compile Time Assertion Macro.
    If the compile time constant condition cond evaluates to true, a compilation error
    will be issued. The error message is compiler dependent, but will refer to an undefined
    struct "CtaOnlyFalse".

    Examples:
    @code
        FEATSTD_COMPILETIME_ASSERT_NOT(sizeof(Int) == 4);
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType);
    @endcode */
#define FEATSTD_COMPILETIME_ASSERT_NOT(cond) \
    FEATSTD_LINT_NEXT_EXPRESSION(866, "Usage of sizeof is not unusual in terms how its used") \
    ((void)sizeof(FeatStd::Diagnostics::Internal::CtaOnlyFalse<(cond)>)) 

namespace FeatStd { namespace Diagnostics { namespace Internal {
    template <Int v> struct CtaOnlyTrue { };
    template <> struct CtaOnlyTrue<0>;

    template <Int v> struct CtaOnlyFalse;
    template <> struct CtaOnlyFalse<0> { };
}}}
/// @}
#endif
