//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Container_Vector_h)
#define FeatStd_Container_Vector_h

#include <FeatStd/Base.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/MemoryManagement/Heap.h> // FILE_LINE
#include <FeatStd/MemoryManagement/Allocator.h>
#include <FeatStd/Util/NumericUtil.h>

FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(VectorTest, Increase)

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
#define VECTOR_ALLOC(nBytes)    Allocator::Alloc(FEATSTD_MEMORYPOOL_CALL_CONTEXT(nBytes))
#else
#define VECTOR_ALLOC(nBytes)    Allocator::Alloc(nBytes, FILE_LINE)
#endif

namespace FeatStd {
    namespace Internal {
        /// @addtogroup FEATSTD_CONTAINER
        /// @{

        namespace VectorPrivate {

            template<typename T, typename RandomAccessIterator, typename Comparator>
            class IntroSort {
                static const Int SIZE_THRESHOLD = 16;

                public:
                    IntroSort(Comparator& comparator) : m_comparator(comparator)
                    {
                        // Not instructions.
                    }

                    void Sort(RandomAccessIterator begin, RandomAccessIterator end)
                    {
                        Introsort(begin, end);
                    }

                private:
                    Comparator& m_comparator;

                    void Introsort(RandomAccessIterator lo, RandomAccessIterator hi);

                    void IntrosortLoop(RandomAccessIterator lo, RandomAccessIterator hi, Int depthLimit);

                    RandomAccessIterator Partition(RandomAccessIterator lo, RandomAccessIterator hi);

                    // Heap sort algorithm
                    void Heapsort(RandomAccessIterator lo, RandomAccessIterator hi);

                    void DownHeap(OffsetType i, OffsetType n, RandomAccessIterator lo);

                    //Insertion sort algorithm
                    void Insertionsort(RandomAccessIterator lo, RandomAccessIterator hi);

                    //Common method for all algorithms exchanges the content of two iterators.
                    void Exchange(RandomAccessIterator i, RandomAccessIterator j)
                    {
                        T t(*i);
                        *i = *j;
                        *j = t;
                    }

                    //forbid assignment, make it private
                    IntroSort& operator=(const IntroSort&);
            };

            /**
            * @brief Iterator and ConstIterator are derived from this base
            */
            template<typename T>
            class IteratorBase {
                public:
                    typedef T ValueType;
                    typedef T& ReferenceType;
                    typedef T* PointerType;

                    IteratorBase() : m_pointer(0)
                    {
                        // No instructions.
                    }

                    IteratorBase(T* pointer) : m_pointer(pointer)
                    {
                        // No instructions.
                    }

                    // Pre-increment
                    IteratorBase& operator++()
                    {
                        ++m_pointer;
                        return *this;
                    }

                    // Post-increment
                    IteratorBase operator++(int)
                    {
                        IteratorBase tmp = *this;
                        ++m_pointer;
                        return tmp;
                    }

                    // Pre-decrement
                    IteratorBase& operator--()
                    {
                        --m_pointer;
                        return *this;
                    }

                    // Post-decrement
                    IteratorBase operator--(int)
                    {
                        IteratorBase tmp = *this;
                        --m_pointer;
                        return tmp;
                    }

                    bool operator!=(const IteratorBase& other) const
                    {
                        return this->m_pointer != other.m_pointer;
                    }

                    bool operator==(const IteratorBase& other) const
                    {
                        return this->m_pointer == other.m_pointer;
                    }

                    bool operator<(const IteratorBase& other) const
                    {
                        return this->m_pointer < other.m_pointer;
                    }

                    bool operator<=(const IteratorBase& other) const
                    {
                        return this->m_pointer <= other.m_pointer;
                    }

                    bool operator>(const IteratorBase& other) const
                    {
                        return this->m_pointer > other.m_pointer;
                    }

                    bool operator>=(const IteratorBase& other) const
                    {
                        return this->m_pointer >= other.m_pointer;
                    }

                    OffsetType operator-(const IteratorBase& other) const
                    {
                        return this->m_pointer - other.m_pointer;
                    }

                    IteratorBase operator-(const OffsetType value) const
                    {
                        return this->m_pointer - value;
                    }

                    IteratorBase operator+(const OffsetType value) const
                    {
                        return this->m_pointer + value;
                    }

                    T& operator*()
                    {
                        return *m_pointer;
                    }

                    T& operator[](OffsetType idx)
                    {
                        return m_pointer[idx];
                    }

                private:
                    T* m_pointer;
            };

        }   // namespace VectorPrivate


        /**
         *  The default increase policy for Vector.
         *  new_capacity = current_capacity * 2
         */
        class DefaultIncreasePolicy {
            public:
                static SizeType CalculateNewSize(SizeType currentSize)
                {
                    return currentSize * 2;
                }
        };

        /**
         *  Increase policy for Vector which increases the capacity linear.
         *  new_capacity = current_capacity + increaseValue
         *  @param  increaseValue   The amount of element to be reserved.
         */
        template<SizeType increaseValue>
        class LinearIncreasePolicy {
            public:
                static SizeType CalculateNewSize(SizeType currentSize)
                {
                    return currentSize + increaseValue;
                }
        };

        /**
         *  @brief A vector is a container that supports random access to elements, constant time insertion and removal of elements at the end.
         *  It supports linear time insertion and removal of elements at the beginning or in the middle.
         *  The number of elements in a vector may vary dynamically; memory management is automatic.
         *
         *  @tparam T               Defines the type of the contained elements. The type needs to have a copy-constructor and has to be
         *                          assignable. A default constructor is not necessarily needed.
         *  @tparam IncreasePolicy  The increase policy defines the behavior for increasing the vector's memory. It defines how much space for new
         *                          elements will be allocated if the vector's size has reached the current capacity and a new element is added.
         *                          The given class has to implement method 'static Int CalculateNewSize(Int currentSize)'.
         *  @tparam Allocator       The allocator to be used for memory allocation.
         **/
        template<typename T, typename IncreasePolicy = DefaultIncreasePolicy, typename Allocator = FeatStd::MemoryManagement::DefaultAllocator>
        class Vector {
            public:
                typedef VectorPrivate::IteratorBase<T> Iterator;
                typedef VectorPrivate::IteratorBase<const T> ConstIterator;

                /**
                 * Constructs a Vector with no memory allocated.
                 * Use Reserve() to allocate a proper amount of memory.
                 * Initialization of the memory takes place when elements are added to the vector.
                 */
                Vector() :
                    m_size(0),
                    m_capacity(0),
                    m_memory(0)
                {
                }

                /**
                 * Destructor calls the destructor for all Size() elements in the Vector
                 * and deallocates the memory.
                 * Note: Non-Virtual, not intended as base class.
                 */
                ~Vector();

                // Copy Constructor is not supported due to undefined behavior when memory allocation fails.
                /**
                 * Copy constructor.
                 * Create a new Vector with the same properties and elements as the copied object.
                 * @param other Vector object that is copied.
                 */
                Vector(const Vector& other);

                /**
                 * Assignment operator for assignments of other Vector objects.
                 * Important note: Object remains unchanged if memory allocation fails.
                 * @param other Vector that is assigned.
                 * @return This object with assigned properties.
                 */
                Vector& operator=(const Vector& other);

                /**
                 * Adds a new elements to the end of the Vector and reallocates memory if necessary.
                 * @param element Element that should be attached.
                 * @return False if Vector is full and reallocation failed.
                 */
                bool Add(const T& element);

                /**
                 * Remove an element at a certain position.
                 * @param index Index of the elements to be removed.
                 * @return False if the index is invalid.
                 */
                bool Remove(SizeType index);

                /**
                 * Clear all elements from the Vector and call their destructor.
                 * Clear doesn't deallocate any memory. After this operation
                 * the new size of the vector is 0.
                 */
                void Clear();

                /**
                 * Clear all elements from the Vector and call their destructor, and deallocate the memory.
                 * After this operation the size and capacity of the vector is 0.
                 */
                void Free();

                /**
                 * Returns the number of elements stored in the Vector.
                 * @return Number of elements in the Vector.
                 */
                SizeType Size() const
                {
                    return m_size;
                }

                /**
                 * Insert a new element into the vector at a certain position.
                 * All elements after the insert position are moved back in the vector.
                 * @param index Position of the new element.
                 * @param element Element that should be inserted.
                 * @return False if index is invalid or memory reallocation failed.
                 */
                bool Insert(SizeType index, const T& element);

                /**
                 * Returns the capacity of the vector.
                 * The capacity defines the number of element for which memory is already allocated.
                 * Note: Use Reserve to change the capacity of the vector.
                 * @return Capacity of the vector.
                 */
                SizeType GetCapacity() const
                {
                    return m_capacity;
                }

                /**
                 * Return true if the vector is empty.
                 * @return True, if the vector doesn't contain any elements.
                 */
                bool Empty() const
                {
                    return (m_size == 0);
                }

                /**
                 * Resizes the vector through memory reallocation.
                 * Important note: Reserve also allows to decrease the capacity of the vector. Affected elements are destructed.
                 * @param capacity New capacity of the Vector.
                 * @return False, if memory allocation failed or capacity parameter is invalid (negative).
                 */
                bool Reserve(SizeType capacity);

                /**
                 * Uses IntroSort to sort the Vector. IntroSort is a modified quick sort that switches to
                 * HeapSort or IntroSort under certain circumstances to reduce performance hit in worst case scenario.
                 * @param comparator Must have the signature bool f(T LHS, T RHS) and returns true if LHS should be before
                 * RHS.
                 */
                template<typename Comparator>
                void Sort(Comparator& comparator)
                {
                    VectorPrivate::IntroSort<T, Iterator, Comparator> sort(comparator);
                    sort.Sort(Begin(), End());
                }

                /**
                 * Uses IntroSort to sort a subsection of the Vector. IntroSort is a modified quick sort that switches to
                 * HeapSort or IntroSort under certain circumstances to reduce performance hit in worst case scenario.
                 * @param comparator Must have the signature bool f(T LHS, T RHS) and returns true if LHS should be before RHS.
                 * @param startIndex The index of the first element of the subsection to sort.
                 * @param endIndex   The index of the last element of the subsection to sort.
                 */
                template<typename Comparator>
                void Sort(Comparator& comparator, SizeType startIndex, SizeType endIndex)
                {
                    if (m_size == 0) {
                        return;
                    }

                    if ((startIndex > endIndex) ||
                        (startIndex >= m_size) ||
                        (endIndex >= m_size)) {
                        FEATSTD_DEBUG_FAIL(); // This assert indicates invalid startIndex and endIndex values.
                        return;
                    }

                    VectorPrivate::IntroSort<T, Iterator, Comparator> sort(comparator);
                    sort.Sort(Iterator(&(m_memory[startIndex])), Iterator(&(m_memory[endIndex+1]))); // end is last element + 1
                }

                /**
                 *  Checks if an object is in the Vector
                 *  @param object Object to look for.
                 *  @return       True if linked list contains object. False otherwise.
                 */
                bool Contains(const T& object) const;

                // ################# Begin: Iterator #################

                Iterator Begin()
                {
                    return Iterator(&m_memory[0]);
                }

                Iterator End()
                {
                    return Iterator(&(m_memory[m_size])); // end is last element + 1
                }

                ConstIterator ConstBegin() const
                {
                    return ConstIterator(&m_memory[0]);
                }

                ConstIterator ConstEnd() const
                {
                    return ConstIterator(&(m_memory[m_size])); // end is last element + 1
                }

                T& operator[](SizeType idx)
                {
                    return m_memory[idx];
                }

                const T& operator[](SizeType idx) const
                {
                    return m_memory[idx];
                }


                //#############################################################################

            protected:
                void GetPreviousPosition(SizeType& current) const
                {
                    if (0 == current) {
                        current = Size();
                    } else {
                        current--;
                    }
                }

                void GetNextPosition(SizeType& current) const
                {
                    current = current + 1;
                }

                SizeType GetHeadPosition() const
                {
                    return 0;
                }

                SizeType GetTailPosition() const
                {
                    return Size();
                }

            private:
                FEATSTD_UNIT_TEST_TESTCASE_FRIEND(VectorTest, Increase);

                static const SizeType c_MemoryInitialCapacity = 8;

                SizeType m_size; ///< Number of elements in the container.
                SizeType m_capacity; ///< Number of elements for which memory has been allocated.
                T* m_memory; ///< Pointer to the allocated memory.
        };

        namespace VectorPrivate {

            template<typename T, typename RandomAccessIterator, typename Comparator>
            RandomAccessIterator IntroSort<T, RandomAccessIterator, Comparator>::Partition(RandomAccessIterator lo, RandomAccessIterator hi)
            {
                RandomAccessIterator mid = lo + (hi - lo) / 2;
                if (m_comparator(*hi, *lo)) {
                    Exchange(hi, lo);
                }
                if (m_comparator(*mid, *lo)) {
                    Exchange(mid, lo);
                }
                if (m_comparator(*hi, *mid)) {
                    Exchange(hi, mid);
                }

                T pivotValue = *mid;
                RandomAccessIterator i, j;
                for (i = lo+1, j = hi-1; ;) {
                    while (m_comparator(*i, pivotValue)) {
                        ++i;
                    }
                    while (m_comparator(pivotValue, *j)) {
                        --j;
                    }
                    if (i >= j) {
                        return i;
                    }
                    Exchange(i, j);
                    ++i;
                    --j;
                }
            }

            template<typename T, typename RandomAccessIterator, typename Comparator>
            void IntroSort<T, RandomAccessIterator, Comparator>::Introsort(RandomAccessIterator lo, RandomAccessIterator hi)
            {
                OffsetType size = hi - lo;
                if (size > SIZE_THRESHOLD) {
                    Int depthLimit = 0;
                    for (; size > 0; ++depthLimit) {
                        size >>= 1;
                    }
                    //depthLimit *= 2; //This would give a higher depthLimit than the standard 2*log(size), hurting performance. Additionally, this could be fine tuned to give optimal performance in the average case. 
                    IntrosortLoop(lo, hi - 1, depthLimit);
                }
                Insertionsort(lo, hi);
            }

            template<typename T, typename RandomAccessIterator, typename Comparator>
            void IntroSort<T, RandomAccessIterator, Comparator>::IntrosortLoop(RandomAccessIterator lo, RandomAccessIterator hi, Int depthLimit)
            {
                if (hi - lo > SIZE_THRESHOLD) {
                    if (depthLimit == 0) {
                        Heapsort(lo, hi);
                        return;
                    }

                    RandomAccessIterator pivot = Partition(lo, hi);

                    IntrosortLoop(lo, pivot, depthLimit - 1);
                    IntrosortLoop(pivot, hi, depthLimit - 1);
                }
            }

            template<typename T, typename RandomAccessIterator, typename Comparator>
            void IntroSort<T, RandomAccessIterator, Comparator>::Heapsort(RandomAccessIterator lo, RandomAccessIterator hi)
            {
                OffsetType n = hi - lo;
                for (OffsetType i = n / 2; i >= 1; --i) {
                    DownHeap(i, n, lo);
                }
                for (OffsetType i = n; i > 1; --i) {
                    Exchange(lo, lo + i - 1);
                    DownHeap(1, i - 1, lo);
                }
            }

            template<typename T, typename RandomAccessIterator, typename Comparator>
            void IntroSort<T, RandomAccessIterator, Comparator>::DownHeap(OffsetType i, OffsetType n, RandomAccessIterator lo)
            {
                T d = lo[i - 1];
                OffsetType child;
                while (i <= n / 2) {
                    child = 2 * i;
                    if ((child < n) && (m_comparator(lo[child - 1], lo[child]))) {
                        ++child;
                    }
                    if (!m_comparator(d, lo[child - 1])) {
                        break;
                    }
                    lo[i - 1] = lo[child - 1];
                    i = child;
                }
                lo[i - 1] = d;
            }

            template<typename T, typename RandomAccessIterator, typename Comparator>
            void IntroSort<T, RandomAccessIterator, Comparator>::Insertionsort(RandomAccessIterator lo, RandomAccessIterator hi)
            {
                RandomAccessIterator i, j;
                for (i = lo; i < hi; ++i) {
                    j = i;
                    T t(*i);
                    while ((j != lo) && (m_comparator(t, j[-1]))) {
                        *j = j[-1];
                        --j;
                    }
                    *j = t;
                }
            }

        }   // namespace VectorPrivate

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        Vector<T, IncreasePolicy, Allocator>::~Vector()
        {
            Free();
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        Vector<T, IncreasePolicy, Allocator>::Vector(const Vector& other) :
            m_size(0),
            m_capacity(0),
            m_memory(0)
        {
            if (Reserve(other.m_capacity)) { // Adapt capacity to the capacity of the copied Vector.       
                for (SizeType i = 0; i < other.m_size; ++i) {
                    // Create new element in uninitialized memory.
                    MemoryManagement::Construct(&m_memory[i], other.m_memory[i]);
                }
            m_size = other.m_size;
            }
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        Vector<T, IncreasePolicy, Allocator>& Vector<T, IncreasePolicy, Allocator>::operator=(const Vector& other)
        {
            if ((this != &other) && (Reserve(other.m_capacity))) { // Adapt capacity to the capacity of the assigned Vector
                for (SizeType i = 0; i < other.m_size; ++i) {
                    if (i < m_size) { // Assign elements in already initialized memory.
                        m_memory[i] = other.m_memory[i];
                    }
                    else { // Create new element in uninitialized memory.
                        MemoryManagement::Construct(&m_memory[i], other.m_memory[i]);
                    }
                }
                // Call the destructor for remaining old elements between new end and old end or new capacity.
                for (SizeType i = other.m_size; (i < m_size) && (i < m_capacity); ++i) {
                    MemoryManagement::Destruct(&m_memory[i]);
                }
                m_size = other.m_size;
                // Allocator is not copied.
            }
            return *this;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        bool Vector<T, IncreasePolicy, Allocator>::Add(const T& element)
        {
            bool success = true;
            if (m_size == m_capacity) {
                SizeType newCapacity = (m_capacity > 0) ? IncreasePolicy::CalculateNewSize(m_capacity) : c_MemoryInitialCapacity;
                success = Reserve(newCapacity);
            }
            if (success) {
                // Use placement new for adding new elements which doesn't allocate new memory.
                MemoryManagement::Construct(&m_memory[m_size++], element);
            }
            return success;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        bool Vector<T, IncreasePolicy, Allocator>::Remove(SizeType index)
        {
            // Function also returns false if m_size is 0.
            if (index >= m_size) {
                return false;
            }
            for (SizeType i = index; (m_size > 0 ) && (i < m_size - 1); ++i) {
                m_memory[i] = m_memory[i + 1];
            }
            // Destroy element at the last position.
            MemoryManagement::Destruct(&m_memory[--m_size]);
            return true;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        void Vector<T, IncreasePolicy, Allocator>::Clear()
        {
            // Call destructors in reverse order.
            for (OffsetType i = m_size - 1; i >= 0; --i) {
                MemoryManagement::Destruct(&m_memory[i]);
            }
            m_size = 0;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        void Vector<T, IncreasePolicy, Allocator>::Free()
        {
            Clear();
            Allocator::Free(m_memory);
            m_memory = 0;
            m_capacity = 0;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        bool Vector<T, IncreasePolicy, Allocator>::Insert(SizeType index, const T& element)
        {
            // It's allowed to insert an element after the last element.
            if (index > m_size) {
                return false;
            }

            bool success = true;
            if (m_size == m_capacity) {
                FeatStd::SizeType newCapacity = (m_capacity > 0) ? IncreasePolicy::CalculateNewSize(m_capacity) : c_MemoryInitialCapacity;
                success = Reserve(newCapacity);
            }
            if (success) {
                // Special case, insert an element at the end or in an empty Vector.
                if (index == m_size) {
                    MemoryManagement::Construct(&m_memory[m_size], element);
                }
                else { // Insert element at the beginning or in the middle.
                    MemoryManagement::Construct(&m_memory[m_size], m_memory[m_size - 1]);
                    for (SizeType i = m_size - 1; (i < m_size) && (i > index); --i) { // m_size - 1 can be the unsigned int max value, as m_size SizeType, that is why (i < m_size) check is required
                        m_memory[i] = m_memory[i - 1];
                    }
                    m_memory[index] = element;
                }
                ++m_size;
            }
            return success;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        bool Vector<T, IncreasePolicy, Allocator>::Reserve(SizeType capacity)
        {
            T* newMemory = 0;
            newMemory = static_cast<T*>(VECTOR_ALLOC((FeatStd::Internal::NumericConversion<UInt32, SizeType>(MemoryManagement::ArrayByteSize<T>(SizeType(capacity))))));
            if ((newMemory == 0) && (capacity != 0)) { // Double check if memory allocation was really successful.
                FEATSTD_DEBUG_FAIL(); //Report that memory allocation failed
                return false;
            }

            if (m_size > 0) {
                SizeType newSize = (capacity < m_size) ? capacity : m_size;
                for (SizeType i = 0; i < newSize; ++i) {
                    MemoryManagement::Construct(&newMemory[i], m_memory[i]);
                    MemoryManagement::Destruct(&m_memory[i]);
                }
                for (SizeType i = newSize; i < m_size; i++) {
                    MemoryManagement::Destruct(&m_memory[i]);
                }
                m_size = newSize;
            }
            if (m_capacity > 0) {
                // Deallocate the old buffer.
                Allocator::Free(m_memory);
            }
            m_capacity = capacity;
            m_memory = newMemory;
            return true;
        }

        template<typename T, typename IncreasePolicy /*= DefaultIncreasePolicy*/, typename Allocator /*= FeatStd::MemoryManagement::DefaultAllocator*/>
        bool Vector<T, IncreasePolicy, Allocator>::Contains(const T& object) const
        {
            bool found = false;
            for (ConstIterator it = ConstBegin(); it != ConstEnd(); ++it) {
                if (object == *it) {
                    found = true;
                    break;
                }
            }
            return found;
        }



        /// @}
    }
}
#endif
