//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Container_SingleLinkedList_h)
#define FeatStd_Container_SingleLinkedList_h

#include <FeatStd/Base.h>
#include <FeatStd/MemoryManagement/Heap.h>
#include <FeatStd/MemoryManagement/Allocator.h>

FEATSTD_UNIT_TEST_TESTCASE_DECLARATION(SingleLinkedListTest, DeleteLastNode)

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
    #define SINGLELINKEDLIST_ALLOC(nBytes)  Allocator::Alloc(FEATSTD_MEMORYPOOL_CALL_CONTEXT(nBytes))
#else
    #define SINGLELINKEDLIST_ALLOC(nBytes)  Allocator::Alloc(nBytes, FILE_LINE)
#endif

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_CONTAINER
/// @{
/** @brief  Implements an non intrusive single linked list.
            The list grows and shrinks dynamically. T must be copy-able.
            @tparam T           item type that should be maintained in the list
            @tparam Allocator   The allocator to be used for memory allocation.
            @code
                SingleLinkedList<SomeObject*> list;
                list.Append(objPtr1);
                list.Prepend(objPtr2);
                list.Append(objPtr3);
                list.Append(objPtr4);

                for (SingleLinkedList<SomeObject*>::Iterator it = list.Begin(); it != list.End(); ++it)
                    SomeObject* objPtr = *it;
                    objPtr->DoSomethingUseful();
                }

                list.Dispose();
            @endcode
 */
template<typename T, typename Allocator = FeatStd::MemoryManagement::DefaultAllocator>
class SingleLinkedList
{
    private:
        class SingleLinkedListNode {
            public:
                SingleLinkedListNode(const T& object, SingleLinkedListNode* next);

                // Appending a node to the current node.
                void InsertNext(SingleLinkedListNode* next);

                // Get the object
                T& operator*();

            private:
                FEATSTD_UNIT_TEST_TESTCASE_FRIEND(SingleLinkedListTest, DeleteLastNode);
                T m_object;
                SingleLinkedListNode* m_next;
                friend class SingleLinkedList<T, Allocator>;
                FEATSTD_MAKE_CLASS_UNCOPYABLE(SingleLinkedListNode);
        };

    public:
        template<typename T2>
        class IteratorBase {
            public:
                typedef T2 ValueType;
                typedef T2& ReferenceType;
                typedef T2* PointerType;

                IteratorBase() : m_currentNode(0) {}
                IteratorBase(SingleLinkedListNode* node) : m_currentNode(node) { }

                T2& operator * () const { return *(*m_currentNode); }

                IteratorBase& operator ++ () {
                    if (m_currentNode != 0) {
                        m_currentNode = m_currentNode->m_next;
                    }
                    return *this;
                }

                IteratorBase operator ++(int) {
                    IteratorBase ret = *this;
                    if (m_currentNode != 0) {
                        m_currentNode = m_currentNode->m_next;
                    }
                    return ret;
                }

                bool operator == (const IteratorBase& it) const { return m_currentNode == it.m_currentNode; }
                bool operator != (const IteratorBase& it) const { return m_currentNode != it.m_currentNode; }

            private:
                SingleLinkedListNode* m_currentNode;

                SingleLinkedListNode* GetCurrentNode() { return m_currentNode; }

                friend class SingleLinkedList<T, Allocator>;
        };
        typedef IteratorBase<T> Iterator;
        typedef IteratorBase<const T> ConstIterator;

        /**
         *  Default constructor
         */
        SingleLinkedList() : m_head(0), m_tail(0), m_size(0) { }

        /**
         *  Destructor
         */
        ~SingleLinkedList() { Clear(); }

        /**
         *  Returns an IteratorBase object referencing the list head
         *  @return IteratorBase object referencing the list head
         */
        Iterator Begin() { return IteratorBase<T>(m_head); }

        /**
         *  Returns an IteratorBase object referencing the list head
         *  @return IteratorBase object referencing the list head
         */
        ConstIterator Begin() const { return IteratorBase<const T>(m_head); }

        /**
         *  Returns the end IteratorBase for list iteration
         *  @return IteratorBase object referencing just past the end of the list
         */
        Iterator End() { return IteratorBase<T>(0); }

        /**
         *  Returns the end IteratorBase for list iteration
         *  @return IteratorBase object referencing just past the end of the list
         */
        ConstIterator End() const { return IteratorBase<const T>(0); }

        /**
         *  To prepend an element to a linked list is to insert that element in front of the first element of the list.
         *  @param  object Object to prepend to the linked list.
         *  @return        True if object was prepended to existing list or otherwise head was created. False if creation of node
         *                 failed.
         */
        bool Prepend(const T& object);

        /**
         *  To append an element to a linked list is to insert that element after the last element of the list.
         *  @param  object Object to append to the linked list.
         *  @return        True if object was appended to existing list or otherwise head was created. False if creation of node
         *                 failed.
         */
        bool Append(const T& object);

        /**
         *  Inserts an object before a given reference object. If refObject is null, the given object is appended to the end of the list.
         *  @param  object      Defines the object to insert into the list.
         *  @param  refObject   References an object at which the new given object shall be inserted before.
         */
        bool Insert(const T& object, const T& refObject);

        /**
         *  Removes an object from the list and deletes the corresponding Node
         *  @param object Object to remove from the linked list.
         *  @return True if object was found and deleted from the linked list. False otherwise.
         */
        bool Remove(const T& object, bool removeAll = false);

        /**
         *  Checks if an object is in the list
         *  @param object Object to look for.
         *  @return       True if linked list contains object. False otherwise.
         */
        bool Contains(const T& object) const;

        /**
         *  Checks if the list is empty or not
         *  @return True if list is empty, indicated by head == 0. False otherwise.
         */
        bool IsEmpty() const { return m_head == 0; }

        /**
         *  Returns the size of the list
         *  @return The size of the list.
         */
        SizeType GetSize() const { return m_size; }

        /**
         *  Clears all nodes from the List
         */
        void Clear();

    private:
        FEATSTD_UNIT_TEST_TESTCASE_FRIEND(SingleLinkedListTest, DeleteLastNode);

        SingleLinkedListNode* m_head;
        SingleLinkedListNode* m_tail;
        SizeType m_size;

        bool CreateHead(const T& object);

        template<typename Arg>
        SingleLinkedListNode* NewNode(const Arg& arg, SingleLinkedListNode* next = 0);

        void DeleteNode(SingleLinkedListNode* node);

        FEATSTD_MAKE_CLASS_UNCOPYABLE(SingleLinkedList);
};

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
SingleLinkedList<T, Allocator>::SingleLinkedListNode::SingleLinkedListNode(const T& object, SingleLinkedListNode* next) : 
    m_object(object), 
    m_next(next) 
{ 
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
void SingleLinkedList<T, Allocator>::SingleLinkedListNode::InsertNext(SingleLinkedListNode* next) 
{
    if (next != 0) {
        SingleLinkedListNode* temp = m_next;
        m_next = next;
        m_next->m_next = temp;
    }
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
T& SingleLinkedList<T, Allocator>::SingleLinkedListNode::operator*() 
{ 
    return m_object; 
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
bool SingleLinkedList<T, Allocator>::Prepend(const T& object) 
{
    bool result = false;
    if (m_head == 0) {
        result = CreateHead(object);
    }
    else {
        SingleLinkedListNode* temp = m_head;
        m_head = NewNode(object, temp);
        if (m_head != 0) {
            result = true;
            ++m_size;
        }
    }
    return result;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
bool SingleLinkedList<T, Allocator>::Append(const T& object) {
    bool result = false;
    if (m_head == 0) {
        result = CreateHead(object);
    }
    else {
        SingleLinkedListNode* ptr = NewNode(object);
        if (ptr != 0) {
            m_tail->InsertNext(ptr);
            m_tail = m_tail->m_next;
            result = true;
            ++m_size;
        }
    }
    return result;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
bool SingleLinkedList<T, Allocator>::Insert(const T& object, const T& refObject) {
    if (m_head == 0) {
        return CreateHead(object);
    }

    Iterator it = Begin();

    if (refObject == 0 || refObject == *it){
        return Prepend(object);
    }
    else {
        Iterator prev = End();

        while (it != End()) {
            if (refObject == *it) {
                SingleLinkedListNode* newNode = NewNode(object);
                if (newNode != 0) {
                    newNode->InsertNext(it.GetCurrentNode());
                    m_size++;

                    if (prev != End()) {
                        prev.GetCurrentNode()->InsertNext(newNode);
                    }

                    return true;
                }
            }

            prev = it;
            ++it;
        }
    }

    return false;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
bool SingleLinkedList<T, Allocator>::Remove(const T& object, bool removeAll) {
    bool result = false;
    SingleLinkedListNode* previousNode = 0;
    Iterator it = Begin();
    while ( it != End() ) {
        SingleLinkedListNode* currentNode = it.GetCurrentNode();
        ++it;
        if (object == **currentNode) {
            if (currentNode == m_head) {
                m_head = m_head->m_next;
            } else {
                previousNode->m_next = currentNode->m_next;
            }
            if (currentNode == m_tail) {
                m_tail = previousNode;
            }
            DeleteNode(currentNode);
            result = true;
            --m_size;
            if (!removeAll) {
                break;
            }
        } else {
            previousNode = currentNode;
        }
    }
    return result;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
bool SingleLinkedList<T, Allocator>::Contains(const T& object) const {
    bool found = false;
    for (ConstIterator it = Begin(); it != End(); ++it) {
        if (object == *it) {
            found = true;
            break;
        }
    }
    return found;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
void SingleLinkedList<T, Allocator>::Clear()
{
    SingleLinkedListNode* ptr = m_head;
    while (ptr) {
        m_head = ptr->m_next;
        DeleteNode(ptr);
        ptr = m_head;
    }
    m_head = 0;
    m_tail = 0;
    m_size = 0;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
bool SingleLinkedList<T, Allocator>::CreateHead(const T& object)
{
    bool result = false;
    m_head = NewNode(object);
    m_tail = m_head;
    if (m_head != 0) {
        result = true;
        ++m_size;
    }
    return result;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/> 
template<typename Arg>
typename SingleLinkedList<T, Allocator>::SingleLinkedListNode* SingleLinkedList<T, Allocator>::NewNode(const Arg& arg, SingleLinkedListNode* next /*= 0*/)
{
    SingleLinkedListNode* ptr = static_cast<SingleLinkedListNode*>(SINGLELINKEDLIST_ALLOC(sizeof(SingleLinkedListNode)));
    if (ptr != 0) {
        new(ptr)SingleLinkedListNode(arg, next);
    }
    return ptr;
}

template<typename T, typename Allocator /* = FeatStd::MemoryManagement::DefaultAllocator*/>
void SingleLinkedList<T, Allocator>::DeleteNode(SingleLinkedListNode* node)
{
    if (node != 0) {
        node->~SingleLinkedListNode();
        Allocator::Free(node);
    }
}

/// @}
}}
#endif
