//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Container_LinkedList_h)
#define FeatStd_Container_LinkedList_h

#include <FeatStd/Base.h>
#include <FeatStd/MemoryManagement/TypeTraits.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_CONTAINER
/// @{
        /**
         * @brief LinkedListNode defines the data required by LinkedList
         */
        template<typename T> class LinkedListNode {
        public:
            /**
             *  Constructs a linked list node.
             */
            LinkedListNode() { m_next = 0; }

            /**
             *  Returns the next item in the linked list.
             *  @return next item in linked list or null
             */
            T* GetNext() const { return m_next; }

            /**
             *  Set the next item in linked list.
            *  This method is only intended to be used by class LinkedList
            *  @param next pointer to the next item in the list or null
            */
            void SetNext(T* next) { m_next = next; }

        private:
            T* m_next;
        };

        /**
         * @brief Implements an intrusive single linked list
         * @param T Item type that should be maintained in the list.
         * @param Node LinkedListNode member that should be used for this list.
         * @code
            class ListItem {
            public:
            ListItem(int x = 0) { _x = x; }

            int GetX() const { return _x; }

            LinkedListNode<ListItem> PrimaryListNode;
            LinkedListNode<ListItem> SecondaryListNode;

            private:
            int _x;
            };

            ...
            typedef LinkedList<ListItem, &ListItem::PrimaryListNode> PrimaryList;
            typedef LinkedList<ListItem, &ListItem::SecondaryListNode> SecondaryList;

            PrimaryList primaryList;
            SecondaryList secondaryList;

            ListItem item;

            primaryList.Add(&item);
            secondaryList.Add(&item);
        * @endcode
        */
        template<typename T, LinkedListNode<T> T::*ListNode> class LinkedList {
        private:
            /**
             * @brief LinkedList iterator base class
             * Iterator and ConstIterator are derived from this base
             */
            template<typename X> class IteratorBase {
            public:
                typedef X ValueType;
                typedef X* PointerType;
                typedef X& ReferenceType;

                /**
                 * Empty iterator construction.
                 */
                IteratorBase() : m_item(0) {}

                /**
                 *  Returns the list item referenced by the Iterator.
                 *  @return Current list item.
                 */
                ReferenceType operator*() const { return *m_item; }
                PointerType operator->() const { return m_item; }

                /**
                 *  Moves the iterator to the next item in the sequence.
                 */
                IteratorBase& operator++() {
                    m_item = (m_item->*ListNode).GetNext();
                    return *this;
                }

                /**
                 *  @copydoc IteratorBase::operator++()
                 */
                IteratorBase operator++(int) {
                    IteratorBase it(m_item);
                    m_item = (m_item->*ListNode).GetNext();
                    return it;
                }

                bool operator==(const IteratorBase &it) const { return m_item == it.m_item; }
                bool operator!=(const IteratorBase &it) const { return m_item != it.m_item; }

                /**
                 *  cast to allow conversion from const iterator to regular iterator.
                 */
                template <typename Y>
                operator IteratorBase<Y>() { return IteratorBase<Y>(m_item); }

                operator PointerType() { return m_item; }

            private:
                X* m_item;

                /**
                 *  Constructs the iterator.
                 *  @param item head of the list.
                 */
                IteratorBase(X* item) : m_item(item) {}
                friend class LinkedList<T, ListNode>;
            };

        public:
            typedef T ElementType;
            typedef typename TypeTraits::Internal::ConstType<ElementType>::Type ConstElementType;
            typedef IteratorBase<ElementType> Iterator;
            typedef IteratorBase<ConstElementType> ConstIterator;

            /**
             *  Constructs an empty list.
             */
            LinkedList() :
                m_count(0),
                m_head(0)
            {
            }

            /**
             *  Constructs a list from a list already known to exist, starting at head, with count elements.
             *  @param count Count of elements in list to create this from.
             *  @param head  Head of list already known, used as this head.
             */
            LinkedList(SizeType count, T* head) :
                m_count(count),
                m_head(head)
            {}

            /**
             *  returns The number of items in the list.
             *  @return Number of items in the list.
             */
            SizeType GetSize() const { return m_count; }
            
            /**
             *  Return an Iterator item referencing the list head.
             *  @return Iterator item referencing the list head.
             */
            Iterator Begin() { return Iterator(m_head); }

            /**
             *  @copydoc Begin()
             */
            ConstIterator Begin() const { return ConstIterator(m_head); }

            /**
             *  Returns the end Iterator for list iteration.
             *  @return Iterator item referencing just past the end of the list.
             */
            Iterator End() { return Iterator(0); }

            /**
             *  @copydoc End()
             */
            ConstIterator End() const { return ConstIterator(0); }

            /**
             *  Empties the list.
             */
            void Clear() {
                m_head = 0;
                m_count = 0;
            }

            /**
             *  Removes the given item from the list.
             *  Removing an item may invalidate existing iterator objects.
             *  @param item defines the item to be removed.
             */
            bool Remove(T* item) {
                T* prev = 0;
                T* p = m_head;
                while (p != 0 && p != item) {
                    prev = p;
                    p = Next(p);
                }
                if (p != 0) {
                    m_count--;
                    if (p == m_head) {
                        m_head = Next(p);
                    }
                    else {
                        SetNext(prev, Next(p));
                    }
                    SetNext(item, 0);
                }
                return p != 0;
            }

            /**
             *  To prepend an element to a linked list is to insert that element in front of the first element of the list.
             *  @param  item    The item to prepend to the list.
             */
            void Prepend(T* item) {
                if (item == 0) {
                    return;
                }
                SetNext(item, m_head);
                m_head = item;
                m_count++;
            }

            /**
             *  To append an element to a linked list is to insert that element after the last element of the list.
             *  @param  item    The item to append to the list.
             */
            void Append(T* item) {
                Insert(item, 0);
            }

            /**
             *  Inserts an item before a reference item given. If refItem is null, the given item is appended to the end of the list.
             *  @param newItem defines the item to insert into the list.
             *  @param refItem references an item at which the new given item shall be inserted before.
             */
            void Insert(T* newItem, T* refItem) {
                if (newItem == 0) {
                    return;
                }
                if (refItem == m_head){
                    Prepend(newItem);
                }
                else{
                    T* prevItem = m_head;
                    while(prevItem != 0 && Next(prevItem) != refItem) {
                        prevItem = Next(prevItem);
                    }
                    if(prevItem == 0){
                        return;
                    }
                    SetNext(newItem, refItem);
                    SetNext(prevItem, newItem);
                    m_count++;
                }
            }

            /**
             *  Checks if an item is in the list
             *  @param item Item to look for.
             *  @return       True if linked list contains item. False otherwise.
             */
            bool Contains(T* item) const {
                bool found = false;
                for (ConstIterator it = Begin(); it != End(); ++it) {
                    if (item == &*it) {
                        found = true;
                        break;
                    }
                }
                return found;
            }

        private:
            SizeType m_count;
            T* m_head;

            /**
             *  return the next item in the sequence
             *  @param item the current item
             *  @return the successor item of parameter item or null in case of end of list
             */
            static T* Next(T* item) { return (item->*ListNode).GetNext(); }

            /**
             *  set the successor item of the given item
             *  @param item the item which successor should be set
             *  @param next the successor item of parameter item
             */
            static void SetNext(T* item, T* next) { (item->*ListNode).SetNext(next); }

            FEATSTD_MAKE_CLASS_UNCOPYABLE(LinkedList);
        };
/// @}
}}
#endif// _LinkedList_h
