//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_ASYNCREQEUSTDISPATCHER_H)
#define FEATSTD_ASYNCREQEUSTDISPATCHER_H

#include <FeatStd/Async/AsyncRequestBase.h>

#include <FeatStd/Diagnostics/Log.h>
#include <FeatStd/Platform/Semaphore.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <FeatStd/Container/Vector.h>
#include <FeatStd/Platform/Thread.h>

namespace FeatStd {

/// @addtogroup FEATSTD_ASYNC
/// @{

/**
* @brief The AsyncRequestDispatcher is used to schedule asynchronous requests at the requester side and
*        to dispatch the requests at the worker side.
*        NOTE: it is not mandatory that the worker is a separate thread. The dispatching can be also performed
*        as part of the main loop.
*/
class AsyncRequestDispatcher
{

public:

    /**
    * The multi threading setup has to be specified for the dispatcher. By default only 1 request queue is used by the dispatcher.
    * To add more request queues for priority handling simply provide the number of additional queues as additionalPriorities.
    * NOTE: if an asynchronous request is scheduled in a single threaded dispatcher then the WaitForFinished will simply
    * execute the request until it is completed or aborted.
    */
    AsyncRequestDispatcher(bool isMultithreaded, FeatStd::UInt32 additionalPriorities = 0);

    /**
    *The destructor.
    */
    ~AsyncRequestDispatcher();

    /**
    * Use this method to schedule an asynchronous request into a dispatcher. Use the priority parameter to specify
    * a higher priority. By default the lowest priority is used (0). The priority parameter will be limited to the highest
    * configured priority (additionalPriorities).
    */
    void Schedule(const AsyncRequestBase::SharedPointer& request, FeatStd::UInt32 priority = 0);

    /**
    * This method will abort all queued requests.
    */
    void Stop();

    /**
    * Waits until all requests are completed or aborted.
    */
    bool Flush();

    /**
    * Returns true if the dispatcher has been started.
    */
    bool IsRunning() const;

    /**
    * Blocks and waits for the next request execution.
    */
    bool WaitForNextExecuteFinished();

    /**
    * Dispatches the next request. If will block until a request is scheduled if waitForNextRequest
    * is set to true. Otherwise it will immediatly return if no request is scheduled.
    */
    void DispatchNext(bool waitForNextRequest);

    /**
    * Starts the dispatcher. If the dispatcher is configured as multithreaded then it will also dispatch requests until the Stop method is called.
    */
    void Start();

    /**
    * @return true when no requests are queued otherwise false.
    */
    bool IsDispatcherEmpty() const;

protected:
    friend class AsyncRequestDispatcherWorkerThread;



private:
    friend class AsyncRequestBase;

    bool SetRunning(bool running);
    void PushRequest(const AsyncRequestBase::SharedPointer& request, FeatStd::UInt32 priority);
    bool PopRequest(AsyncRequestBase::SharedPointer& request);
    bool SignalRequest();
    bool SignalIdle();
    bool SignalExecuteFinished();
    bool WaitForNextRequest();
    FeatStd::UInt32 LimitPriority(FeatStd::UInt32 priority) const;

#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSection& GetCriticalSection() const;
#endif
    FeatStd::Internal::Semaphore m_requestSemaphore;
    FeatStd::Internal::Semaphore m_executedFinishedSemaphore;
    FeatStd::Internal::Semaphore m_idleSemaphore;

    struct RequestQueue
    {
        AsyncRequestBase::SharedPointer m_nextRequest;
        AsyncRequestBase::SharedPointer m_lastRequest;
    };

    FeatStd::Internal::Vector<RequestQueue> m_requestQueues;
    bool m_running;
    bool m_isMultithreaded;

    bool m_isRequestSemaphoreCreated;
    bool m_isExecutedFinishedSemaphoreCreated;
    bool m_isIdleSemaphoreCreated;
};

/**
* @brief The AsyncRequestDispatcherWorkerThread contains an AsyncRequestDispatcher and dispatches the request in a separate thread.
*/
class AsyncRequestDispatcherWorkerThread : public FeatStd::Internal::Thread
{
public:
    AsyncRequestDispatcherWorkerThread(FeatStd::UInt32 additionalPriorities = 0);

    ~AsyncRequestDispatcherWorkerThread()
    {
        if (m_isStopSemaphoreCreated) {
            static_cast<void>(m_stopSemaphore.Destroy());
        }
    }

    AsyncRequestDispatcher& GetDispatcher()
    {
        return m_dispatcher;
    }

    bool Start(Internal::ThreadPriority::Enum priority = Internal::ThreadPriority::_PLATFORM_DEFAULT,
                SetPriorityCallback Callback = static_cast<SetPriorityCallback>(0))
    {
        if (!m_isStopSemaphoreCreated) {
            return false;
        }
        return Run(priority, Callback);
    }

    bool IsRunning() const
    {
        return m_dispatcher.IsRunning();
    }

    bool Flush()
    {
        return m_dispatcher.Flush();
    }

    bool Stop()
    {
        m_dispatcher.Stop();
        if (!m_isStopSemaphoreCreated) {
            return false;
        }
        return m_stopSemaphore.Obtain();
    }

    bool IsDispatcherThread() const
    {
        return m_threadId == GetCurrentId();
    }

    FeatStd::UInt32 GetThreadId() const { return m_threadId; }

protected:
    virtual FeatStd::Int ThreadFn() {
        m_threadId = GetCurrentId();
        m_dispatcher.Start();
        if (m_isStopSemaphoreCreated) {
            static_cast<void>(m_stopSemaphore.Release());
        }
        return 0;
    }

private:
    FeatStd::Internal::Semaphore m_stopSemaphore;

    AsyncRequestDispatcher m_dispatcher;
    FeatStd::UInt32 m_threadId;

    bool m_isStopSemaphoreCreated;
};

/// @}

}

#endif // FEATSTD_ASYNCREQEUSTDISPATCHER_H

