//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_ASYNCREQUESTBASE_H)
#define FEATSTD_ASYNCREQUESTBASE_H

#include <FeatStd/Platform/Semaphore.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Container/Vector.h>
#include <FeatStd/Platform/Thread.h>

namespace FeatStd {
/// @addtogroup FEATSTD_ASYNC
/// @{

class AsyncRequestDispatcher;

/**
* @brief The AsyncRequestBase represents the base interface between kind all asynchronous requests.
*        The interface provides the current state of the asynchronous requests. The request can be
*        aborted as long as it is still in the Waiting state. However if an asynchronous request
*        reschedules itself, an abort will deny the reschedule. An asynchronous request is considered
*        Completed as soon as it has been executed and no more reschedule is performed. During
*        the execution its state will be reflected as Running. Rescheduled asynchronous requests
*        will be put back into Waiting state while they are waiting for their execution.
*/

class AsyncRequestBase
{

public:
    FEATSTD_TYPEDEF_SHARED_POINTER(AsyncRequestBase);

    enum State {
        Waiting = 0,
        Running,
        Completed,
        Aborted,
        Invalid
    };

    virtual ~AsyncRequestBase();

    /**
    * Returns true if the asynchronous request state is set to Completed.
    */
    bool IsCompleted() const
    {
        return Completed == GetState();
    }

    /**
    * Returns the current state of the asynchronous request.
    */
    State GetState() const;

    /**
    * If no other task have to be performed, nor other tasks are available or the result of
    * the asynchronous request is required, then the WaitForFinished method will wait until
    * the asynchronous request is executed or aborted. Keep in mind that the finished Execution
    * does not necessarily mean that it is also completed (if the request is rescheduled then
    * each execute will trigger the signal to leave the wait).
    */
    bool WaitForFinished();

    /**
    * Aborts the asynchronous request if it is still in the Waiting state or prevents a reschedule if already in Running state.
    */
    bool Abort();

protected:

    /**
    * The default constructor has to be used for normal asynchronous requests.
    */
    AsyncRequestBase();

    /**
    * This constructor has to be used if the result of the request is already known (e.g. in case of a cached result).
    * This enables the usage of the asynchronous request as a common interface and prevent the overhead and delay of
    * the request execution.
    * NOTE: The only allowed state is Completed.
    */
    AsyncRequestBase(State state);

    /**
    * The Execute method has to be implemented by the concrete request.
    */
    virtual void Execute() {}

    /**
    * Call the Reschedule method to reschedule the execution of the asynchronous request.
    * It can be used to easily implement the active object pattern.
    */
    void Reschedule();

#ifdef FEATSTD_THREADSAFETY_ENABLED
    /**
    * Use this critical section to synchronize the result set and get functionality.
    */
    FeatStd::Internal::CriticalSection& GetCriticalSection() const { return m_criticalSection; }
#endif

private:
    friend class AsyncRequestDispatcher;

    mutable FeatStd::Internal::CriticalSection m_criticalSection;

    bool Finished();
    bool SignalFinished();

    mutable FeatStd::Internal::Semaphore m_semaphore;

    FeatStd::UInt32 m_originalPriority;
    bool m_reschedule;
    bool m_abort;

    State m_state;
    AsyncRequestDispatcher* m_dispatcher;

    AsyncRequestBase::SharedPointer m_nextRequest;
    FEATSTD_SHARED_POINTER_DECLARATION();
};

/// @}

}

#endif // FEATSTD_ASYNCREQUESTBASE_H

