//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AsyncRequestBase.h"
#include <FeatStd/Async/AsyncRequestDispatcher.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace FeatStd {

FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdAsync);

// ------------------------------------------------------------------------
AsyncRequestBase::AsyncRequestBase() :
    m_originalPriority(0),
    m_reschedule(false),
    m_abort(false),
    m_state(Waiting),
    m_dispatcher(0)
{
    if (!m_semaphore.Create()) {
        m_state = Invalid;
    }
}

// ------------------------------------------------------------------------
AsyncRequestBase::AsyncRequestBase(State state) :
    m_originalPriority(0),
    m_reschedule(false),
    m_abort(false),
    m_state(state),
    m_dispatcher(0)
{
    FEATSTD_DEBUG_ASSERT(m_state == Completed);
    if (Completed != m_state) {
        m_state = Completed;
    }

    if (!m_semaphore.Create()) {
        m_state = Invalid;
        FEATSTD_LOG_ERROR("Failed to create semaphore.\n");
    }
}

// ------------------------------------------------------------------------
AsyncRequestBase::~AsyncRequestBase()
{
    if (m_state != Invalid) {
        static_cast<void>(m_semaphore.Destroy());
    }
    m_dispatcher = 0;
}

// ------------------------------------------------------------------------
AsyncRequestBase::State AsyncRequestBase::GetState() const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    return m_state;
}

// ------------------------------------------------------------------------
bool AsyncRequestBase::WaitForFinished()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    if ((Running == GetState()) || (Waiting == GetState())) {
        if ((0 != m_dispatcher) && (!m_dispatcher->m_isMultithreaded)) {
            do {
                m_dispatcher->DispatchNext(false);
            } while ((Running == GetState()) || (Waiting == GetState()));
        }
        else {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            lock.Release();
#endif
            return m_semaphore.Obtain();
        }
    }
    return true;
}

// ------------------------------------------------------------------------
bool AsyncRequestBase::Abort()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    bool rc = false;
    m_abort = true;
    if (Waiting == m_state) {
        m_state = Aborted;
        AsyncRequestDispatcher* dispatcher = m_dispatcher;
        m_dispatcher = 0;
#ifdef FEATSTD_THREADSAFETY_ENABLED
        lock.Release();
#endif
        rc = SignalFinished();
        if (0 != dispatcher) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lockDispatcher(&dispatcher->GetCriticalSection());
#endif
            for (FeatStd::SizeType i = 0; ((0 != dispatcher) && (i < dispatcher->m_requestQueues.Size())); ++i)
            {
                AsyncRequestBase::SharedPointer previous = dispatcher->m_requestQueues[i].m_nextRequest;
                if (!previous.PointsToNull()) {
                    if (previous.GetPointerToSharedInstance() == this) {
                        dispatcher->m_requestQueues[i].m_nextRequest = previous->m_nextRequest;
                        if (dispatcher->m_requestQueues[i].m_lastRequest.GetPointerToSharedInstance() == this) {
                            dispatcher->m_requestQueues[i].m_lastRequest = AsyncRequestBase::SharedPointer();
                        }
                        dispatcher = 0;
                        break;
                    }
                    do {
                        AsyncRequestBase::SharedPointer current = previous->m_nextRequest;
                        if (current.GetPointerToSharedInstance() == this) {
                            previous->m_nextRequest = current->m_nextRequest;
                            if (dispatcher->m_requestQueues[i].m_lastRequest.GetPointerToSharedInstance() == this) {
                                dispatcher->m_requestQueues[i].m_lastRequest = previous;
                            }
                            dispatcher = 0;
                            break;
                        }
                        previous = current;
                    } while (!previous.PointsToNull());
                }
            }
        }
        FEATSTD_UNUSED(dispatcher);
        return rc;
    }
    return rc;
}

// ------------------------------------------------------------------------
void AsyncRequestBase::Reschedule()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    m_reschedule = true;
}

// ------------------------------------------------------------------------
bool AsyncRequestBase::Finished()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    m_state = Completed;
#ifdef FEATSTD_THREADSAFETY_ENABLED
    lock.Release();
#endif
    return SignalFinished();
}

// ------------------------------------------------------------------------
bool AsyncRequestBase::SignalFinished()
{
    if (m_state != Invalid) {
        return m_semaphore.Release();
    }
    return true;
}

}
