//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewScene.h"

#include "IViewHandler.h"
#include <Courier/Diagnostics/Log.h>
#include "VisualizationMsgs.h"
#include "ViewController.h"
#include "RenderHint.h"
#include "Renderer.h"

#if defined(CANDERA_3D_ENABLED)
// software layers and the composition camera are only available if 3D is enabled
#include "ViewScene3D.h"
#include "ViewVisitor.h"
#endif
#include "ViewHandler.h"
#include "Courier/Transitions/CanderaTransitionHandler.h"

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;

//////////////////////////////////////////////////////////////////////////
/// BEGIN implementation for CameraRefCounter struct
//////////////////////////////////////////////////////////////////////////
// ------------------------------------------------------------------------
void ViewScene::CameraRefCounter::UpdateActivateRefCounter(bool activate)
{
    if (activate) {
        m_activateRefCount++;
    }
    else {
        if (m_activateRefCount > 0) {
            m_activateRefCount--;
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::CameraRefCounter::UpdateRenderingRefCounter(bool enable)
{
    if (enable) {
        m_renderingRefCount++;
    }
    else {
        if (m_renderingRefCount > 0) {
            m_renderingRefCount--;
        }
    }
}
//////////////////////////////////////////////////////////////////////////
/// END implementation for CameraRefCounter struct
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
/// BEGIN implementation for ViewScene class
//////////////////////////////////////////////////////////////////////////

// ------------------------------------------------------------------------

bool ViewScene::mClearOnSceneLoading = true;

bool ViewScene::mInvalidateCamerasOnSceneClearing = false;

ViewScene::LoadError ViewScene::mLastLoadError = ViewScene::LoadSuccess;

ViewScene::WidgetSortingStrategy ViewScene::mWidgetSortingStrategy = ViewScene::NameWidgetSorting;

bool ViewScene::mDetectViewReleationShip = true;

// ------------------------------------------------------------------------
ViewScene::ViewScene(bool managed) :
    Base(managed),
    mIsActive(false),
    mIsUploaded(false),
    mRenderingEnabled(false),
    mConsecutiveRenderingEnabled(false),
    mConsecutiveActivate(false),
    mRefCounterVct()
{
}

// ------------------------------------------------------------------------
ViewScene::~ViewScene()
{
}

// ------------------------------------------------------------------------
bool ViewScene::Init(IViewHandler * viewHandler, const Char * viewName)
{
    return ViewScene::Init(viewHandler,viewName,0);
}

// ------------------------------------------------------------------------
bool ViewScene::Init(IViewHandler * viewHandler, const Char * viewName, ViewController * viewController)
{
    FEATSTD_DEBUG_ASSERT(viewHandler!=0);
    FEATSTD_DEBUG_ASSERT(viewName!=0);
    bool rc = (viewHandler!=0) && (viewName!=0);
    if(rc) {
        mSceneName = ViewId(viewName);
        rc = Base::Init(viewHandler, viewName, viewController);
    }
    return rc;
}

// ------------------------------------------------------------------------
const ViewId & ViewScene::GetId() const
{
    return mSceneName;
}

void ViewScene::OnScopeMaskChanged()
{
    Message::ScopeMask effectiveScopeMask = GetScopeMask();
    for (SizeType i = 0; i < mFrameworkWidgets.Size(); ++i) {
        effectiveScopeMask |= mFrameworkWidgets[i]->GetScopeMask();
    }
    if (0 != mViewController) {
        effectiveScopeMask |= mViewController->GetScopeMask();
    }
    if (effectiveScopeMask != mEffectiveScopeMask) {
        mEffectiveScopeMask = effectiveScopeMask;
        Base::OnScopeMaskChanged();
    }
}

// ------------------------------------------------------------------------
bool ViewScene::IsInitialized()
{
    FEATSTD_DEBUG_ASSERT(GetViewHandler()!=0);
    FEATSTD_DEBUG_ASSERT(GetViewHandler()->GetContentLoader()!=0);
    return (GetViewHandler()!=0) && (GetViewHandler()->GetContentLoader()!=0);
}

// ------------------------------------------------------------------------
void ViewScene::Update(RenderHint * renderHint)
{
    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        if((renderHint->ShallRender2D()==Is2D()) || (renderHint->ShallRender3D()==Is3D())) {
            static_cast<WidgetBase*>(mFrameworkWidgets[i])->Update();
        }
    }
}

// ------------------------------------------------------------------------
bool ViewScene::PartialLoad(bool forceUpload)
{
    bool rc = false;
    COURIER_UNUSED(forceUpload);
    return rc;
}

// ------------------------------------------------------------------------
bool ViewScene::AsyncLoadContent(bool loadContent, bool forceUpload)
{
    return View::LoadContent(loadContent, forceUpload);
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewScene::GetFrameworkWidget(const CompositePath & compositePath, const ItemId & widgetId)
{
    for (SizeType i=0; i < mFrameworkWidgets.Size(); ++i) {
        WidgetBase * baseWidget = static_cast<WidgetBase*>(mFrameworkWidgets[i]);
        if (baseWidget->mItemId.CStr() != baseWidget->GetName()) {
            baseWidget->mItemId = ItemId(baseWidget->GetName());
            if ((0 != baseWidget->GetStringId()) && (0 != baseWidget->GetStringId()->GetOwner())) {
                baseWidget->mCompositePath = CompositePath(baseWidget->GetStringId()->GetOwner());
            }
            else {
                baseWidget->mCompositePath = CompositePath();
            }
        }
        if ((baseWidget->mItemId == widgetId) && (baseWidget->mCompositePath == compositePath)) {
            return baseWidget;
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
bool ViewScene::DistributeMessage(const Message & msg)
{
    bool lMsgConsumed = false;
    for (SizeType i=0; (i<mFrameworkWidgets.Size()) && (! lMsgConsumed); ++i) {
        FrameworkWidget * frameworkWidget = mFrameworkWidgets[i];
        if(msg.InScopeOf(frameworkWidget->GetScopeMask())) {
            lMsgConsumed = frameworkWidget->OnMessage(msg);
        }
    }
    return lMsgConsumed;
}

// ------------------------------------------------------------------------
View * ViewScene::FindView(const ViewId & viewId)
{
    if (viewId==GetId()) {
        return this;
    }
    return 0;
}

// ------------------------------------------------------------------------
void ViewScene::ResetWidgets()
{
    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        mFrameworkWidgets[i]->SetParentView(0);
    }
    mFrameworkWidgets.Clear();
}

// ------------------------------------------------------------------------
void ViewScene::Activate(bool activate)
{
    const ParentViewActivateEvent activateEvent(activate);

    if((mViewController!=0) && activate) {
        mViewController->DispatchEvent(activateEvent);
    }

    bool lostFocus = false;
    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        FrameworkWidget * frameworkWidget = mFrameworkWidgets[i];
        if((! activate) && (! lostFocus)) {
            lostFocus = GetViewHandler()->LoseFocus(frameworkWidget);
        }
        // only if the state has changed notify widgets
        FEATSTD_LINT_NEXT_EXPRESSION(731, "this boolean comparison is correct ...")
        if(mIsActive!=activate) {
            frameworkWidget->DispatchEvent(activateEvent);
        }
    }
    // change the state
    mIsActive = activate;

    if((mViewController!=0) && (! activate)) {
        mViewController->DispatchEvent(activateEvent);
    }
}

// ------------------------------------------------------------------------
void ViewScene::ActivateForCamera(bool activate, const Candera::CanderaObject* camera)
{
    CameraRefCounter* refCounter  = FindInRefCounterVct(mRefCounterVct, camera);
    if (0 != refCounter) {
        refCounter->UpdateActivateRefCounter(activate);
        if ((activate != mIsActive) && // only if the activate state changed
            ((activate && (1 == refCounter->m_activateRefCount)) ||
            ((!activate) && (0 == refCounter->m_activateRefCount) && AllRefCountersZero(mRefCounterVct, ACTIVATE_REF_COUNTER)))) {
                OnActivate(activate);
                mIsActive = activate;
                OnPostActivate(activate);
        }
    }
    COURIER_LOG_DEBUG("activate = %s, camera = %s, activateRefCount = %d",
        (activate ? "TRUE" : "FALSE"),
        ((0 != camera) ? camera->GetName() : "NULL"),
        ((0 != refCounter) ? refCounter->m_activateRefCount : FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max()));
}

// ------------------------------------------------------------------------
bool ViewScene::IsActive() const
{
    return mIsActive;
}

// ------------------------------------------------------------------------
void ViewScene::EnableRenderingForCamera(bool enable, const Candera::CanderaObject* camera)
{
    if (enable) {
        // invalidate the mProcessingInvalidationDependencyId to ensure that the view is rendered
        mProcessingInvalidationDependencyPerCamera.Clear();
    }
    CameraRefCounter* camRefCounter = FindInRefCounterVct(mRefCounterVct, camera);
    if (0 != camRefCounter) {
        camRefCounter->UpdateRenderingRefCounter(enable);
        if ((enable && (1 == camRefCounter->m_renderingRefCount)) ||
            ((!enable) && (0 == camRefCounter->m_renderingRefCount))) {
                UpdateCameraRenderingState(enable, camRefCounter->m_camera);
        }
        if ((enable != mRenderingEnabled) && // only if the rendering state changed
            ((enable && (1 == camRefCounter->m_renderingRefCount))||
            ((!enable) && (0 == camRefCounter->m_renderingRefCount) && AllRefCountersZero(mRefCounterVct, RENDERING_REF_COUNTER)))) {
                UpdateViewSceneRenderingState(enable);
        }
    }
    COURIER_LOG_DEBUG("enable = %s, camera = %s, renderingRefCount = %d",
        (enable ? "TRUE" : "FALSE"),
        ( (0 != camera) ? camera->GetName() : "NULL"),
        ((0 != camRefCounter) ? camRefCounter->m_renderingRefCount : FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max()));
}

// ------------------------------------------------------------------------
bool ViewScene::IsRenderingEnabled() const
{
    return mRenderingEnabled;
}

// ------------------------------------------------------------------------
void ViewScene::OnInitContent(bool init)
{
    const ParentViewInitContentEvent initContentEvent(init);

    if((mViewController!=0) && (! init)) {
        mViewController->DispatchEvent(initContentEvent);
    }
    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        mFrameworkWidgets[i]->DispatchEvent(initContentEvent);

        // Fire ParentViewActivateEvent because Views maybe already activated but not loaded
        // After loading widgets are available and shall be informed that they are now also active.
        // This is only necessary if SetLoadSceneOnViewActivation(false) is set.
        if(mIsActive) {
            const ParentViewActivateEvent activateEvent(mIsActive);
            mFrameworkWidgets[i]->DispatchEvent(activateEvent);
        }
    }
    if((mViewController!=0) && (init)) {
        mViewController->DispatchEvent(initContentEvent);
    }
}

// ------------------------------------------------------------------------
void ViewScene::OnLoad(bool load)
{
    const ParentViewLoadEvent loadEvent(load);

    if((mViewController!=0) && load) {
        mViewController->DispatchEvent(loadEvent);
    }
    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        mFrameworkWidgets[i]->DispatchEvent(loadEvent);

        // Fire ParentViewActivateEvent because Views maybe already activated but not loaded
        // After loading widgets are available and shall be informed that they are now also active.
        // This is only necessary if SetLoadSceneOnViewActivation(false) is set.
        if(mIsActive) {
            const ParentViewActivateEvent activateEvent(mIsActive);
            mFrameworkWidgets[i]->DispatchEvent(activateEvent);
        }
    }
    if((mViewController!=0) && (! load)) {
        mViewController->DispatchEvent(loadEvent);
    }
}

// ------------------------------------------------------------------------
void ViewScene::OnTransitionStarted(const TransitionStartedEvent & msg)
{
    if(mViewController!=0) {
        mViewController->DispatchEvent(msg);
    }

    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        mFrameworkWidgets[i]->DispatchEvent(msg);
    }
}

// ------------------------------------------------------------------------
void ViewScene::OnTransitionFinished(const TransitionFinishedEvent & msg)
{
    if(mViewController!=0) {
        mViewController->DispatchEvent(msg);
    }

    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        mFrameworkWidgets[i]->DispatchEvent(msg);
    }
}


void ViewScene::SetMessagingTransitionDeactivationHandler(MessagingTransitionDeactivationHandler mssagingTransitionDeactivationHandler)
{
    Internal::CanderaTransitionHandler::SetMessagingTransitionDeactivationHandler(mssagingTransitionDeactivationHandler);
}

// ------------------------------------------------------------------------
void ViewScene::OnActivate(bool activate)
{
    COURIER_LOG_DEBUG("ViewScene = %s, activate = %s", GetId().CStr(), (activate ? "TRUE" : "FALSE"));
    const ParentViewActivateEvent activateEvent(activate);
    if ((0 != mViewController) && activate) {
        mViewController->DispatchEvent(activateEvent);
    }
    bool lostFocus = false;
    for (SizeType i = 0; i < mFrameworkWidgets.Size(); ++i) {
        FrameworkWidget * frameworkWidget = mFrameworkWidgets[i];
        if ((!activate) && (!lostFocus)) {
            lostFocus = GetViewHandler()->LoseFocus(frameworkWidget);
        }
        frameworkWidget->DispatchEvent(activateEvent);
    }
    if ((0 != mViewController) && (! activate)) {
        mViewController->DispatchEvent(activateEvent);
    }
}

// ------------------------------------------------------------------------
void ViewScene::OnPostActivate(bool activate) const
{
    ViewActivationEvent event(activate, *this);
    GetEventSource().DispatchEvent(event);
}

// ------------------------------------------------------------------------
void ViewScene::OnRenderingEnabled(bool enable)
{
    COURIER_LOG_DEBUG("ViewScene = %s, enable = %s", GetId().CStr(), (enable ? "TRUE" : "FALSE"));
    const ParentViewRenderingEnabledEvent renderingEvent(enable);

    if((mViewController!=0) && enable) {
        mViewController->DispatchEvent(renderingEvent);
    }

    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        mFrameworkWidgets[i]->DispatchEvent(renderingEvent);
    }

    if((mViewController!=0) && (! enable)) {
        mViewController->DispatchEvent(renderingEvent);
    }
}

// ------------------------------------------------------------------------
void ViewScene::UnfocusWidgets()
{
    for (SizeType i=0; i<mFrameworkWidgets.Size(); ++i) {
        FrameworkWidget * frameworkWidget = mFrameworkWidgets[i];
        (void)GetViewHandler()->LoseFocus(frameworkWidget);
    }
}

// ------------------------------------------------------------------------
void ViewScene::EnableRenderTargets(bool enable, const RTPtrVector & rtPtrVector, bool forceUpload)
{
    // free now the rendertargets
    Renderer * renderer = GetViewHandler()->GetRenderer();
    if(renderer!=0) {
        for(SizeType idx=0; idx<rtPtrVector.Size(); idx++) {
            RenderTarget * rt = rtPtrVector[idx];
            renderer->EnableLayer(enable,rt, forceUpload);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::InitWidgets()
{
    if (IsInitialized()) {
        ResetWidgets();
        FillWidgetList();
        if (mWidgetSortingStrategy == NameWidgetSorting) {
            FEATSTD_LINT_NEXT_EXPRESSION(1502, "no static members, thats okay")
                static WidgetComparerLessByName objSort;
            mFrameworkWidgets.Sort(objSort);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::EnableRenderTarget(bool enable, Candera::RenderTarget* renderTarget, bool forceUpload)
{
    Renderer * renderer = GetViewHandler()->GetRenderer();
    if (0 != renderer) {
        // check if automatic load
        bool callEnableLayer = true;
        if (enable && IsContentLoaded()) {
            Gdu*  gdu = renderer->GetGdu(renderTarget);
            callEnableLayer = (0 != gdu) && (!gdu->IsLoaded());
        }
        if (callEnableLayer) {
            renderer->EnableLayer(enable, renderTarget, forceUpload);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::AddWidgets(const ViewScene::WidgetIterator& widgetIterator)
{
    for (WidgetIterator it = widgetIterator; it.IsValid(); ++it) {
        WidgetBase* widget = *it;
        widget->SetAnimationTimeDispatcher(GetViewHandler()->GetAnimationTimeDispatcherSharedPointer());
        (void)mFrameworkWidgets.Add(widget);
        FrameworkWidget * fw = static_cast<FrameworkWidget*>(widget);
        fw->SetParentView(this);
        OnScopeMaskChanged();
    }
}

// ------------------------------------------------------------------------
void ViewScene::ChangeViewGduRelation(ViewRelationOperation op, Gdu* gdu)
{
    if (0 != gdu) {
        const Renderer::ViewGduRelation relation(this, gdu);
        if (op == AddViewRelation) {
            GetViewHandler()->GetRenderer()->AddViewGduRelation(relation);
        }
        else {
            GetViewHandler()->GetRenderer()->RemoveViewGduRelation(relation);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::ChangeViewSurfaceRelation(ViewRelationOperation op, const Candera::Surface* surface)
{
    if (0 != surface) {
        GraphicDeviceUnit* graphicDeviceUnit = surface->GetGraphicDeviceUnit();
        if (0 != graphicDeviceUnit) {
            ChangeViewRenderTargetRelation(op, graphicDeviceUnit->ToRenderTarget2D());
            ChangeViewRenderTargetRelation(op, graphicDeviceUnit->ToRenderTarget3D());
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::ChangeViewRenderTargetRelation(ViewRelationOperation op, RenderTarget* renderTarget)
{
    if (0 != renderTarget) {
        ChangeViewGduRelation(op, GetViewHandler()->GetRenderer()->GetGdu(renderTarget));
    }
}

// ------------------------------------------------------------------------
ViewScene::CameraRefCounter* ViewScene::FindInRefCounterVct(CameraRefCounterVct& refCounterVct, const CanderaObject* camera)
{
    CameraRefCounter* refCounter = 0;
    if (0 != camera) {
        for (SizeType i = 0;  i < refCounterVct.Size(); ++i) {
            if (camera == refCounterVct[i].m_camera) {
                refCounter = &refCounterVct[i];
            }
        }
    }
    return refCounter;
}

// ------------------------------------------------------------------------
bool ViewScene::AllRefCountersZero(CameraRefCounterVct& refCounterVct, const RefCounterType& type)
{
    bool result =  false;
    bool allZero = true;
    for (SizeType i = 0;  i < refCounterVct.Size(); ++i) {
        SizeType count = FeatStd::Internal::NativeTypeLimit<SizeType>::Max();
        if (ACTIVATE_REF_COUNTER == type) {
            count = refCounterVct[i].m_activateRefCount;
        }
        else {
            if (RENDERING_REF_COUNTER == type) {
                count = refCounterVct[i].m_renderingRefCount;
            }
        }
        allZero = allZero && (0 == count);
    }
    if (allZero) {
        result = true;
    }
    return result;
}

// ------------------------------------------------------------------------
void ViewScene::ResetRefCounterVct(CameraRefCounterVct& refCounterVct, const RefCounterType& type)
{
    for (SizeType i = 0; i < refCounterVct.Size(); ++i) {
        if (ACTIVATE_REF_COUNTER == type) {
            refCounterVct[i].m_activateRefCount = 0;
        } else {
            if (RENDERING_REF_COUNTER == type) {
                refCounterVct[i].m_renderingRefCount = 0;
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene::UpdateViewSceneRenderingState(bool enable)
{
    EnableRenderingScene(enable);
    OnRenderingEnabled(enable);
    mRenderingEnabled = enable;
}

// ------------------------------------------------------------------------
void ViewScene::UpdateCameraRenderingState(bool enable, Candera::CanderaObject* camera)
{
    COURIER_UNUSED(enable);
    COURIER_UNUSED(camera);
    COURIER_LOG_ERROR("Virtual method must be overridden with needed impl. Not applicable for '%s'.",GetId().CStr());
}

// ------------------------------------------------------------------------
void ViewScene::ResetActivateData()
{
    ResetRefCounterVct(mRefCounterVct, ACTIVATE_REF_COUNTER);
}

// ------------------------------------------------------------------------
void ViewScene::ResetRenderingData()
{
    mRenderingEnabled = false;
    mConsecutiveRenderingEnabled = false;
    ResetRefCounterVct(mRefCounterVct, RENDERING_REF_COUNTER);
    for (SizeType i = 0;  i < mRefCounterVct.Size(); ++i) {
        UpdateCameraRenderingState(false, mRefCounterVct[i].m_camera);
    }
    EnableRenderingScene(false);
}

//////////////////////////////////////////////////////////////////////////
/// END implementation for ViewScene class
//////////////////////////////////////////////////////////////////////////

}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {

template<> UInt32 StringBufferAppender< ::Courier::ViewScene::LoadError >::Append(StringBuffer& stringBuffer, ::Courier::ViewScene::LoadError const & object)
{
    UInt32 tcharCount = 0;
    switch (object) {
    case ::Courier::ViewScene::LoadSuccess:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::LoadSuccess");
        break;
    case ::Courier::ViewScene::AssetLoadError:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::AssetLoadError");
        break;
    case ::Courier::ViewScene::VRamUploadError:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::VRamUploadError");
        break;
    case ::Courier::ViewScene::VRamUnloadError:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::VRamUnloadError");
        break;
    case ::Courier::ViewScene::PendingLoadingError:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::PendingLoadingError");
        break;
    case ::Courier::ViewScene::GeneralError:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::GeneralError");
        break;
    default:
        tcharCount += stringBuffer.Append("##unknown##");
        break;
    }
    return tcharCount;
}

template<> UInt32 StringBufferAppender< ::Courier::ViewScene::WidgetSortingStrategy >::Append(StringBuffer& stringBuffer, ::Courier::ViewScene::WidgetSortingStrategy const & object)
{
    UInt32 tcharCount = 0;
    switch (object) {
    case ::Courier::ViewScene::NoneWidgetSorting:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::NoneWidgetSorting");
        break;
    case ::Courier::ViewScene::NameWidgetSorting:
        tcharCount += stringBuffer.Append("::Courier::ViewScene::NameWidgetSorting");
        break;
    default:
        tcharCount += stringBuffer.Append("##unknown##");
        break;
    }
    return tcharCount;
}

}
#endif
