//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewFactory_h)
#define Courier_Visualization_ViewFactory_h

#include <Courier/Base.h>

namespace Courier {
    class View;
    class ViewId;
    class IViewHandler;
    class ViewScene;
    class ViewContainer;
}

namespace Courier {
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A ViewFactory is responsible for creating Views.
    An instance of this implementation might be used when automatic View creation is used.
    A derived class has to be implemented and Create and Destroy methods have to be override
    if the ViewHandler get commands to create specific Views identfied by its name (scene name).
    In this case the derived factory is responsible for allocating/deallocating memory for the view
    instances.
*/

class ViewFactory
{
    friend class ViewHandler;

    public:
        ///
        ViewFactory();

        ///
        virtual ~ViewFactory();

        /** Creates a View object identified by its View id.
            This method has to be override in case view objects are created using ViewAction::Create command.
            @param viewId the viewId of the View.
            @return <em>View</em> if the View was created,
                    <em>0</em> otherwise. */
        virtual View * Create(const ViewId & viewId);

        /** Creates a View object identified by its View id string.
            This method has to be override in case view objects are created using ViewAction::Create command.
            @param viewId the string Id of the View.
            @return <em>View</em> if the View was created,
                    <em>0</em> otherwise. */
        virtual View * Create(const Char * viewId);

        /** Destroys a View object.
            This method has to be override in case view objects are created using ViewAction::Destroy command.
            @param view the pointer to the view object. */
        virtual void Destroy(View * view);

        /** Creates a dynamically allocated empty ViewScene2D or ViewScene3D object when using ViewAction::CreateAll command.
            This method might be override if View classes derived from ViewScene2D and/or ViewScene3D shall be created.
            @param is2D if an 2D view shall be created, 3D otherwise.
            @return <em>ViewScene</em> a 2D or 3D object. */
        virtual ViewScene * CreateViewScene(bool is2D);

        /** Creates a dynamically allocated ViewContainer when using when using ViewAction::CreateAll command.
            This method shall be override if an own class derived from ViewContainer shall be created.
            @return <em>ViewContainer</em> a ViewContainer object. */
        virtual ViewContainer * CreateViewContainer();

        /** Destroys dynamically allocated ViewContainer, ViewScene2D or ViewScene3D objects.
            This method shall be override if an own allocation method was used for creation.
            @param view the pointer to the view object. */
        virtual void DestroyView(View * view);

    protected:
        /// Access method to the uses ViewHandler.
        IViewHandler * GetViewHandler() {
            FEATSTD_DEBUG_ASSERT(mViewHandler!=0);
            return mViewHandler;
        }

    private:
        /// Sets the ViewHandler which shall be used.
        void SetViewHandler(IViewHandler * viewHandler) {
            FEATSTD_DEBUG_ASSERT(viewHandler!=0);
            mViewHandler = viewHandler;
        }

        /// The used ViewHandler.
        IViewHandler * mViewHandler;
};
/// @}
}
#endif
