//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewFacade.h"

#include "ViewFactory.h"
#include "IViewHandler.h"
#include "ViewHandler.h"
#include "VisualizationMsgs.h"
#include "Transition.h"

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

// ------------------------------------------------------------------------
bool ViewFacade::mLoadSceneOnViewActivation = true;
bool ViewFacade::mLoadSceneOnViewEnableRendering = true;

// ------------------------------------------------------------------------
ViewFacade::ViewFacade() : mViewHandler(0), mViewPlaceholderVct()
{
}

// ------------------------------------------------------------------------
ViewFacade::~ViewFacade()
{
    mViewHandler = 0;
    mViewPlaceholderVct.Clear();
}

// ------------------------------------------------------------------------
bool ViewFacade::Init(IViewHandler * viewHandler)
{
    FEATSTD_DEBUG_ASSERT(viewHandler!=0);
    mViewHandler = viewHandler;
    return (0 != mViewHandler);
}

// ------------------------------------------------------------------------
bool ViewFacade::Render(RenderHint * renderHint)
{
    FEATSTD_DEBUG_ASSERT(renderHint!=0);
    bool rc = false;
    if((0 != mViewHandler) && (0!=renderHint)) {
        rc = mViewHandler->Render(renderHint);
    }
    if (View::IncrementInvalidationId() && (0 != mViewHandler)) {
        View* view = mViewHandler->FindView(ViewId());
        if (0 != view) {
            view->OnInvalidationIdOverflow();
        }
    }
    return rc;
}

// ------------------------------------------------------------------------
void ViewFacade::Update(RenderHint * renderHint)
{
    if((0 != mViewHandler) && (0!=renderHint)) {
        mViewHandler->Update(renderHint);
    }
}

// ------------------------------------------------------------------------
bool ViewFacade::PartialLoad()
{
    if(0 != mViewHandler) {
        return mViewHandler->PartialLoad();
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewFacade::OnRenderComponentMessage(ComponentId id, RenderHint * renderHint, const Message & msg)
{
    COURIER_UNUSED(renderHint);

    bool rc = false;
    if(0 != mViewHandler) {
        switch(msg.GetId()) {
            // ------------------------------------------------------------
            case RenderReqMsg::ID: {
                const RenderReqMsg * msgPtr = Courier::message_cast<const RenderReqMsg*>(&msg);
                const bool invalidateAllRendertargets = msgPtr->GetRenderAll();
                COURIER_LOG_DEBUG("RenderReqMsg: invalidateAllRendertargets(%s)",(invalidateAllRendertargets ? "TRUE" : "FALSE"));
                mViewHandler->InvalidateAll(invalidateAllRendertargets, FeatStd::Optional<Candera::Rectangle>());
                Message * postMsg = (COURIER_MESSAGE_NEW(RenderResMsg)(id,rc));
                rc = (postMsg!=0) && (postMsg->Post());
                break;
            }
            // ------------------------------------------------------------
            case LayerReqMsg::ID: {
                const LayerReqMsg * msgPtr = Courier::message_cast<const LayerReqMsg*>(&msg);
                const Int layerId = msgPtr->GetLayer();
                const bool activate = msgPtr->GetActivate();
                COURIER_LOG_DEBUG("LayerReqMsg: %s layer %d",(activate ? "ACTIVATE" : "DEACTIVATE"),layerId);
                rc = mViewHandler->ActivateLayer(layerId,activate);
                if(! rc) {
                    COURIER_LOG_ERROR("LayerReqMsg: Cannot %s layer %d",(activate ? "ACTIVATE" : "DEACTIVATE"),layerId);
                }
                Message * postMsg = (COURIER_MESSAGE_NEW(LayerResMsg)(id,rc));
                rc = (postMsg!=0) && (postMsg->Post());
                break;
            }
            // ------------------------------------------------------------
            default: {
                rc = false;
            }
        }
    }
    return rc;
}

// ------------------------------------------------------------------------
bool ViewFacade::OnViewComponentMessage(const Message & msg)
{
    bool rc = false;
    if(0 != mViewHandler) {
        // check if we have a message which shall be sent directly to a view
        if (msg.GetTag() == MessageTag::View) {
            // cast it to ViewMsgBase without using message_cast (because of checking IDs)
            FEATSTD_LINT_NEXT_EXPRESSION(826, "this cast is okay")
            const ViewMsgBase * msgPtr = static_cast<const ViewMsgBase*>(&msg);
            const ViewId & viewId = msgPtr->GetViewId();
            View * view = mViewHandler->FindView(viewId);
            rc = (view!=0);
            if(rc) {
                COURIER_LOG_DEBUG("Routing %s to View '%s'", msg.GetName(), viewId.CStr());
                rc = view->OnMessage(msg);
            } else {
                COURIER_LOG_ERROR("View '%s' not existing, message %s", viewId.CStr(),msg.GetName());
            }
        } else if (msg.GetTag() == MessageTag::Widget) {
            // cast it to WidgetMsgBase without using message_cast (because of checking IDs)
            FEATSTD_LINT_NEXT_EXPRESSION(826, "this cast is okay")
            const WidgetMsgBase * msgPtr = static_cast<const WidgetMsgBase*>(&msg);
            const ViewId & viewId = msgPtr->GetViewId();
            View * view = mViewHandler->FindView(viewId);
            rc = (view!=0);
            if(rc) {
                const CompositePath & compositePath = msgPtr->GetCompositePath();
                const ItemId & itemId = msgPtr->GetItemId();
                FrameworkWidget * widget = view->GetFrameworkWidget(compositePath, itemId);
                if(widget!=0) {
                    COURIER_LOG_DEBUG("Routing %s to Widget '%s' View '%s'", msg.GetName(), itemId.CStr(),viewId.CStr());
                    rc = widget->OnMessage(msg);
                } else {
                    COURIER_LOG_ERROR("Widget '%s' not existing, message %s", itemId.CStr(),msg.GetName());
                }
            } else {
                COURIER_LOG_ERROR("View '%s' not existing, message %s", viewId.CStr(),msg.GetName());
            }
        } else {
            switch(msg.GetId()) {
                // ------------------------------------------------------------
                case ViewReqMsg::ID: {
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                    const ViewReqMsg * msgPtr = Courier::message_cast<const ViewReqMsg*>(&msg);
                    const ViewAction::Enum action = msgPtr->GetViewAction();
                    const ViewId & viewId = msgPtr->GetViewId();
                    const bool shallInitScene = msgPtr->GetShallInitScene();
                    const bool shallLoadScene = msgPtr->GetShallLoadScene();
                    COURIER_LOG_DEBUG("ViewReqMsg: Action %d on '%s', shallInitScene(%s), shallload(%s)",action,viewId.CStr(),
                                      (shallInitScene ? "TRUE" : "FALSE"), (shallLoadScene ? "TRUE" : "FALSE"));
                    rc = mViewHandler->ExecuteViewAction(action,viewId,shallLoadScene,shallInitScene);
                    if(! rc) {
                        COURIER_LOG_ERROR("ViewReqMsg: Action %d on '%s' failed, shallInitScene(%s), shallload(%s), loaderror(%d)",action,viewId.CStr(),
                                          (shallInitScene ? "TRUE" : "FALSE"), (shallLoadScene ? "TRUE" : "FALSE"),ViewScene::GetLastLoadError());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(ViewResMsg)(action,viewId,rc,ViewScene::GetLastLoadError()));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case InvalidateReqMsg::ID: {
                    const InvalidateReqMsg * msgPtr = Courier::message_cast<const InvalidateReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    COURIER_LOG_DEBUG("InvalidateReqMsg: invalidate '%s'",viewId.CStr());
                    View * view = mViewHandler->FindView(viewId);
                    rc = (view!=0);
                    if(rc) {
                        view->Invalidate(FeatStd::Optional<Candera::Rectangle>());
                    } else {
                        COURIER_LOG_ERROR("InvalidateReqMsg: View '%s' not existing", viewId.CStr(),msg.GetName());
                    }
                    break;
                }

                // ------------------------------------------------------------
                case TransitionReqMsg::ID: {
                    const TransitionReqMsg * msgPtr = Courier::message_cast<const TransitionReqMsg*>(&msg);
                    const TransitionAction::Enum action = msgPtr->GetTransitionAction();
                    const ItemId & transitionId = msgPtr->GetTransitionId();
                    const ViewId & firstViewId = msgPtr->GetFirstViewId();
                    const ViewId & secondViewId = msgPtr->GetSecondViewId();
                    COURIER_LOG_DEBUG("TransitionReqMsg: Action %d on '%s'", action, transitionId.CStr());
                    COURIER_LOG_DEBUG("TransitionReqMsg: - first(%s) second(%s)", firstViewId.CStr(),secondViewId.CStr());
                    rc = mViewHandler->ExecuteTransitionAction(action,transitionId,firstViewId,secondViewId,msgPtr->GetOptionalPayload());
                    if(! rc) {
                        if (action == TransitionAction::Finish) {
                            COURIER_LOG_INFO("TransitionReqMsg: Action %d on '%s' failed", action, transitionId.CStr());
                            COURIER_LOG_INFO("TransitionReqMsg: - first(%s) second(%s)", firstViewId.CStr(),secondViewId.CStr());
                        }
                        else {
                            COURIER_LOG_ERROR("TransitionReqMsg: Action %d on '%s' failed", action, transitionId.CStr());
                            COURIER_LOG_ERROR("TransitionReqMsg: - first(%s) second(%s)", firstViewId.CStr(),secondViewId.CStr());
                        }
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(TransitionResMsg)(action,transitionId,firstViewId,secondViewId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }
#ifdef CANDERA_TRANSITIONS_ENABLED
                // ------------------------------------------------------------
                case CanderaTransitionRequestMsg::ID: {
                    const CanderaTransitionRequestMsg * msgPtr = Courier::message_cast<const CanderaTransitionRequestMsg*>(&msg);
                    const CanderaTransitionRequestAction::Enum action = msgPtr->GetRequestAction();
                    const Candera::Transitions::Identifier identifier = msgPtr->GetRequestIdentifier();
                    const Candera::Transitions::Hint hint = msgPtr->GetRequestHint();
                    rc = mViewHandler->ExecuteCanderaTransitionReqAction(action, identifier, hint);
                    Message * postMsg = (COURIER_MESSAGE_NEW(CanderaTransitionResponseMsg)(action, rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }
                // ------------------------------------------------------------
                case CanderaTransitionControlFlowReqMsg::ID: {
                    const CanderaTransitionControlFlowReqMsg* msgPtr = Courier::message_cast<const CanderaTransitionControlFlowReqMsg*>(&msg);
                    const CanderaTransitionControlFlowAction::Enum action = msgPtr->GetControlFlowAction();
                    rc = mViewHandler->ExecuteCanderaTransitionControlFlowAction(action);
                    Message * postMsg = (COURIER_MESSAGE_NEW(CanderaTransitionControlFlowResMsg)(action, rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }
#endif
                // ------------------------------------------------------------
                case ThemeReqMsg::ID: {
                    const ThemeReqMsg * msgPtr = Courier::message_cast<const ThemeReqMsg*>(&msg);
                    const ItemId & themeId = msgPtr->GetThemeId();
                    COURIER_LOG_DEBUG("ThemeReqMsg: Setting '%s'",themeId.CStr());
                    rc = mViewHandler->ChangeTheme(themeId);
                    if(! rc) {
                        COURIER_LOG_ERROR("ThemeReqMsg: Setting '%s' failed",themeId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(ThemeResMsg)(rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case ActivationReqMsg::ID: {
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                    const ActivationReqMsg * msgPtr = Courier::message_cast<const ActivationReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (view!=0);
                    if(rc) {
                        const bool activation = msgPtr->GetActivation();
                        const bool enableRendering = msgPtr->GetEnableRendering();
                        COURIER_LOG_DEBUG("ActivationReqMsg: View '%s' Activate(%s), Render(%s) ",viewId.CStr(),
                                          (activation ? "TRUE" : "FALSE"), (enableRendering ? "TRUE" : "FALSE"));
                        // check if rendering is enabled and then if we shall load the Scene;
                        // for backwards compatibility check also, if LoadSceneOnViewEnableRendering is turned off, if activation shall load the Scene
                        if ((enableRendering && mLoadSceneOnViewEnableRendering) || 
                            ((!mLoadSceneOnViewEnableRendering) && activation && mLoadSceneOnViewActivation)) {
                                // force load
                                ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                                COURIER_LOG_DEBUG("ActivationReqMsg: LoadContent enableRendering = %s, LoadSceneOnViewEnableRendering = %s, activate = %s, LoadSceneOnViewActivation = %s",
                                    (enableRendering) ? "TRUE" : "FALSE",
                                    (mLoadSceneOnViewEnableRendering) ? "TRUE" : "FALSE",
                                    (activation) ? "TRUE" : "FALSE",
                                    (mLoadSceneOnViewActivation) ? "TRUE" :"FALSE");
                                rc = view->LoadContent(true, true);
                                if (!rc) {
                                    COURIER_LOG_ERROR("ActivationReqMsg: Cannot load '%s' loaderror(%d)", viewId.CStr(), ViewScene::GetLastLoadError());
                                }
                        }
                        view->Activate(activation);
                        view->EnableRendering(enableRendering);
                        view->Invalidate(FeatStd::Optional<Candera::Rectangle>());
                    } else {
                        COURIER_LOG_ERROR("ActivationReqMsg: '%s' not found",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(ActivationResMsg)(viewId,rc,ViewScene::GetLastLoadError()));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case ViewRenderingReqMsg::ID: {
                    const ViewRenderingReqMsg * msgPtr = Courier::message_cast<const ViewRenderingReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (0 != view);
                    if(rc) {
                        const bool enableRendering = msgPtr->GetEnableRendering();
                        if (enableRendering && (! view->IsContentLoaded()) && mLoadSceneOnViewEnableRendering) {
                            // force load
                            ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                            COURIER_LOG_DEBUG("ViewRenderingReqMsg: LoadContent enableRendering = %s, LoadSceneOnViewEnableRendering = %s",
                                (enableRendering) ? "TRUE" : "FALSE",
                                (mLoadSceneOnViewEnableRendering) ? "TRUE" : "FALSE");
                            rc = view->LoadContent(true, true);
                            if (!rc) {
                                COURIER_LOG_ERROR("ViewRenderingReqMsg: Cannot load '%s' loaderror(%d)", viewId.CStr(), ViewScene::GetLastLoadError());
                            }
                        }
                        COURIER_LOG_DEBUG("ViewRenderingReqMsg: View '%s' Render(%s) ", viewId.CStr(), (enableRendering ? "TRUE" : "FALSE"));
                        view->EnableRendering(enableRendering);
                        view->Invalidate(FeatStd::Optional<Candera::Rectangle>());
                    } else {
                        COURIER_LOG_ERROR("ViewRenderingReqMsg: '%s' not found",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(ViewRenderingResMsg)(viewId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case ViewMessagingReqMsg::ID: {
                    const ViewMessagingReqMsg * msgPtr = Courier::message_cast<const ViewMessagingReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (0 != view);
                    if(rc) {
                        const bool enableMessaging = msgPtr->GetEnableMessaging();
                        COURIER_LOG_DEBUG("ViewMessagingReqMsg: View '%s' EnableMessaging(%s) ", viewId.CStr(), (enableMessaging ? "TRUE" : "FALSE"));
                        if (enableMessaging && (!view->IsContentLoaded()) && mLoadSceneOnViewActivation) {
                            // force load
                            ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                            COURIER_LOG_DEBUG("ViewMessagingReqMsg: LoadContent enableMessaging = %s, LoadSceneOnViewActivation = %s",
                                (enableMessaging) ? "TRUE" : "FALSE",
                                (mLoadSceneOnViewActivation) ? "TRUE" :"FALSE");
                            rc = view->LoadContent(true, true);
                            if (!rc) {
                                COURIER_LOG_ERROR("ViewMessagingReqMsg: Cannot load '%s' loaderror(%d)",viewId.CStr(),ViewScene::GetLastLoadError());
                            }
                        }
                        view->Activate(enableMessaging);
                    } else {
                        COURIER_LOG_ERROR("ViewMessagingReqMsg: '%s' not found",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(ViewMessagingResMsg)(viewId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case TryLoadReqMsg::ID: {
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                    const TryLoadReqMsg * msgPtr = Courier::message_cast<const TryLoadReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (view!=0);
                    if(rc) {
                        rc = view->LoadContent(true, false);
                        COURIER_LOG_DEBUG("TryLoadReqMsg: LOAD on '%s' ", viewId.CStr());
                        if(! rc) {
                            COURIER_LOG_ERROR("TryLoadReqMsg: Cannot LOAD '%s' loaderror(%d)",viewId.CStr(),ViewScene::GetLastLoadError());
                        }
                    } else {
                        COURIER_LOG_ERROR("TryLoadReqMsg: '%s' not found",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(LoadResMsg)(viewId,rc,ViewScene::GetLastLoadError()));
                    rc = (0 != postMsg) && postMsg->Post();
                    break;
                }

                // ------------------------------------------------------------
                case LoadReqMsg::ID: {
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                    const LoadReqMsg * msgPtr = Courier::message_cast<const LoadReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (view!=0);
                    if(rc) {
                        const bool loadContent = msgPtr->GetLoad();
                        COURIER_LOG_DEBUG("LoadReqMsg: %s on '%s' ",(loadContent ? "LOAD" : "UNLOAD"),viewId.CStr());
                        rc = view->LoadContent(loadContent, true);
                        if(! rc) {
                            COURIER_LOG_ERROR("LoadReqMsg: Cannot %s '%s' loaderror(%d)",(loadContent ? "LOAD" : "UNLOAD"),viewId.CStr(),ViewScene::GetLastLoadError());
                        }
                    } else {
                        COURIER_LOG_ERROR("LoadReqMsg: '%s' not found",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(LoadResMsg)(viewId,rc,ViewScene::GetLastLoadError()));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case TryAsyncLoadReqMsg::ID: {
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                    const TryAsyncLoadReqMsg * msgPtr = Courier::message_cast<const TryAsyncLoadReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (view!=0) && (! view->IsContainer());
                    ViewScene * viewScene = 0;
                    if(rc) {
                        viewScene = static_cast<ViewScene*>(view);
                    }
                    bool startSuccess = false;
                    if(rc && (viewScene!=0)) {
                        COURIER_LOG_DEBUG("TryAsyncLoadReqMsg: LOAD on '%s' ",viewId.CStr());
                        startSuccess = viewScene->AsyncLoadContent(true, false);
                        if(! startSuccess) {
                            COURIER_LOG_ERROR("TryAsyncLoadReqMsg: Cannot Start LOAD '%s' loaderror(%d)",viewId.CStr(),ViewScene::GetLastLoadError());
                        }
                    } else {
                        COURIER_LOG_ERROR("TryAsyncLoadReqMsg: '%s' not found (or ViewContainer)",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(AsyncLoadResMsg)(viewId,rc,ViewScene::GetLastLoadError()));
                    rc = (postMsg!=0) && (postMsg->Post());
                    if(rc && (viewScene!=0) && startSuccess) {
                        // check if we shall send the async message if we are already finished. 
                        if(viewScene->IsUploaded()) {
                            COURIER_LOG_DEBUG("AsyncLoadIndMsg: LOAD on '%s' already finished",viewId.CStr());
                            postMsg = (COURIER_MESSAGE_NEW(AsyncLoadIndMsg)(viewId,true,true,ViewScene::GetLastLoadError()));
                            rc = (postMsg!=0) && (postMsg->Post());
                        }
                    }
                    break;
                }

                // ------------------------------------------------------------
                case AsyncLoadReqMsg::ID: {
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                    const AsyncLoadReqMsg * msgPtr = Courier::message_cast<const AsyncLoadReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    const bool loadContent = msgPtr->GetLoad();
                    View * view = mViewHandler->FindView(viewId);
                    rc = (view!=0) && (! view->IsContainer());
                    ViewScene * viewScene = 0;
                    if(rc) {
                        viewScene = static_cast<ViewScene*>(view);
                    }
                    bool startSuccess = false;
                    if(rc && (viewScene!=0)) {
                        COURIER_LOG_DEBUG("AsyncLoadReqMsg: %s on '%s' ",(loadContent ? "LOAD" : "UNLOAD"),viewId.CStr());
                        // check if we scene is not already loaded / unloaded. 
                        if((loadContent && viewScene->IsUploaded()) || ( (!loadContent) && (!viewScene->IsUploaded()))) {
                            COURIER_LOG_DEBUG("AsyncLoadIndMsg: %s on '%s' already finished",(loadContent ? "LOAD" : "UNLOAD"),viewId.CStr());
                            Message *postMsg = (COURIER_MESSAGE_NEW(AsyncLoadIndMsg)(viewId,loadContent,true,ViewScene::GetLastLoadError()));
                            rc = (postMsg!=0) && (postMsg->Post());
                        }
                        else {
                            startSuccess = viewScene->AsyncLoadContent(loadContent, true);
                            if(! startSuccess) {
                                COURIER_LOG_ERROR("AsyncLoadReqMsg: Cannot Start %s '%s' loaderror(%d)",(loadContent ? "LOAD" : "UNLOAD"),viewId.CStr(),ViewScene::GetLastLoadError());
                            }
                        }
                    } else {
                        COURIER_LOG_ERROR("AsyncLoadReqMsg: '%s' not found (or ViewContainer)",viewId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(AsyncLoadResMsg)(viewId,rc,ViewScene::GetLastLoadError()));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case AnimationReqMsg::ID: {
                    const AnimationReqMsg * msgPtr = Courier::message_cast<const AnimationReqMsg*>(&msg);
                    const AnimationAction::Enum action = msgPtr->GetAnimationAction();
                    const ViewId & viewId = msgPtr->GetViewId();
                    const CompositePath & compositePath = msgPtr->GetCompositePath();
                    const ItemId & animationId = msgPtr->GetAnimationId();
                    // TODO: support StringIdentifier in Log
                    const FeatStd::Char* compositePathName = (0 == compositePath.ToCString())?"StringIdentifierNotSupportedInLog":compositePath.ToCString();
                    FEATSTD_UNUSED(compositePathName); // if logging is disabled, variable/parameter is not used
                    COURIER_LOG_DEBUG("AnimationReqMsg: Action %d on '%s'/'%s'/'%s'", action, viewId.CStr(), compositePathName, animationId.CStr());
                    rc = mViewHandler->ExecuteAnimationAction(action, viewId, compositePath, animationId, msgPtr->GetAnimationProperties());
                    if(! rc) {
                        COURIER_LOG_ERROR("AnimationReqMsg: Action %d on '%s'/'%s'/'%s' failed", action, viewId.CStr(), compositePathName, animationId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(AnimationResMsg)(action, viewId, compositePath, animationId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }
                // ------------------------------------------------------------
                case SetPropertyReqMsg::ID: {
                    const SetPropertyReqMsg * msgPtr = Courier::message_cast<const SetPropertyReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    const CompositePath & compositePath = msgPtr->GetCompositePath();
                    const ItemId & widgetId = msgPtr->GetWidgetId();
                    const ItemId & propertyId = msgPtr->GetPropertyId();
                    const Payload & value = msgPtr->GetValue();
                    // TODO: support StringIdentifier in Log
                    const FeatStd::Char* compositePathName = (0 == compositePath.ToCString())?"StringIdentifierNotSupportedInLog":compositePath.ToCString();
                    FEATSTD_UNUSED(compositePathName); // if logging is disabled, variable/parameter is not used
                    COURIER_LOG_DEBUG("SetPropertyReqMsg: to '%s'/'%s'/'%s'", viewId.CStr(), compositePathName, widgetId.CStr());
                    COURIER_LOG_DEBUG("SetPropertyReqMsg: -- property/value '%s'/'%s'", propertyId.CStr(), value.CStr());
                    rc = mViewHandler->SetWidgetProperty(viewId, compositePath, widgetId, propertyId, value.CStr());
                    if (!rc) {
                        COURIER_LOG_ERROR("SetPropertyReqMsg: to '%s'/'%s'/'%s' failed", viewId.CStr(), compositePathName, widgetId.CStr());
                        COURIER_LOG_ERROR("SetPropertyReqMsg: -- property/value '%s'/'%s'", propertyId.CStr(),value.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(SetPropertyResMsg)(viewId, compositePath, widgetId, propertyId, rc));
                    rc = (postMsg != 0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case SetFocusReqMsg::ID: {
                    const SetFocusReqMsg * msgPtr = Courier::message_cast<const SetFocusReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    const CompositePath & compositePath = msgPtr->GetCompositePath();
                    const ItemId & widgetId = msgPtr->GetWidgetId();
                    // TODO: support StringIdentifier in Log
                    const FeatStd::Char* compositePathName = (0 == compositePath.ToCString())?"StringIdentifierNotSupportedInLog":compositePath.ToCString();
                    FEATSTD_UNUSED(compositePathName); // if logging is disabled, variable/parameter is not used
                    COURIER_LOG_DEBUG("SetFocusReqMsg: set focus to '%s'/'%s'/'%s'", viewId.CStr(), compositePathName, widgetId.CStr());
                    rc = mViewHandler->SetFocus(viewId, compositePath, widgetId);
                    if(! rc) {
                        COURIER_LOG_ERROR("SetFocusReqMsg: set focus to '%s'/'%s'/'%s' failed", viewId.CStr(), compositePathName, widgetId.CStr());
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(SetFocusResMsg)(viewId, compositePath, widgetId, rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }
                // ------------------------------------------------------------
                case SetCultureReqMsg::ID: {
                    const SetCultureReqMsg * msgPtr = Courier::message_cast<const SetCultureReqMsg*>(&msg);
                    const ItemId & itemId = msgPtr->GetLocale();
                    const Char * locale = itemId.CStr();
                    FEATSTD_UNUSED(locale); // if logging is disabled, variable/parameter is not used
                    COURIER_LOG_DEBUG("SetCultureReqMsg: change culture to '%s'", locale);
                    rc = mViewHandler->SetCurrentCulture(itemId.CStr());
                    Message * postMsg = (COURIER_MESSAGE_NEW(SetCultureResMsg)(rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case CameraGroupReqMsg::ID: {
                    const CameraGroupReqMsg * msgPtr = Courier::message_cast<const CameraGroupReqMsg*>(&msg);
                    const ViewAction::Enum action = msgPtr->GetViewAction();
                    const ItemId & cameraGroupId = msgPtr->GetCameraGroupId();
                    const bool shallLoadScene = msgPtr->GetShallLoadScene();
                    COURIER_LOG_DEBUG("CameraGroupReqMsg: Action %d on '%s', shallload(%s)",action,cameraGroupId.CStr(),
                                      (shallLoadScene ? "TRUE" : "FALSE"));
                    CameraGroupHandler * cgh = mViewHandler->GetCameraGroupHandler();
                    rc = (cgh!=0);
                    if(rc) {
                        rc = cgh->CreateCameraGroupViews(action,cameraGroupId,shallLoadScene);
                    }
                    if(! rc) {
                        COURIER_LOG_ERROR("CameraGroupReqMsg: Action %d on '%s' failed, shallload(%s)",action,cameraGroupId.CStr(),
                                          (shallLoadScene ? "TRUE" : "FALSE"));
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(CameraGroupResMsg)(action,cameraGroupId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case CameraGroupActivationReqMsg::ID: {
                    const CameraGroupActivationReqMsg * msgPtr = Courier::message_cast<const CameraGroupActivationReqMsg*>(&msg);
                    const ItemId & cameraGroupId = msgPtr->GetCameraGroupId();
                    const bool activation = msgPtr->GetActivation();
                    const bool enableRendering = msgPtr->GetEnableRendering();
                    const bool onlyCameraGroup = msgPtr->GetOnlyCameraGroupCameras();

                    COURIER_LOG_DEBUG("CameraGroupActivationReqMsg: CamGrp '%s' Activate(%s), Render(%s), OnlyCamGrp(%s)",cameraGroupId.CStr(),
                                        (activation ? "TRUE" : "FALSE"), (enableRendering ? "TRUE" : "FALSE"), (onlyCameraGroup ? "TRUE" : "FALSE"));
                    CameraGroupHandler * cgh = mViewHandler->GetCameraGroupHandler();
                    rc = (cgh!=0);
                    if(rc) {
                        rc = cgh->ActivateCameraGroupCameras(cameraGroupId,activation,enableRendering,onlyCameraGroup);
                    }
                    if(! rc) {
                        COURIER_LOG_ERROR("CameraGroupActivationReqMsg: CamGrp '%s' Activate(%s), Render(%s), OnlyCamGrp(%s)",cameraGroupId.CStr(),
                                          (activation ? "TRUE" : "FALSE"), (enableRendering ? "TRUE" : "FALSE"), (onlyCameraGroup ? "TRUE" : "FALSE"));
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(CameraGroupActivationResMsg)(cameraGroupId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }

                // ------------------------------------------------------------
                case CameraGroupLoadReqMsg::ID: {
                    const CameraGroupLoadReqMsg * msgPtr = Courier::message_cast<const CameraGroupLoadReqMsg*>(&msg);
                    const ItemId & cameraGroupId = msgPtr->GetCameraGroupId();
                    const bool loadContent = msgPtr->GetLoad();
                    COURIER_LOG_DEBUG("CameraGroupLoadReqMsg: %s on '%s' ",(loadContent ? "LOAD" : "UNLOAD"),cameraGroupId.CStr());
                    CameraGroupHandler * cgh = mViewHandler->GetCameraGroupHandler();
                    rc = (cgh!=0);
                    if(rc) {
                        rc = cgh->LoadCameraGroupViews(cameraGroupId,loadContent);
                    }
                    if(! rc) {
                        COURIER_LOG_ERROR("CameraGroupLoadReqMsg: CamGrp '%s' failed, loadContent(%s)",cameraGroupId.CStr(),
                                         (loadContent ? "TRUE" : "FALSE"));
                    }
                    Message * postMsg = (COURIER_MESSAGE_NEW(CameraGroupLoadResMsg)(cameraGroupId,rc));
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                }
                // ------------------------------------------------------------
                case ViewPlaceholderReqMsg::ID: {
                    const ViewPlaceholderReqMsg * msgPtr = Courier::message_cast<const ViewPlaceholderReqMsg*>(&msg);
                    const ViewId & viewId = msgPtr->GetViewId();
                    View* view = mViewHandler->FindView(viewId);
                    const bool load = msgPtr->GetLoad();

                    COURIER_LOG_DEBUG("ViewPlaceholderReqMsg: View '%s'  ", viewId.CStr());

                    rc = false;
                    ViewPlaceholderLoadError loadError = ViewPlaceholderSuccess;
                    FeatStd::SizeType placeholderPos = FindViewPlaceholder(viewId);

                    if (load) {
                        if (0 != view) {
                            if (FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max() != placeholderPos) {
                                loadError = ViewPlaceholderAlreadyLoaded;
                            } else {
                                if (mViewPlaceholderVct.Add(Internal::ViewPlaceholder())) {
                                    rc = mViewPlaceholderVct[mViewPlaceholderVct.Size() - 1].Upload(view,  mViewHandler->GetRenderer());
                                }
                            }
                        } else {
                            loadError = NoViewForViewPlaceholder;
                        }
                    } else {
                        if (FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max() != placeholderPos) {
                            rc = mViewPlaceholderVct[placeholderPos].Unload(mViewHandler->GetRenderer());
                            static_cast<void>(mViewPlaceholderVct.Remove(placeholderPos));
                        } else {
                            loadError = ViewPlaceholderNotLoaded;
                        }
                    }
                    Message * postMsg = COURIER_MESSAGE_NEW(ViewPlaceholderResMsg)(load, rc, loadError);
                    rc = (postMsg!=0) && (postMsg->Post());
                    break;
                 }

                // ------------------------------------------------------------
                default: {
                    const CultureChangeIndMsg * cultureChangeIndMsgPtr = Courier::message_cast<const CultureChangeIndMsg*>(&msg);
                    if(cultureChangeIndMsgPtr!=0) {
                        Candera::Globalization::Culture::SharedPointer culturePtr = cultureChangeIndMsgPtr->GetCulturePtr();
                        COURIER_LOG_INFO("CultureChangeIndMsg received: Locale(%s)",(!culturePtr.PointsToNull()) ? culturePtr->GetLocale() : "not set");
                    }

                    // process messages and send them to the views ...
                    COURIER_LOG_DEBUG("Routing %s through View tree", msg.GetName());
                    rc = mViewHandler->OnMessage(msg);
                }
            }
        }
    }
    return rc;
}

// ------------------------------------------------------------------------
FeatStd::SizeType ViewFacade::FindViewPlaceholder(const ViewId& viewId)
{
    FeatStd::SizeType placeholderPosition = FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max();
    for (FeatStd::SizeType i = 0; i <  mViewPlaceholderVct.Size(); ++i) {
        if (viewId == mViewPlaceholderVct[i].GetViewId()) {
            placeholderPosition = i;
        }
    }
    return placeholderPosition;
}

} // namespace Courier

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {

template<> UInt32 StringBufferAppender<Candera::Globalization::TextDirection>::Append(StringBuffer& stringBuffer, Candera::Globalization::TextDirection const & object)
{
    UInt32 tcharCount = 0;
    switch (object) {
    case ::Candera::Globalization::_undefined:
        tcharCount += stringBuffer.Append("::Candera::AnimationPlayer::_undefined");
        break;
    case ::Candera::Globalization::LeftToRight:
        tcharCount += stringBuffer.Append("::Candera::AnimationPlayer::LeftToRight");
        break;
    case ::Candera::Globalization::RightToLeft:
        tcharCount += stringBuffer.Append("::Candera::AnimationPlayer::RightToLeft");
        break;
    default:
        tcharCount += stringBuffer.Append("##unknown##");
        break;
    }
    return tcharCount;
}

template<> UInt32 StringBufferAppender<Candera::Globalization::Culture>::Append(StringBuffer& stringBuffer, Candera::Globalization::Culture const & object)
{
    UInt32 tcharCount = 0;
    tcharCount += stringBuffer.Append("Candera::Globalization::Culture { locale = ");
    tcharCount += stringBuffer.AppendObject(object.GetLocale());
    tcharCount += stringBuffer.Append(", displayName = ");
    tcharCount += stringBuffer.AppendObject(object.GetDisplayName());
    tcharCount += stringBuffer.Append(", textDirection = ");
    tcharCount += stringBuffer.AppendObject(object.GetTextDirection());
    tcharCount += stringBuffer.Append(" }");
    return tcharCount;
}

}
#endif
