//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewControllerFactory_h)
#define Courier_Visualization_ViewControllerFactory_h

#include <Courier/Base.h>

namespace Courier { 
    class ViewController;
    class ViewId;
}

namespace Courier { 
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A ViewControllerFactory is responsible for creating ViewControllers.
    Usage of a ViewControllerFactory is optional. It has to be derived from this class. If an instance is set to the
    ViewHandler using the SetViewControllerFactory method the ViewControllerFactory::Create method will be called each time a View is created.
    This allows returning a specific ViewController for each View. Returning 0 is allowed if a View shall not use a ViewController.
*/
class ViewControllerFactory  {
    public:
        ///
        ViewControllerFactory();

        ///
        virtual ~ViewControllerFactory();

        /** Returns a ViewController object identified by its ViewId.
            This method shall be overwritten by the derived factory and it is upto the 
            customer if the ViewController instances are allocated dynamically or statically.
            Because the used parameter is ViewId the overwritten method also may use static ViewId constants for 
            comparison and therefore is faster then comparing strings.
            If this method is not overwritten it is delegated to the Create(const Char * viewId) method,
            which does a slower string compare and is backwards compatible. 
            @param viewId the ViewId of the view. 
            @return <em>ViewController</em> if the ViewController was created,
                    <em>0</em> otherwise. */
        virtual ViewController * Create(const ViewId & viewId);

        /** Returns a ViewController object identified by its view string.
            This method is still available for backward compatibility. Overwriting method Create(const ViewId & viewId) is prefered.
            @param viewId the string Id of the view. 
            @return <em>ViewController</em> if the ViewController was created,
                    <em>0</em> otherwise. */
        virtual ViewController * Create(const Char * viewId);

        /** Destroys a ViewController object.
            This method has to be overwritten by the derived factory.
            @param viewController the pointer to the ViewController. */
        virtual void Destroy(ViewController * viewController) = 0;
};
/// @}

}

#endif
