//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewController_h)
#define Courier_Visualization_ViewController_h

#include <Courier/Base.h>
#include <Courier/Visualization/View.h>

namespace Courier {
    class Message;
}

// ----------------------------------------------------------------------------
namespace Courier {
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A ViewController might control a View and Widgets. It is possible to process messages
    before they are delegated to the child views and widgets. ViewController are always
    created by the ViewControllerFactory and are an optional entity for message processing.
*/

class ViewController
{
    public:
        ///
        ViewController();

        ///
        virtual ~ViewController();

        /** Returns the View the ViewController belongs to.
            This allows accessing functionality of the view.
            @return <em>View</em> the pointer to the View object,
                    <em>0</em> otherwise. */
        View * GetView() const {
            FEATSTD_DEBUG_ASSERT(mView!=0);
            return mView;
        }

        /** Sets the View the ViewController belongs to. This method is called automatically during ViewController creation.
            @param view View controlled by the ViewController. */
        void SetView(View * view);

        /** Release the View the ViewController belongs to. A static allocated ViewController may be reused again because its View was freed.
        */
        void ReleaseView();

        /** Is triggered when a message is received.
            @note The return parameter defines that the message shall be routed to the succeeding message
                  receiving entities of the dispatching path (<em>true</em>) or if the message is fully
                  consumed and any further dispatching of that message shall be stopped (<em>false</em>).
            @param msg Message object to be processed.
            @return <em>true</em> if the message is consumed
                    <em>false</em> if the message shall be sent to following message receiving entities. */
        virtual bool OnMessage(const Message & msg) = 0;

        /** Is called whenever the parent view (scene) is loaded or unloaded.
            @param load if the scene context was loaded or before it gets unloaded. */
        virtual void OnParentViewLoad(bool load);

        /** Is called whenever the parent view is activated or deactivated.
            @param activate if the view was activated or deactivated. */
        virtual void OnParentViewActivate(bool activate);

        /** Is called whenever enable rendering of the parent view is enabled or disabled.
            @param enable if the rendering is enabled or disabled. */
        virtual void OnParentViewRenderingEnabled(bool enable);

        /// Returns the ScopeMask of the ViewController.
        virtual Message::ScopeMask GetScopeMask() const;

    protected:
        ///
        void DispatchEvent(const Message & msg);

    private:
        ///
        friend class View;
        friend class ViewScene;
        friend class ViewContainer;

        /// The View controlled by this ViewController.
        View * mView;
};
/// @}
}

#endif
