//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Courier_Visualization_ViewContextTraverser_h
#define Courier_Visualization_ViewContextTraverser_h

#include "Renderer.h"
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

namespace Candera {
    class RenderTarget;
}

namespace Courier
{
    class Renderer;

template<typename Camera, typename Scene, typename SceneContext, typename NodeTraverser, bool Is2D = false>
class ContextTraverser
{
public:
    typedef Vector<Camera* > CameraPtrVector;

    ContextTraverser():
        m_renderer(0),
        m_cameras(0),
        m_sceneContext(0),
        m_scene(0),
        m_currentScope(false),
        m_firstCall(true),
        m_isSuccessful(false),
        m_isFinished(false),
        m_currentRenderTarget(0),
        m_defaultResourcePoolScope(false)
    {}

    /// @return true if all the context have been traversed (activated and nodes uploaded).
    bool IsFinished() const { return m_isFinished; }

    /// @return true if the node traverser succeeded for all contexts.
    bool IsSuccessful() const { return m_isFinished && m_isSuccessful; }

    /// Init the scope masks for all the GDUs. The scope mask for a GDU contains the merged scope masks of all the cameras attached
    /// to that context.
    /// @param scene
    /// @param sceneContext
    /// @param renderer
    /// @param cameras
    void InitTraverser(Scene *scene, SceneContext *sceneContext, Renderer* renderer, const CameraPtrVector *cameras)
    {
        m_scene = scene;
        m_sceneContext = sceneContext;
        m_renderer = renderer;
        m_cameras = cameras;
        m_renderTargets.Clear();
        m_renderTargetScopeMap.Clear();
        m_currentRenderTarget = 0;
        for (typename CameraPtrVector::ConstIterator camerasIt = m_cameras->ConstBegin(); camerasIt != m_cameras->ConstEnd(); ++camerasIt) {
            Camera *camera = *(camerasIt);
            if (camera != 0) {
                Candera::RenderTarget* rendertarget = camera->GetRenderTarget();
                if (0 != rendertarget) {
                    Candera::ScopeMask* scope = m_renderTargetScopeMap.Find(rendertarget);
                    if (0 != scope) {
                        (*scope) |= camera->GetScopeMask();
                    }
                    else {
                        m_renderTargets.Add(rendertarget);
                        m_renderTargetScopeMap.Insert(rendertarget, camera->GetScopeMask());
                    }
                }
                else {
                    m_defaultResourcePoolScope |= camera->GetScopeMask();
                }
            }
        }
        m_firstCall = true;
        m_isSuccessful = true;
        m_isFinished = false;
    }

    /// Call NodeTraverser::Traverse. When all nodes have been traversed for current active context, activate next context and reset
    /// the Node Traverser
    /// @param traverser
    void Traverse(NodeTraverser *traverser)
    {
        if ((m_renderer == 0) || (traverser == 0) || (m_scene == 0) || (m_sceneContext == 0)) {
            return;
        }

        if (m_firstCall) {
            m_isFinished = MoveToNextRenderTarget();
            m_firstCall = false;
            if (!m_isFinished) {
                ActivateCurrentRenderTarget();
                // InitialTraverse should always be called
                traverser->InitialTraverse(*m_scene, *m_sceneContext, m_currentScope);
            }
        }
        else {
            if (!m_isFinished) {
                ActivateCurrentRenderTarget();
                traverser->Traverse();
            }
        }

        if (!m_isFinished && traverser->Finished())
        {
            m_isSuccessful = m_isSuccessful && traverser->IsEffectiveResultSuccessful();

            // Check if still have GDUs that were not activated
            m_isFinished = MoveToNextRenderTarget();
            if (!m_isFinished)
            {
                traverser->Reset(m_currentScope);
            }
        }
    }

private:

    /// Activate a new context (if any), reset the Node Traverser and update the scope mask for that traverser.
    bool MoveToNextRenderTarget()
    {
        if ((!m_firstCall) && (m_currentRenderTarget < m_renderTargets.Size())) {
            ++m_currentRenderTarget;
        }
        // Reset current upload scope
        m_currentScope = Candera::ScopeMask(false);
        Candera::ScopeMask allsetScopeMask(true);
        while (m_currentRenderTarget < m_renderTargets.Size()) {
            Candera::RenderTarget* renderTarget = m_renderTargets[m_currentRenderTarget];
            if (0 != renderTarget) {
                Candera::ScopeMask *scope = m_renderTargetScopeMap.Find(renderTarget);
                if (0 != scope) {
                    m_currentScope = *scope;
                    if (m_currentScope.OverlapsWith(allsetScopeMask)) {
                        return false;
                    }
                }
            }
            ++m_currentRenderTarget;
        }
        if (m_defaultResourcePoolScope.OverlapsWith(allsetScopeMask)) {
            m_currentScope = m_defaultResourcePoolScope;
            return false;
        }
        m_renderTargets.Clear();
        m_renderTargetScopeMap.Clear();
        m_currentRenderTarget = 0;
        return true; // is finished
    }

    void ActivateCurrentRenderTarget()
    {
        if (m_currentRenderTarget < m_renderTargets.Size()) {
            Candera::RenderTarget* renderTarget = m_renderTargets[m_currentRenderTarget];
            if (0 != renderTarget) {
                renderTarget->Activate();
            }
        }
        else {
            Candera::ScopeMask allsetScopeMask(true);
            if (m_defaultResourcePoolScope.OverlapsWith(allsetScopeMask)) {
                Candera::ContextResourcePool::GetDefault().Activate();
                m_defaultResourcePoolScope = Candera::ScopeMask(false);
            }
        }
    }

private:
    Renderer *m_renderer;
    Courier::Vector<Candera::RenderTarget*>  m_renderTargets;
    FeatStd::SizeType m_currentRenderTarget;

    const CameraPtrVector *m_cameras;

    SceneContext *m_sceneContext;
    Scene *m_scene;

    Candera::ScopeMask m_currentScope;

    bool m_firstCall;
    bool m_isSuccessful;
    bool m_isFinished;
    Candera::ScopeMask m_defaultResourcePoolScope;

    Candera::Internal::Map<Candera::RenderTarget*, Candera::ScopeMask> m_renderTargetScopeMap;
};

}
#endif // Courier_Visualization_ViewContextTraverser_h
