//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewContainer_h)
#define Courier_Visualization_ViewContainer_h

#include <Courier/Base.h>
#include <Courier/Visualization/View.h>

namespace Courier { 
    class ViewAcessor;
}

namespace Courier { 
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A ViewContainer is a specialized View which represents a sorted list of other Views.
*/

class ViewContainer : public View
{
    typedef View Base;

    public:
        ///
        ViewContainer(bool managed = false);

        ///
        virtual ~ViewContainer();

        ///
        virtual bool Init(IViewHandler * viewHandler, const Char * viewName, ViewController * viewController);

        ///
        virtual bool Init(IViewHandler * viewHandler, const Char * viewName);

        /** Returns the string identification of the ViewContainer.
            @return <em>string</em> the string identifier, in this case the name of the folder. */
        virtual const ViewId & GetId() const;

        ///
        virtual bool AddView(View * view);

        ///
        virtual bool RemoveView(View * view);

        /** Activates all child views for message receiving, regardless of their rendering state.
            Optional ViewController is informed by calling method OnParentViewActivate.
            @param activate Activates or deactivates message receiving. */
        virtual void Activate(bool activate);

        /** Indicates if one of the child views is activated, means if its capable of message receiving.
            @return <em>true</em> if at least one child view is activated,
                    <em>false</em> otherwise. */
        virtual bool IsActive() const;

        /** Enables rendering of all child views and their cameras.
            @param enable Enables or disables cameras. */
        virtual void EnableRendering(bool enable);

        /** Returns if one of the child views has rendering enabled.
            @return <em>true</em> if the view will be rendered.
                    <em>false</em> otherwise. */
        virtual bool IsRenderingEnabled() const;

        ///
        virtual ViewContainer* ToViewContainer() { return this; }

        /// Invalidates all enabled views and their enabled cameras, regardless of their activation state, using default render counters.
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See View.h)")
        virtual void Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        /// Invalidates all enabled views and their enabled cameras, regardless of their activation state.
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See View.h)")
        virtual void Invalidate(UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        ///
        FEATSTD_LINT_NEXT_EXPRESSION(1411, "the hidden virtual Invalidate is unlikely to be mistaken with the other Invalidate functions (receiving different parameter types but having the same purpose)")
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See View.h)")
        virtual void Invalidate(void * excludeCamera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        ///
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See View.h)")
        virtual void Invalidate(Candera::RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        ///
        virtual void Finalize();

        ///
        virtual bool InitContent(bool init);

        ///
        virtual bool LoadContent(bool load, bool forceUpload);

        /** Indicates if the content of one child view is initalized.
            @return <em>true</em> if at least one child view is initalized,
                    <em>false</em> otherwise. */
        virtual bool IsContentInitialized() const;

        /** Indicates if one of the child views is loaded.
            @return <em>true</em> if at least one child view is loaded,
                    <em>false</em> otherwise. */
        virtual bool IsContentLoaded() const;

        /// 
        virtual void Update(RenderHint * renderHint);

        ///
        virtual FrameworkWidget * GetFrameworkWidget(const CompositePath & compositePath, const ItemId & widgetId);
        using View::GetFrameworkWidget;
        ///
        virtual Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> GetAnimation(const CompositePath & compositePath, const ItemId & animationId);

        ///
        virtual View * FindView(const ViewId & viewId);

        ///
        ViewLinkedList & GetViews() { return mViews; }

        ///
        virtual void Accept(ViewVisitor & visitor);

        ///
        virtual bool GetRenderTargetPtrVector(RTPtrVector & vector, bool clear);
        ///
        virtual bool GetRenderTargetPtrVector(RTPtrVector & vector);

        /** Messages will be distributed only to active child views.
            @return <em>true</em> if at least one child view has processed the message,
                    <em>false</em> otherwise. */
        virtual bool DistributeMessage(const Message & msg);

        ///
        virtual void OnTransitionStarted(const TransitionStartedEvent & msg);

        ///
        virtual void OnTransitionFinished(const TransitionFinishedEvent & msg);

        ///
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See View.h)")
        virtual void Clear(bool onlyCameraViewPorts, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        /// Returns the touched FrameworkWidget. Iterates over all child views and calls GetTouchedWidget of each view if it is active.
        virtual FrameworkWidget * GetTouchedWidget(const TouchInfo & info);

    protected:
        virtual void OnScopeMaskChanged();

        virtual void OnInvalidationIdOverflow() override;

    private:
        /// Returns if ViewContainer is correctly initialized.
        bool IsInitialized();

        /// String represenation of the view container.
        View::Name mName;

        /// The Id of the view container.
        ViewId mContainerName;

        /// The sorted list of child views.
        ViewLinkedList mViews;
};

/// @}
}

#endif
