//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_View_h)
#define Courier_Visualization_View_h

#include <Courier/Util/Util.h>
#include <Courier/Messaging/Message.h>
#include <Courier/Visualization/ViewId.h>
#include <Courier/Visualization/ItemId.h>
#include <Courier/Visualization/CompositePath.h>
#include <FeatStd/Util/Optional.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Courier/Macros.h>
#include <FeatStd/Event/Event.h>
#include <Candera/System/Container/Map.h>

COURIER_UNIT_TEST_TESTCASE_DECLARATION(ViewScene3DTest, InvalidateTest)
COURIER_UNIT_TEST_TESTCASE_DECLARATION(ViewScene2DTest, InvalidateTest)

namespace Courier {
    class FrameworkWidget;
    class IViewHandler;
    class ViewController;
    class ViewVisitor;
    class ItemId;
    class Message;
    class RenderHint;
    struct TouchInfo;
    class TransitionStartedEvent;
    class TransitionFinishedEvent;
    class ViewScene2D;
    class ViewScene3D;
    class ViewContainer;

#if !defined(COURIER_DEFAULT_RENDER_COUNT)
    static const UInt8 cCOURIER_DEFAULT_RENDER_COUNT = 3;
    #define COURIER_DEFAULT_RENDER_COUNT Courier::cCOURIER_DEFAULT_RENDER_COUNT
#else
    static const UInt8 cCOURIER_DEFAULT_RENDER_COUNT = COURIER_DEFAULT_RENDER_COUNT;
#endif

#if !defined(COURIER_DEFAULT_RENDER_2D_COUNT)
    static const UInt8 cCOURIER_DEFAULT_RENDER_2D_COUNT = cCOURIER_DEFAULT_RENDER_COUNT;
    #define COURIER_DEFAULT_RENDER_2D_COUNT Courier::cCOURIER_DEFAULT_RENDER_2D_COUNT
#else
    static const UInt8 cCOURIER_DEFAULT_RENDER_2D_COUNT = COURIER_DEFAULT_RENDER_2D_COUNT;
#endif

#if !defined(COURIER_DEFAULT_RENDER_3D_COUNT)
    static const UInt8 cCOURIER_DEFAULT_RENDER_3D_COUNT = cCOURIER_DEFAULT_RENDER_COUNT;
    #define COURIER_DEFAULT_RENDER_3D_COUNT Courier::cCOURIER_DEFAULT_RENDER_3D_COUNT
#else
    static const UInt8 cCOURIER_DEFAULT_RENDER_3D_COUNT = COURIER_DEFAULT_RENDER_3D_COUNT;
#endif

#if !defined(COURIER_DEFAULT_MAX_VIEW_NAMELENGTH)
    static const UInt8 cCOURIER_DEFAULT_MAX_VIEW_NAMELENGTH = 64;
    #define COURIER_DEFAULT_MAX_VIEW_NAMELENGTH Courier::cCOURIER_DEFAULT_MAX_VIEW_NAMELENGTH
#else
    static const UInt8 cCOURIER_DEFAULT_MAX_VIEW_NAMELENGTH = COURIER_DEFAULT_MAX_VIEW_NAMELENGTH;
#endif

}

namespace Candera {
    class RenderTarget;
    namespace Animation {
        class AnimationPlayerBase;
    }
}

namespace FeatStd {
    class EventSource;
}

// ----------------------------------------------------------------------------
namespace Courier {

///
typedef Courier::Vector< Candera::RenderTarget *> RTPtrVector;

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A View represents an abstract visualization entity.
    It is the base class of some concrete implementations, like the ViewContainer which itself contains a list of views,
    and ViewScene2D or ViewScene3D which represents one 2D or 3D Scene which themselves contain a list of widgets. This structure allows to
    implement a hierachical view tree. A View is bound to an optional ViewController object which might control the behaviour of its child
    views and their containing widgets.
*/

class View  {
    public:
        /** A invalidation dependency represents the view and an optional camera in this view that has to be invalidated
            each time when the target view (on which this dependency is set) is invalidated by the method Invalidate().
        */
        class InvalidationDependency {
        public:
            /// The view that will be invalidated.
            const ViewId& GetViewId() const
            {
                return mViewId;
            }
            /// The camera in the dependent view that will be used for the invalidation. If not set then all cameras in the view will be invalidated.
            const ItemId& GetCameraId() const
            {
                return mCameraId;
            }
            /// The dirty area that will be used for the invalidation.
            const FeatStd::Optional<Candera::Rectangle>& GetDirtyArea() const
            {
                return mDirtyArea;
            }
            /// The next invalidation dependency or a null pointer if the end of the list has been reached.
            const InvalidationDependency* GetNext() const { return mNext; }
            ///
            bool Equals(const ViewId& viewId, const ItemId& cameraId) const
            {
                return (viewId == mViewId) && (mCameraId == cameraId);
            }

        private:
            friend class View;

            ///
            InvalidationDependency(const ViewId& viewId, const ItemId& cameraId, const FeatStd::Optional<Candera::Rectangle>& dirtyArea) :
                mViewId(viewId),
                mCameraId(cameraId),
                mDirtyArea(dirtyArea),
                mNext(0)
            {
            }

            ///
            InvalidationDependency* GetNext()
            {
                return mNext;
            }

            ///
            void SetNext(InvalidationDependency* next)
            {
                mNext = next;
            }

            ViewId mViewId;
            ItemId mCameraId;
            FeatStd::Optional<Candera::Rectangle> mDirtyArea;
            InvalidationDependency * mNext;
        };

        ///
        typedef FixedSizeString<cCOURIER_DEFAULT_MAX_VIEW_NAMELENGTH> Name;

        ///
        static const UInt8 cDefaultRender2DCounter = cCOURIER_DEFAULT_RENDER_2D_COUNT;
        ///
        static const UInt8 cDefaultRender3DCounter = cCOURIER_DEFAULT_RENDER_3D_COUNT;

        /**
         * Returns the event source of Courier Views.
         */
        static FeatStd::EventSource& GetEventSource();
        /** Constructor of the view.
            @param managed is true in case the ViewHandler created the View object using the
                           methods CreateViewScene or CreateViewContainer of the ViewFactory. */
        View(bool managed);

        /// Destructor
        virtual ~View();

        /// Frees resources used by the view.
        virtual void Finalize() = 0;

        /// Node member for using inside a LinkedList
        Internal::LinkedListNode<View> ViewListNode;

        /** Initializes the view.
            @param viewHandler IViewHandler used by the view.
            @param viewName Name of the view.
            @param viewController optional ViewController used by the view.
            @return <em>true</em> if initialization was successful,
                    <em>false</em> otherwise. */
        virtual bool Init(IViewHandler * viewHandler, const Char * viewName, ViewController * viewController);

        ///
        virtual bool Init(IViewHandler * viewHandler, const Char * viewName);

        /** Returns the identification of the view.
            @return <em>ViewId</em> the identifier. */
        virtual const ViewId & GetId() const = 0;

        /** Returns the path identification of the view, which is the identification of the views parent.
            @return <em>ViewId</em> the identifier. */
        virtual const ViewId & GetPathId() const;

        /** Adds a view object to the view list.
            @param view the view object.
            @return <em>true</em> if adding was successful,
                    <em>false</em> otherwise. */
        virtual bool AddView(View * view);

        /** Removes a view object from the view list.
            @param view the view object.
            @return <em>true</em> if removing was successful,
                    <em>false</em> otherwise. */
        virtual bool RemoveView(View * view);

        /** Enables or disables the rendering of the view and its childs.
            @param enable boolean value which enables or disables rendering. */
        virtual void EnableRendering(bool enable) = 0;

        /** Enables or disables the rendering of the scene, the rendering state of the cameras remain unchanged.
            No notifications are sent to widgets. Applicable only for ViewScene2D and ViewScene3D objects.
            @param enable boolean value which enables or disables rendering. */
        virtual void EnableRenderingScene(bool enable);

        /** Returns if the cameras of the view will be rendered.
            @return <em>true</em> if the view will be rendered.
                    <em>false</em> otherwise. */
        virtual bool IsRenderingEnabled() const = 0;

        /** Returns the widget object identified by the widget identifier.
            @return <em>FrameworkWidget</em> if the widget could be found,
                    <em>0</em> otherwise. */
        virtual FrameworkWidget * GetFrameworkWidget(const ItemId & widgetId)
        {
            return GetFrameworkWidget(CompositePath(), widgetId);
        }

        /** Returns the widget object identified by the widget identifier and the composite path.
            @return <em>FrameworkWidget</em> if the widget could be found,
                    <em>0</em> otherwise. */
        virtual FrameworkWidget * GetFrameworkWidget(const CompositePath & compositePath, const ItemId & widgetId) = 0;

        /** Returns the animation player identified by the animation identifier and the composite path.
            @return <em>AnimationPlayerBase</em> if the animation could be found,
                    <em>0</em> otherwise. */
        virtual Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> GetAnimation(const CompositePath & compositePath, const ItemId & animationId) = 0;

        /** Init or destroys the scene content of a view. This includes widgets and camera lists.
            @param init boolean value which indicates initializing or destroying the views content.
            @return <em>true</em> if init/destroy was successful,
                    <em>false</em> otherwise. */
        virtual bool InitContent(bool init) = 0;

        /** Loads or unloads the content of a view.
            @param load boolean value which indicates loading or unloading the views content.
            @param forceUpload boolean value which indicates to force a load of the render target to ensure the upload of the content. If set to false the upload is allowed to fail.
            @return <em>true</em> if loading/unloading was successful,
                    <em>false</em> otherwise. */
        virtual bool LoadContent(bool load, bool forceUpload) = 0;

        /** Indicates if scene content, including widget/cameras are initialized or not.
            @return <em>true</em> if content initialized,
                    <em>false</em> otherwise. */
        virtual bool IsContentInitialized() const = 0;

        /** Indicates if scene content is uploaded or not.
            @return <em>true</em> if scene content is uploaded to nvram,
                    <em>false</em> otherwise. */
        virtual bool IsContentLoaded() const = 0;

        /** Activates or deactivates the view for message receiving.
            @param activate boolean value which activates or deactivates the view. */
        virtual void Activate(bool activate) = 0;

        /** Indicates if view is activated for message receiving.
            @return <em>true</em> true if activated,
                    <em>false</em> otherwise. */
        virtual bool IsActive() const = 0;

        /// Delegates Update method to all childs, regardless of their activation or rendering state.
        virtual void Update(RenderHint * renderHint) = 0;

        /** Retrieves a view with a certain ViewId.
            @param viewId the identifier to be found.
            @return <em>View</em> if view was found,
                    <em>0</em> otherwise. */
        virtual View * FindView(const ViewId & viewId) = 0;

        /** Calls the Visit method of the visitor object.
            @param visitor the visitor object to be visited. */
        virtual void Accept(ViewVisitor & visitor) = 0;

        ///
        virtual ViewScene2D* ToViewScene2D()
        {
            return 0;
        }

        ///
        virtual ViewScene3D* ToViewScene3D()
        {
            return 0;
        }

        ///
        virtual ViewContainer* ToViewContainer()
        {
            return 0;
        }

        ///
        virtual bool Is2D()
        {
            return 0 != ToViewScene2D();
        }

        ///
        virtual bool Is3D()
        {
            return 0 != ToViewScene3D();
        }

        ///
        virtual bool IsContainer()
        {
            return 0 != ToViewContainer();
        }

        /** Invalidates a view. This will cause rendering of its cameras until render counter reached zero. */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual void Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        /** Invalidates a view. This will cause rendering of its cameras until render counter reached zero. */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual void Invalidate(UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>()) = 0;

        /** Invalidates all cameras which are rendering into the given RenderTarget. */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual void Invalidate(Candera::RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>()) = 0;

        /** Is called when a message shall be distributed. Messages are only processed if the view is in state active.
            First the optional attached ViewController will receive the message.
            If it has not consumed the message it will be distributed to its children,
            either child views or in case of ViewScenes to the widgets.
            @note The return parameter defines that the message shall be routed to the succeeding message
                  receiving entities of the dispatching path (<em>true</em>) or if the message is fully
                  consumed and any further dispatching of that message shall be stopped (<em>false</em>).
            @param msg Message object to be processed.
            @return <em>true</em> if the message is consumed
                    <em>false</em> if the message shall be sent to following message receiving entities. */
        virtual bool OnMessage(const Message & msg);

        /** Distributes message to the views childs, either other views or widgets.
            @param msg Message object to be processed.
            @return <em>true</em> if the message is consumed
                    <em>false</em> otherwise. */
        virtual bool DistributeMessage(const Message & msg) = 0;

        /** Is called after a transition has been started. OnTransitionStarted is delegated also to the child views,
            of the firstView and secondView.
            @param msg The event which contains a pointer to the first and second view. */
        virtual void OnTransitionStarted(const TransitionStartedEvent & msg) = 0;

        /** Is called after a transition has been finished. OnTransitionFinished is delegated also to the child views,
            of the firstView and secondView.
            @param msg The event which contains a pointer to the first and second view and if the transition was reset. */
        virtual void OnTransitionFinished(const TransitionFinishedEvent & msg) = 0;

        /** Is called for detecting the touched widget when a TouchMsg occurs.
            @param info The touch information.
            @return <em>FrameworkWidget</em> if a widget was found
                    <em>0</em> otherwise. */
        virtual FrameworkWidget * GetTouchedWidget(const TouchInfo & info) = 0;

        /// IViewHandler access method.
        IViewHandler * GetViewHandler()
        {
            FEATSTD_DEBUG_ASSERT(0 != mViewHandler);
            return mViewHandler;
        }

        /// Implements the clear method.
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual void Clear(bool onlyCameraViewPorts = false, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>()) = 0;


        /// Sets the parent view.
        void SetParentView(View * view)
        {
            mParentView = view;
        }

        /// Returns the parent view.
        View * GetParentView() const
        {
            return mParentView;
        }

        /// Returns the ViewController, pointer might by 0.
        ViewController * GetViewController()
        {
            return mViewController; 
        }

        /** Returns the ScopeMask of the view needed for routing purposes.
         * @return the actual scope mask of the view, set using SetScopeMask.
         */
        Message::ScopeMask GetScopeMask() const
        {
            return mScopeMask;
        }

        /** Returns the effective ScopeMask of the view, composed from the mask of the view and the scope mask of its children. This is needed for routing purposes.
         * @return the effective scope mask of the view.
         */
        Message::ScopeMask GetEffectiveScopeMask() const
        {
            return mEffectiveScopeMask;
        }

        /** Sets the actual scope mask of the view.
         * @param scopeMask the scope mask of the view
         */
        virtual void SetScopeMask(Message::ScopeMask scopeMask);

        /// Is called by the childs to change the views own ScopeMask needed for routing purposes.
        virtual void COURIER_DEPRECATED_MSG("DEPRECATED! - The effective scope mask of the view and its children is composed internally. Please use GetEffectiveScopeMask to get it.") OnChangeScopeMask(Message::ScopeMask scopeMask);

        /** Returns if the view object was dynamically allocated by the ViewHandler.
            @return <em>true</em> if the message was allocated by the ViewHandler,
                    <em>false</em> if allocated outside. */
        bool IsViewHandlerManaged() const
        {
            return mViewHandlerManaged;
        }

        /** Add render targets used by the view (based on views cameras).
            @param vector Vector to be filled.
            @param clear flag which controls clearing the vector before it will be filled.
            @return <em>true</em> if RenderTargets has been added correctly,
                    <em>false</em> otherwise. */
        virtual bool GetRenderTargetPtrVector(RTPtrVector & vector, bool clear) = 0;

        ///
        virtual bool GetRenderTargetPtrVector(RTPtrVector & vector) = 0;

        /** Add render targets to the vector and looks if the renderTarget is already inside the vector.
            @param vector Vector to be filled.
            @param renderTarget which shall be added.
            @return <em>true</em> if RenderTargets has been added correctly or is already inside vector,
                    <em>false</em> otherwise. */
        static bool AddRenderTarget(RTPtrVector & vector, Candera::RenderTarget * renderTarget);

        /**
         * Wakes up all render components (NOTE: the wakeup message is only posted once per frame. This method is only allowed to be called within the view thread!).
         */
        static void WakeUpAllRenderComponents();

        ///
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual void InvalidateOffscreenHelper(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>())
        {
            FEATSTD_UNUSED(dirtyArea);
        }

        /** */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual bool AddInvalidationDependency(const ViewId& viewId, const Char* cameraName = 0, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
        /** */
        FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
        virtual bool RemoveInvalidationDependency(const ViewId& viewId, const Char* cameraName = 0);
        /** */
        const InvalidationDependency* GetFirstInvalidationDependency() const
        {
            return mFirstInvalidationDependency;
        }

        /**
         * Returns true if the view has been already invalidated with the dirty area (NOTE: a different dirty area will reset the flag).
         */
        bool IsAlreadyInvalidated(const ItemId& camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea);

        // used for solving Lint problem of pointer clearing of members inside generated source file of events (TransitionEvents messages).
        /// Wrapper around View pointer, use either cast operators or GetView method to get the View pointer.
        class Ptr {
            public:
                Ptr() :
                    mPtr(0)
                {
                }
                explicit Ptr(View * view) :
                    mPtr(view)
                {
                }
                Ptr(const Ptr & p) :
                    mPtr(p.mPtr)
                {
                }
                ~Ptr()
                {
                    mPtr = 0;
                }
                Ptr & operator = (const Ptr & p)
                {
                    FEATSTD_LINT_SYMBOL(1555, Courier::View::Ptr::mPtr, It is direct copy of pointer but it points to const.);
                    mPtr = p.mPtr;
                    return *this;
                }
                operator const View *() const
                {
                    return mPtr;
                }
                operator View *() const
                {
                    return mPtr;
                }
                operator View *()
                {
                    return mPtr;
                }
                const View * GetView() const
                {
                    return mPtr;
                }
                View * GetView()
                {
                    return mPtr;
                }
            private:
                View * mPtr;
        };

    protected:
        friend class ViewController;
        friend class ViewHandler;
        friend class FrameworkWidget;

        virtual void InvalidateImpl(const ItemId &camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        static FeatStd::UInt GetCurrentInvalidationId();

        /// Reseting the ViewController
        void ReleaseViewController();

        /// Creates the effective scope mask of the view, by applying a logical OR on the actual scope mask of the view and the scope mask of its children.
        virtual void OnScopeMaskChanged();

        virtual void OnInvalidationIdOverflow()
        {
            ResetInvalidationFlag();
        }

        void ResetInvalidationFlag()
        {
            mProcessingInvalidationDependencyPerCamera.Clear();
        }

        /// The ViewController implementing the control logic of the view.
        ViewController * mViewController;

        /// The IViewHandler use for accessing and controlling the widgets.
        IViewHandler * mViewHandler;

        /// The head of the invalidation dependency list
        InvalidationDependency* mFirstInvalidationDependency;
        bool mProcessingInvalidationDependencyFlag;
        Candera::Internal::Map<ItemId, FeatStd::UInt> mProcessingInvalidationDependencyPerCamera;
        FeatStd::Optional<Candera::Rectangle> mProcessingInvalidationDependencyDirtyArea;

        /// The effective ScopeMask used for routing purposes, constructed from the actual scope mask of the view and the scope mask of its children.
        Message::ScopeMask mEffectiveScopeMask;

    private:
        friend class ViewFacade;
        friend class ViewContainer;
        friend class Gdu;
        COURIER_UNIT_TEST_TESTCASE_FRIEND(ViewScene3DTest, InvalidateTest);
        COURIER_UNIT_TEST_TESTCASE_FRIEND(ViewScene2DTest, InvalidateTest);

        static bool IncrementInvalidationId();

        /// The parent view.
        View * mParentView;

        /// The actual scope mask of the view, set using SetScopeMask
        Message::ScopeMask mScopeMask;

        /// Defines if the View object is memory managed by the ViewHandler
        const bool mViewHandlerManaged;

        /// 
        bool mContentInitialized;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(View);
};

///
typedef Internal::LinkedList<View, &View::ViewListNode> ViewLinkedList;

///
typedef Vector< View *> ViewPtrVector;

/// @}
}
#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {

template<> UInt32 StringBufferAppender< ::Courier::View::Ptr >::Append(StringBuffer& stringBuffer, ::Courier::View::Ptr const & object);

template<> UInt32 StringBufferAppender< ::Courier::View >::Append(StringBuffer& stringBuffer, ::Courier::View const & object);
}
#endif

FEATSTD_LINT_NEXT_EXPRESSION(1960, "this is a correct global typedef to guarantee backwards compatibility")
typedef Courier::RTPtrVector RenderTargetPtrVector;

#endif
