//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "View.h"

#include "IViewHandler.h"
#include "ViewController.h"
#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Diagnostics/Log.h>
#include <Courier/Platform/Memory.h>
#if defined(CANDERA_2D_ENABLED)
#include "ViewScene2D.h"
#endif
#if defined(CANDERA_3D_ENABLED)
#include "ViewScene3D.h"
#endif
#include "Renderer.h"
#include "RenderComponent.h"
#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Event/EventSource.h>

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

FEATSTD_RTTI_DEFINITION(ViewActivationEvent, FeatStd::Event)
FEATSTD_RTTI_DEFINITION(RenderComponentUpdateEvent, FeatStd::Event)

using namespace Candera;

FeatStd::EventSource& View::GetEventSource()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::EventSource, s_viewEventSource);
    return s_viewEventSource;
}

static FeatStd::EventSource& s_forceInitViewGetEventSource = View::GetEventSource();

// ------------------------------------------------------------------------
View::View(bool managed) :
    mViewController(0),
    mViewHandler(0),
    mFirstInvalidationDependency(0),
    mProcessingInvalidationDependencyFlag(false),
    mEffectiveScopeMask(0x00000000),
    mParentView(0),
    mScopeMask(0x00000000),
    mViewHandlerManaged(managed),
    mContentInitialized(false)
{
}

// ------------------------------------------------------------------------
View::~View()
{
    mViewHandler = 0;
    mParentView = 0;
    mViewController = 0;
    mScopeMask = 0;
    mEffectiveScopeMask = 0;

    // Clean up InvalidationDependency dynamic list
    while (0 != mFirstInvalidationDependency) {
        InvalidationDependency* next = mFirstInvalidationDependency->GetNext();
        COURIER_DELETE(mFirstInvalidationDependency);
        mFirstInvalidationDependency = next;
    }
}

// ------------------------------------------------------------------------
bool View::Init(IViewHandler * viewHandler, const Char * viewName)
{
    return Init(viewHandler,viewName,0);
}

// ------------------------------------------------------------------------
bool View::Init(IViewHandler * viewHandler, const Char * viewName, ViewController * viewController)
{
    // viewName is set in base class
    FEATSTD_UNUSED(viewName);
    // ViewController is optional, so it is not checked
    FEATSTD_DEBUG_ASSERT(viewHandler!=0);
    mViewHandler = viewHandler;
    mViewController = viewController;
    if(mViewController!=0) {
        mViewController->SetView(this);
        OnScopeMaskChanged();
    }
    return (0 != mViewHandler);
}

// ------------------------------------------------------------------------
bool View::AddView(View * view)
{
    FEATSTD_UNUSED(view);
    return false;
}

// ------------------------------------------------------------------------
FEATSTD_LINT_CURRENT_SCOPE(1762, "No we don't want to make it const because customer might overwrite and maintain content")
bool View::RemoveView(View * view)
{
    FEATSTD_UNUSED(view);
    return false;
}

// ------------------------------------------------------------------------
const ViewId & View::GetPathId() const
{
    if(GetParentView()!=0) {
        return GetParentView()->GetId();
    }
    static ViewId emptyId("");
    return emptyId;
}

// ------------------------------------------------------------------------
void View::Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    // ItemId() for camera represents all cameras
    InvalidateImpl(ItemId(), dirtyArea);
}

// ------------------------------------------------------------------------
bool View::OnMessage(const Message & msg)
{
    bool lMsgConsumed = false;

    // only route if view is active
    if (IsActive()) {
        // Route message to view controller instance
        if(0!=mViewController) {
            if(msg.InScopeOf(mViewController->GetScopeMask())) {
                lMsgConsumed = mViewController->OnMessage(msg);
            }
        }
        // Route message if not yet consumed
        if (!lMsgConsumed) {
            lMsgConsumed = DistributeMessage(msg);
        }
    }

    // Do not check consumed flag with shutdown message
    if (Equals<ShutdownMsg>(msg)) {
        ///@todo Do whatever there is to do to cleanup view
        lMsgConsumed = true;
    }
    return lMsgConsumed;
}

// ------------------------------------------------------------------------
void View::OnChangeScopeMask(Message::ScopeMask scopeMask)
{
    COURIER_UNUSED(scopeMask);
    OnScopeMaskChanged();
}

// ------------------------------------------------------------------------
bool View::AddRenderTarget(RTPtrVector & vector, RenderTarget * renderTarget)
{
    for (FeatStd::SizeType i = 0; i < vector.Size(); ++i) {
        if(vector[i]==renderTarget) {
            return true;
        }
    }
    return vector.Add(renderTarget);
}

// ------------------------------------------------------------------------
bool View::AddInvalidationDependency(const ViewId& viewId, const Char* cameraName, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    ItemId camerId= (0 == cameraName) ? ItemId() : ItemId(cameraName);
    if (0 == mFirstInvalidationDependency) {
        mFirstInvalidationDependency = COURIER_NEW(InvalidationDependency)(viewId, camerId, dirtyArea);
    }
    else {
        InvalidationDependency* current = mFirstInvalidationDependency;
        while (0 != current->GetNext()) {
            if (current->Equals(viewId, camerId)) {
                return false;
            }
            current = current->GetNext();
        }
        if (current->Equals(viewId, camerId)) {
            return false;
        }
        current->SetNext(COURIER_NEW(InvalidationDependency)(viewId, camerId, dirtyArea));
    }
    return true;
}

// ------------------------------------------------------------------------
bool View::RemoveInvalidationDependency(const ViewId& viewId, const Char* cameraName)
{
    ItemId camerId = (0 == cameraName) ? ItemId() : ItemId(cameraName);
    if (0 != mFirstInvalidationDependency) {
        InvalidationDependency* current = mFirstInvalidationDependency;
        InvalidationDependency* next = current->GetNext();
        if (mFirstInvalidationDependency->Equals(viewId, camerId)) {
            mFirstInvalidationDependency->SetNext(0);
            COURIER_DELETE(mFirstInvalidationDependency);
            mFirstInvalidationDependency = next;
            return true;
        }
        while (0 != next ) {
            if (next->Equals(viewId, camerId)) {
                current->SetNext(next->GetNext());
                next->SetNext(0);
                COURIER_DELETE(next);
                return true;
            }
            current = next;
            next = next->GetNext();
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void View::EnableRenderingScene(bool enable)
{
    COURIER_UNUSED(enable);
    COURIER_LOG_ERROR("Not applicable for '%s'.",GetId().CStr());
}

// ------------------------------------------------------------------------
bool View::InitContent(bool init)
{
    mContentInitialized = init;
    return true;
}

// ------------------------------------------------------------------------
bool View::LoadContent(bool load, bool /*forceLoad*/)
{
    if (load && (!mContentInitialized)) {
        return InitContent(true);
    }
    return true;
}

// ------------------------------------------------------------------------
void View::ReleaseViewController()
{
    if (0 != mViewController) {
        mViewController->ReleaseView();
        mViewController = 0;
    }
}

// ------------------------------------------------------------------------
void View::SetScopeMask(Message::ScopeMask scopeMask)
{
    if (mScopeMask != scopeMask) {
        mScopeMask = scopeMask;
        OnScopeMaskChanged();
    }
}

// ------------------------------------------------------------------------
void View::OnScopeMaskChanged()
{
    if (0 != mViewController) {
        mEffectiveScopeMask |= mViewController->GetScopeMask();
    }
    if (0 != GetParentView()) {
        GetParentView()->OnScopeMaskChanged();
    }
}

// ------------------------------------------------------------------------
static FeatStd::UInt& InvalidationId()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::UInt, s_invalidationId, 1);
    return s_invalidationId;
}

// ------------------------------------------------------------------------
static FeatStd::UInt& WakeupId()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::UInt, s_wakeupId, 0);
    return s_wakeupId;
}

// ------------------------------------------------------------------------
bool View::IsAlreadyInvalidated(const ItemId& camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    UInt *invalidationIdForCamera = mProcessingInvalidationDependencyPerCamera.Find(camera);
    UInt *invalidationIdForAllCameras = mProcessingInvalidationDependencyPerCamera.Find(ItemId());
    if ( (invalidationIdForCamera == 0 || *invalidationIdForCamera != GetCurrentInvalidationId()) &&
        (invalidationIdForAllCameras == 0 || *invalidationIdForAllCameras != GetCurrentInvalidationId()) )
    {
        return false;
    }
    if (mProcessingInvalidationDependencyDirtyArea != dirtyArea) {
        return false;
    }
    if ((dirtyArea.IsSet()) && (*mProcessingInvalidationDependencyDirtyArea != *dirtyArea)) {
        return false;
    }
    return true;
}

void View::InvalidateImpl(const ItemId& camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if ((0 != GetViewHandler()) && (0 != GetViewHandler()->GetRenderer()) && (!mProcessingInvalidationDependencyFlag) && (!IsAlreadyInvalidated(camera, dirtyArea))) {
        IViewHandler* viewHandler = GetViewHandler();
        RenderJobStrategy& renderJobStrategy = viewHandler->GetRenderer()->GetRenderJobStrategy();
        mProcessingInvalidationDependencyDirtyArea = dirtyArea;
        mProcessingInvalidationDependencyFlag = true;
        mProcessingInvalidationDependencyPerCamera.Update(camera, GetCurrentInvalidationId());
        InvalidationDependency* current = mFirstInvalidationDependency;
        while (0 != current) {
            View* view = viewHandler->FindView(current->GetViewId());
            if (0 != view) {
                bool ok = true;
                FeatStd::Optional<Candera::Rectangle> transformedDirtyArea;
                if (current->GetCameraId() == ItemId()) {
                    RenderJobStrategy::InvalidationDependencyData invalidationDependencyData(*view, *current);
                    renderJobStrategy.TransformDirtyAreaToInvalidationDependency(dirtyArea, transformedDirtyArea, *this, invalidationDependencyData);
                    view->Invalidate(transformedDirtyArea);
                }
#if defined(CANDERA_2D_ENABLED)
                else if (view->Is2D()) {
                    ViewScene2D* view2D = static_cast<ViewScene2D*>(view);
                    Camera2D* camera2D = view2D->FindCamera2D(current->GetCameraId());
                    if (0 != camera2D) {
                        RenderJobStrategy::InvalidationDependencyData invalidationDependencyData(*view, *current);
                        renderJobStrategy.TransformDirtyAreaToInvalidationDependency(dirtyArea, transformedDirtyArea, *this, invalidationDependencyData);
                        view2D->Invalidate(camera2D, transformedDirtyArea);
                    }
                    else {
                        ok = false;
                    }
                }
#endif
#if defined(CANDERA_3D_ENABLED)
                else if (view->Is3D()) {
                    ViewScene3D* view3D = static_cast<ViewScene3D*>(view);
                    Camera* camera3D = view3D->FindCamera3D(current->GetCameraId());
                    if (0 != camera3D) {
                        RenderJobStrategy::InvalidationDependencyData invalidationDependencyData(*view, *current);
                        renderJobStrategy.TransformDirtyAreaToInvalidationDependency(dirtyArea, transformedDirtyArea, *this, invalidationDependencyData);
                        view3D->Invalidate(camera3D, transformedDirtyArea);
                    }
                    else {
                        ok = false;
                    }
                }
#endif
                else {
                    ok = false;
                }
                if (!ok) {
                    COURIER_LOG_ERROR("Invalidation camera '%s' missing for invalidation dependency view '%s'.", current->GetCameraId().CStr(), current->GetViewId().CStr());
                }
            }
            current = current->GetNext();
        }
        mProcessingInvalidationDependencyFlag = false;
    }
}

// ------------------------------------------------------------------------
FeatStd::UInt View::GetCurrentInvalidationId()
{
    return InvalidationId();
}

// ------------------------------------------------------------------------
bool View::IncrementInvalidationId()
{
    FeatStd::UInt oldInvalidationId = InvalidationId();
    InvalidationId()++;
    if (oldInvalidationId > InvalidationId()) {
        WakeupId() = 0;
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
void View::WakeUpAllRenderComponents()
{
    if (WakeupId() != InvalidationId()) {
        WakeupId() = InvalidationId();
        Renderer::WakeUpAllRenderComponents();
    }
}

}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {

template<> UInt32 StringBufferAppender< ::Courier::View::Ptr >::Append(StringBuffer& stringBuffer, ::Courier::View::Ptr const & object)
{
    const ::Courier::View* view = object.GetView();
    return stringBuffer.AppendObject(view);
}

template<> UInt32 StringBufferAppender< ::Courier::View >::Append(StringBuffer& stringBuffer, ::Courier::View const & object)
{
    UInt32 tcharCount = 0;
    tcharCount += stringBuffer.Append("::Courier::View { Id =");
    tcharCount += stringBuffer.AppendObject(object.GetId());
    tcharCount += stringBuffer.Append(", RenderingEnabled = ");
    tcharCount += stringBuffer.AppendObject(object.IsRenderingEnabled());
    tcharCount += stringBuffer.Append(", ContentInitialized = ");
    tcharCount += stringBuffer.AppendObject(object.IsContentInitialized());
    tcharCount += stringBuffer.Append(", ContentLoaded = ");
    tcharCount += stringBuffer.AppendObject(object.IsContentLoaded());
    tcharCount += stringBuffer.Append(", ContentLoaded = ");
    tcharCount += stringBuffer.AppendObject(object.IsContentLoaded());
    tcharCount += stringBuffer.Append(", Active = ");
    tcharCount += stringBuffer.AppendObject(object.IsActive());
    tcharCount += stringBuffer.Append(", 2D = ");
    tcharCount += stringBuffer.AppendObject(const_cast< ::Courier::View& >(object).Is2D());
    tcharCount += stringBuffer.Append(", 3D = ");
    tcharCount += stringBuffer.AppendObject(const_cast< ::Courier::View& >(object).Is3D());
    tcharCount += stringBuffer.Append(", Container = ");
    tcharCount += stringBuffer.AppendObject(const_cast< ::Courier::View& >(object).IsContainer());
    tcharCount += stringBuffer.Append(" }");
    return tcharCount;
}

}
#endif
