//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TransitionHandler.h"
#include "TransitionFactory.h"
#include "VisualizationMsgs.h"

namespace Courier {
// ------------------------------------------------------------------------
TransitionHandler::TransitionHandler() : mViewHandler(0), mTransitionFactory(0)
{
}

// ------------------------------------------------------------------------
TransitionHandler::~TransitionHandler()
{
    mViewHandler = 0;
    mTransitionFactory = 0;
    mTransitions.Clear();
}

// ------------------------------------------------------------------------
bool TransitionHandler::Init(IViewHandler * viewHandler, TransitionFactory * factory)
{
    FEATSTD_DEBUG_ASSERT(viewHandler!=0);
    FEATSTD_DEBUG_ASSERT(factory!=0);
    mTransitions.Clear();
    mViewHandler = viewHandler;
    mTransitionFactory = factory;
    return (mViewHandler!=0) && (mTransitionFactory!=0);
}

// ------------------------------------------------------------------------
void TransitionHandler::Finalize()
{
    for (FeatStd::SizeType i = 0; i < mTransitions.Size(); ++i) {
        if (0 != mTransitions[i]) {
            if (mTransitions[i]->IsActive()) {
                static_cast<void>(mTransitions[i]->Finish());
            }
            if (0 != mTransitionFactory) {
                mTransitionFactory->Destroy(mTransitions[i]);
            }
            mTransitions[i] = 0;
        }
    }
    mTransitions.Clear();
}

// ------------------------------------------------------------------------
Transition * TransitionHandler::Create(const ItemId & transitionId)
{
    FEATSTD_DEBUG_ASSERT(mTransitionFactory!=0);
    if(mTransitionFactory!=0) {
        bool deleteTransition;
        do {
            deleteTransition = false;
            FeatStd::SizeType idx;
            for (idx = 0; idx < mTransitions.Size(); ++idx) {
                if(! mTransitions[idx]->IsActive()) {
                    mTransitionFactory->Destroy(mTransitions[idx]);
                    deleteTransition = true;
                    break;
                }
            }
            if(deleteTransition) {
                if(mTransitions.Remove(idx)) {
                }
            }
        } while(deleteTransition);

        Transition * newTransition = mTransitionFactory->Create(transitionId);
        if((newTransition!=0) && mTransitions.Add(newTransition)) {
            newTransition->SetTransitionHandler(this);
        }
        return newTransition;
    }
    return 0;
}

// ------------------------------------------------------------------------
Transition * TransitionHandler::Get(const ItemId & transitionId, const View * firstView, const View * secondView)
{
    for (FeatStd::SizeType idx = 0; idx<mTransitions.Size(); ++idx) {
        Transition * transition = mTransitions[idx];
        if((transitionId==transition->GetId()) && (firstView==transition->GetFirstView()) && (secondView==transition->GetSecondView())) {
            return transition;
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
bool TransitionHandler::OnTransitionFinished(const Transition * transition, const View * firstView, const View * secondView, bool reset) const
{
    FEATSTD_DEBUG_ASSERT(transition!=0);
    if(transition!=0) {
        static const ViewId emptyViewId;
        const ViewId & firstViewId = ((firstView!=0) ? firstView->GetId() : emptyViewId);
        const ViewId & secondViewId = ((secondView!=0) ? secondView->GetId() : emptyViewId);
        Message * postMsg = (COURIER_MESSAGE_NEW(TransitionIndMsg)(transition->GetId(),firstViewId,secondViewId,reset));
        return (postMsg!=0) && postMsg->Post();
    }
    return false;
}
}

