//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_Transition_h)
#define Courier_Visualization_Transition_h

#include <Courier/Util/LinkedList.h>
#include <Courier/Visualization/ItemId.h>
#include <Courier/Visualization/IViewHandler.h>

namespace Courier { 
    class TransitionHandler;
    class View;
}

namespace Courier { 

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The Transition is the base class for concrete transition implementations. 
    When a transition will be started, first of all Start method will be called which itself will call OnExecute
    which has to be overwritten with specific functionality. The same is valid when Reset and Finish is called.
    In this case overwritten OnReset and OnFinish methods are called. 
*/
class Transition
{
    public:
        friend class TransitionHandler;

        ///
        enum TransitionError {
            /// The last method call on a transition was successful.
            TransitionSuccess,
            /// The creation of a specific transition failed.
            CreateTransitionError,
            /// Failed to lookup a specific transition.
            MissingTransitionError,
            /// The Start method has been called on an active transition.
            StartOnActiveTransitionError,
            /// The Reset method has been called on an inactive transition.
            ResetOnInactiveTransitionError,
            /// The Finish method has been called on an inactive transition.
            FinishOnInactiveTransitionError,
            /// Another error occurred.
            GeneralTransitionError
        };

        ///
        Transition();

        /// 
        virtual ~Transition();

        /** Returns the identification of the transition. This id must be a unique user specific transition implementation name.
            The transition id parameter of TransitionReqMsg is used for identifying the specific transition implementation
            @return <em>ItemId</em> the identifier. */
        virtual const ItemId & GetId() const = 0;

        /** Starts the transition using the two views and an optional payload string.
            @param firstView the origin view.
            @param secondView the destination view, which might be also set to 0, in case only one view shall be processed (e.g. fade out)
            @param optionalPayload a string which may contain specific control strings.
            @return <em>true</em> if starting was successful and the transition is not in use,
                    <em>false</em> otherwise. */
        bool Start(View * firstView, View * secondView, const Payload & optionalPayload);

        /** Resets the transition.
            @return <em>true</em> if reset was successful and a transition was started and is not finished yet.
                    <em>false</em> otherwise. */
        bool Reset();

        /** Finishes the transition.
            @return <em>true</em> if finishing was successful and a transition was started and is not finished yet.
                    <em>false</em> otherwise. */
        bool Finish();

        /** Returns the current state of the transition.
            @return <em>true</em> if the transition is in progress.
                    <em>false</em> otherwise. */
        bool IsActive();

        /// Access to the parent transition handler which hold this transition. 
        TransitionHandler * GetTransitionHandler();

        /** This method has to be called by the transition in all use cases.
            If the transition was finished automatically, if the transition was reset and if it was stopped. 
            It will call the OnTransitionFinished of the TransitionHandler.
            @return <em>true</em> if finishing was successful,
                    <em>false</em> otherwise. */
        bool OnTransitionFinished();

        /// Returns the first view.
        View * GetFirstView() { return mFirstView; }

        /// Returns the second view.
        View * GetSecondView() { return mSecondView; }

        /** */
        static void SetLastTransitionError(Transition::TransitionError lastTransitionError) { mLastTransitionError = lastTransitionError; }

        /** */
        static Transition::TransitionError GetLastTransitionError() { return mLastTransitionError; }

    protected:
        /** Has to be overwritten by the specific implementation.
            @param firstView the origin view.
            @param secondView the destination view, which might be also set to 0, in case only one view shall be processed (e.g. fade out)
            @param optionalPayload a string which may contain specific control strings.
            @return <em>true</em> if starting was successful and the transition is not in use,
                    <em>false</em> otherwise. */
        virtual bool OnExecute(View * firstView, View * secondView, const Payload & optionalPayload) = 0;

        /** Has to be overwritten by the specific implementation.
            @return <em>true</em> if reset was successful and a transition was started and is not finished yet.
                    <em>false</em> otherwise. */
        virtual bool OnReset() = 0;

        /** Has to be overwritten by the specific implementation.
            @return <em>true</em> if finishing was successful and a transition was started and is not finished yet.
                    <em>false</em> otherwise. */
        virtual bool OnFinish() = 0;

        /// Sets the parent transition handler.
        void SetTransitionHandler(TransitionHandler * handler);

        /// The transition handler
        TransitionHandler * mTransitionHandler;

        /// The first view (origin view in case we a second one is in use)
        View * mFirstView;
    
        /// The second view (destination view in case it is set)
        View * mSecondView;

        /// indicate if transition is active
        bool mIsActive;

        /// used for setting the reset flag
        bool mWasReset;

    private:
        ///
        static Transition::TransitionError mLastTransitionError;
};

/// @}
}

#endif
