//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_RenderJob_h)
#define Courier_Visualization_RenderJob_h

#include <Candera/Candera.h>
#include <Courier/Visualization/View.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>

namespace Courier { 
    class Renderer;
    class RenderJobs;
    class Gdu;
    class ViewScene;
}

namespace Courier { 
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** RenderJob represents an invalidated camera which shall be rendered n times.
*/
class RenderJob {
    public:
        /**
         * The RenderState will reflect the current state of the render job.
         */
        typedef enum {
            /**
             * The render job will be in the RenderPassFailed state if a render call fails. In this state it is undefined if the content is complete or incomplete.
             */
            RenderPassFailed,
            /**
            * The render job will be in the RenderPassIncomplete state if only parts of the render job have been rendered (due to a custom Candera CameraRenderStrategy).
            */
            RenderPassIncomplete,
            /**
            * The render job will be in the RenderPassIncomplete state if it has been completly rendered.
            */
            RenderPassCompleted
        } RenderState;

        /// Create a new empty render job.
        RenderJob() :
            mView(0),
            mGdu(0),
            mFrameId(0),
            mCurrentDirtyArea(0),
            mRenderState(RenderPassIncomplete),
            mRenderCounter(0),
            mIs2D(false),
            mIsClear(false)
        {
        }
#ifdef CANDERA_2D_ENABLED
        /// Create a new render job for a specific camera.
        RenderJob(ViewScene* view, Candera::Camera2D* camera, Gdu * gdu, bool is2D, bool isClear, UInt8 renderCounter) :
            mView(view),
            mCamera(camera),
            mGdu(gdu),
            mFrameId(0),
            mCurrentDirtyArea(0),
            mRenderState(RenderPassIncomplete),
            mRenderCounter(0),
            mIs2D(is2D),
            mIsClear(isClear)
        {
            SetRenderCounter(renderCounter);
        }
#endif

#ifdef CANDERA_3D_ENABLED
        /// Create a new render job for a specific camera.
        RenderJob(ViewScene* view, Candera::Camera* camera, Gdu * gdu, bool is2D, bool isClear, UInt8 renderCounter) :
            mView(view),
            mCamera(camera),
            mGdu(gdu),
            mFrameId(0),
            mCurrentDirtyArea(0),
            mRenderState(RenderPassIncomplete),
            mRenderCounter(0),
            mIs2D(is2D),
            mIsClear(isClear)
        {
            SetRenderCounter(renderCounter);
        }
#endif
        /// Copy a render job.
        RenderJob(const RenderJob& renderJob) :
            mDirtyAreas(renderJob.mDirtyAreas),
            mView(renderJob.mView),
            FEATSTD_LINT_NEXT_EXPRESSION(1554, "Direct pointer copy of member, thats what we want")
            mCamera(renderJob.mCamera),
            FEATSTD_LINT_NEXT_EXPRESSION(1554, "Direct pointer copy of member, thats what we want")
            mGdu(renderJob.mGdu),
            mFrameId(renderJob.mFrameId),
            mCurrentDirtyArea(renderJob.mCurrentDirtyArea),
            mRenderState(renderJob.mRenderState),
            mRenderCounter(renderJob.mRenderCounter),
            mIs2D(renderJob.mIs2D),
            mIsClear(renderJob.mIsClear)
        {
        }
        ///
        ~RenderJob()
        {
            Clear();
        }
        /// Copy a render job.
        RenderJob & operator = (const RenderJob& renderJob);
        ///
        void Clear()
        {
            mDirtyAreas.Clear();
            mView = 0;
            mCamera = ::Candera::AbstractNodePointer();
            mGdu = 0;
            mRenderCounter = 0;
            mCurrentDirtyArea = 0;
        }
        /// Courier internal clear cameras will be disabled.
        void DisableIfClearCamera() const;
        ///
        ViewScene* GetView() const
        {
            return mView;
        }
        ///
        const ::Candera::AbstractNodePointer& GetCamera() const
        {
            return mCamera;
        }
        ///
        UInt8 GetRenderCounter() const
        {
            return mRenderCounter;
        }
        ///
        void SetRenderCounter(UInt8 renderCounter)
        {
            mRenderCounter = renderCounter;
            if (mDirtyAreas.Size() < renderCounter) {
                (void)mDirtyAreas.Reserve(renderCounter);
                FeatStd::Internal::Vector<FeatStd::Optional<Candera::Rectangle> > empty;
                while (mDirtyAreas.Size() < renderCounter) {
                    (void)mDirtyAreas.Add(empty);
                }
            }
        }
        ///
        bool DecRenderCounter();
        /// Validate the Candera Layout.
        void Validate() const;
        /// Render the render job and return its current render state.
        RenderState Render(const Vector<FeatStd::Optional<Candera::Rectangle> >& dirtyAreas);
        ///
        RenderState GetRenderState() const
        {
            return mRenderState;
        }
        ///
        Gdu* GetGdu() const
        {
            return mGdu;
        }
        ///
        Int32 GetSequenceNumber() const;
        ///
        bool IsClear() const
        {
            return mIsClear;
        }
        ///
        bool Is2D() const
        {
#ifdef CANDERA_2D_ENABLED
            return 0 != mCamera.ToNode2D();
#else
            return false;
#endif
        }
        ///
        void SetFrameId(UInt32 frameId)
        {
            mFrameId = frameId;
        }
        ///
        UInt32 GetFrameId() const
        {
            return mFrameId;
        }
        ///
        void ResetRenderState()
        {
            mRenderState = RenderPassIncomplete;
        }
        ///
        const FeatStd::Char* GetSceneName() const;
        ///
        const FeatStd::Char* GetCameraName() const;

    private:
        friend class RenderJobs;

        void SetView(ViewScene* view)
        {
            mView = view;
        }

        void CloneStateFrom(const RenderJob& renderJob)
        {
            mFrameId = renderJob.mFrameId;
            mRenderState = renderJob.mRenderState;
        }

        Vector<Vector<FeatStd::Optional<Candera::Rectangle> > > mDirtyAreas;
        ViewScene* mView;
        ::Candera::AbstractNodePointer mCamera;
        Gdu* mGdu;
        UInt32 mFrameId;
        UInt32 mCurrentDirtyArea;
        RenderState mRenderState;
        UInt8  mRenderCounter;
        bool   mIs2D : 1;
        bool   mIsClear : 1;
};
/// @}

///
typedef Courier::Vector< RenderJob > RenderJobVector;

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** RenderJobs represents a vector of RenderJob objects which shall be rendered.
*/
class RenderJobs 
{
    public:
        ///
        RenderJobs() : mActiveVector(&mRenderJob1Vector), mInactiveVector(&mRenderJob2Vector) {}
        ///
        bool AddJob(const RenderJob & newjob, Vector<FeatStd::Optional<Candera::Rectangle> >*& dirtyAreas);
        ///
        bool Render(RenderHint * renderHint, Renderer& renderer);
        ///
        void DeleteJobs(const Candera::RenderTarget * renderTarget);
        ///
        void DeleteJobs(const Candera::CanderaObject* camera);
        ///
        void RemoveViewRelations(const View* view);
        ///
        FeatStd::SizeType Size() const { return (0 != mActiveVector) ? mActiveVector->Size() : 0; }
        ///
        bool HasJob(bool is2D) const;

    private:
        ///
        void SwapVectors();
        ///
        Vector<FeatStd::Optional<Candera::Rectangle> > mDirtyAreas;
        ///
        RenderJobVector * mActiveVector;
        ///
        RenderJobVector * mInactiveVector;
        ///
        RenderJobVector mRenderJob1Vector;
        ///
        RenderJobVector mRenderJob2Vector;
};
/// @}

}

#endif
