//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_RenderConfiguration_h)
#define Courier_Visualization_RenderConfiguration_h

#include <Courier/Visualization/View.h>

// ----------------------------------------------------------------------------
namespace Courier { 
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The RenderConfiguration object defines how rendering will be used.
    The default behaviour (without setting a new RenderConfiguration to the Renderer) is the following:
    mLoadAllRenderTargetsAtStartup(false), mUseInvalidation(true), mUseKickDisplayUpdate(true), 
    mUseCourierSwapping(true), mUseGlobalCameraSequenceNumber(true), mUse2DBefore3D(true), 
    mUseWakeUpRenderMechanism(false), mEnableValidatingLayout(false).
*/
class RenderConfiguration
{
    public:
        ///
        RenderConfiguration(bool loadAllRenderTargetsAtStartup = false, 
                            bool useInvalidation = true, 
                            bool useKickDisplayUpdate = true, 
                            bool useCourierSwapping = true, 
                            bool useGlobalCameraSequenceNumber = true,
                            bool useWakeUpRenderMechanism = false,
                            bool clearRenderTargetOnUpload = false,
                            bool useOffscreenInvalidation = true) 
                : mLoadAllRenderTargetsAtStartup(loadAllRenderTargetsAtStartup), mUseInvalidation(useInvalidation),
                  mUseKickDisplayUpdate(useKickDisplayUpdate), mUseCourierSwapping(useCourierSwapping), 
                  mUseGlobalCameraSequenceNumber(useGlobalCameraSequenceNumber),
                  mUse2DBefore3D(true), mUseWakeUpRenderMechanism(useWakeUpRenderMechanism),
                  mEnableSingleSwapping(false), mShall2DSwapIfSingleSwapping(false),
                  mEnableValidatingLayout(true), mClearRenderTargetOnUpload(clearRenderTargetOnUpload),
                  mUseOffscreenInvalidation(useOffscreenInvalidation) {}

        /// Copy constructor.
        RenderConfiguration(const RenderConfiguration & rc) 
                : mLoadAllRenderTargetsAtStartup(rc.mLoadAllRenderTargetsAtStartup), mUseInvalidation(rc.mUseInvalidation),
                  mUseKickDisplayUpdate(rc.mUseKickDisplayUpdate), mUseCourierSwapping(rc.mUseCourierSwapping),
                  mUseGlobalCameraSequenceNumber(rc.mUseGlobalCameraSequenceNumber),
                  mUse2DBefore3D(rc.mUse2DBefore3D), mUseWakeUpRenderMechanism(rc.mUseWakeUpRenderMechanism),
                  mEnableSingleSwapping(rc.mEnableSingleSwapping), mShall2DSwapIfSingleSwapping(rc.mShall2DSwapIfSingleSwapping), 
                  mEnableValidatingLayout(rc.mEnableValidatingLayout), mClearRenderTargetOnUpload(rc.mClearRenderTargetOnUpload),
                  mUseOffscreenInvalidation(rc.mUseOffscreenInvalidation) {}

        /// Assignment operator.
        RenderConfiguration & operator = (const RenderConfiguration & rc) { 
            mLoadAllRenderTargetsAtStartup = (rc.mLoadAllRenderTargetsAtStartup);
            mUseInvalidation = (rc.mUseInvalidation);
            mUseKickDisplayUpdate = (rc.mUseKickDisplayUpdate);
            mUseCourierSwapping = (rc.mUseCourierSwapping);
            mUseGlobalCameraSequenceNumber = (rc.mUseGlobalCameraSequenceNumber);
            mUse2DBefore3D = (rc.mUse2DBefore3D);
            mUseWakeUpRenderMechanism = (rc.mUseWakeUpRenderMechanism);
            mEnableSingleSwapping = (rc.mEnableSingleSwapping);
            mShall2DSwapIfSingleSwapping = (rc.mShall2DSwapIfSingleSwapping);
            mEnableValidatingLayout = (rc.mEnableValidatingLayout);
            mClearRenderTargetOnUpload = rc.mClearRenderTargetOnUpload;
            mUseOffscreenInvalidation = rc.mUseOffscreenInvalidation;
            return *this;
        }

        /** Controls if all rendertargets are loaded at startup. Default = disabled; render targets are loaded on their first usage
            when loading a scene and its cameras. */    
        void LoadAllRenderTargetsAtStartup(bool val) { mLoadAllRenderTargetsAtStartup = val; }

        /** Render only invalidated cameras and swap buffers which has been invalidated therefore. 
            Default = enabled: Cameras always have to be invalidated using the Invalidate methods of the view classes.
            This can be done inside the Widget implementations or ViewController implementations. 
            The render order is defined by the invalidation order, inside a View the cameras will be sorted by their sequence number.
            Disabling this feature will disable the invalidation mechanism and will use the Candera defined way of rendering, by using the 
            Candera methods RenderAllCameras, the swapping is also controlled by the asset content. */
        void UseInvalidation(bool val) { mUseInvalidation = val; }

        /** Enables or disables the KickDisplayUpdate on simulated displays; Default = enabled, this causes calling KickDisplayUpdate on
            every render cycle. */    
        void UseKickDisplayUpdate(bool val) { mUseKickDisplayUpdate = val; }

        /** Enables or disables the Courier specific swapping; Default = enabled, this means that Courier is responsible for swapping
            the buffers. Only invalidated buffers are swapped. By disabling this feature swapping is done internally by Candera, dependent
            on the asset content. Compared to UseInvalidation UseCourierSwapping does not disable the whole invalidation mechanism, but
            only disables the Courier swapping. */    
        void UseCourierSwapping(bool val) { mUseCourierSwapping = val; }

        /** Enables or disables the usage of the camera sequence order. Default = enabled, this means that Courier is now sorting the 
            Cameras by their sequence number before they are rendered. To remain backwards compatibility (<= Courier 1.5) sorting may be
            disabled. */
        void UseGlobalCameraSequenceNumber(bool val) { mUseGlobalCameraSequenceNumber = val; }

        /** Enables or disables the usage of the 2D before 3D order criteria. Default = enabled, this 
            means that Courier is now sorting the jobs by this criteria (i.e. 2D jobs before 3D ones) 
            before they are rendered. */
        void Use2DBefore3D(bool val) { mUse2DBefore3D = val; }

        /** Enables or disables the usage of the rendering wakeup mechanism. Default = disable. Enabling this feature means that if nothing is rendered,
            and no messages are sent to the ComponentMessageReceiver which has the RenderComponent(s) attachted, the receiver is waiting for the next invalidation 
            of a camera which causes sending a Courier::RenderWakeUpMsg. Of course other messages are processed during this "render sleeping" period.
            The message pool for this message is defined inside Courier, there is no need to define it inside the message configuration.
            A Disabled feature means that the render loop is executed every time even nothing has to be rendered. To remain backwards compatibility (<= Courier 1.5) 
            it is disabled. */
        void UseWakeUpRenderMechanism(bool val) { mUseWakeUpRenderMechanism = val; }

        /** Forces the RenderComponent2D (if existing) to swap both 2D and 3D RenderTargets. */    
        void UseRenderComponent2DForSwapBuffer() { mEnableSingleSwapping = true; mShall2DSwapIfSingleSwapping = true; }

        /** Forces the RenderComponent3D (if existing) to swap both 2D and 3D RenderTargets. */    
        void UseRenderComponent3DForSwapBuffer() { mEnableSingleSwapping = true; mShall2DSwapIfSingleSwapping = false; }

        /** Enables or disables calling ValidateLayout (2D Scene) before rendering a 2D camera. */    
        void EnableValidatingLayout(bool val) { mEnableValidatingLayout = val; }
        
        /** Enables/ disables clearing of the render targets when uploading them. Default = false. */  
        void ClearRenderTargetOnUpload(bool val) { mClearRenderTargetOnUpload = val; }

        /** Enables/disables invalidation of offscreen targets */ 
        void UseOffscreenInvalidation(bool val) { mUseOffscreenInvalidation = val; }

        bool ShallLoadAllRenderTargetsAtStartup() const { return mLoadAllRenderTargetsAtStartup; }
        bool ShallUseInvalidation() const { return mUseInvalidation; }
        bool ShallUseKickDisplayUpdate() const { return mUseKickDisplayUpdate; }
        bool ShallUseCourierSwapping() const { return mUseCourierSwapping; }
        bool ShallUseGlobalCameraSequenceNumber() const { return mUseGlobalCameraSequenceNumber; }
        bool ShallUse2DBefore3D() const { return mUse2DBefore3D; }
        bool ShallUseWakeUpRenderMechanism() const { return mUseWakeUpRenderMechanism; }
        bool ShallUseRenderComponent2DForSwapBuffer() const { return mEnableSingleSwapping && mShall2DSwapIfSingleSwapping; }
        bool ShallUseRenderComponent3DForSwapBuffer() const { return mEnableSingleSwapping && ! mShall2DSwapIfSingleSwapping; }
        bool ShallUseValidatingLayout() const { return mEnableValidatingLayout; }
        bool ShallClearRenderTargetOnUpload() const { return mClearRenderTargetOnUpload; }      
        bool ShallUseOffscreenInvalidation() const { return mUseOffscreenInvalidation; }

    private:
        bool mLoadAllRenderTargetsAtStartup;
        bool mUseInvalidation;
        bool mUseKickDisplayUpdate;
        bool mUseCourierSwapping;
        bool mUseGlobalCameraSequenceNumber;
        bool mUse2DBefore3D;
        bool mUseWakeUpRenderMechanism;
        bool mEnableSingleSwapping;
        bool mShall2DSwapIfSingleSwapping;
        bool mEnableValidatingLayout;
        bool mClearRenderTargetOnUpload;
        bool mUseOffscreenInvalidation;
};

/// @}
}

#endif
