//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Foundation_RenderComponentBase_h)
#define Courier_Foundation_RenderComponentBase_h

#include <FeatStd/Event/EventListener.h>
#include <FeatStd/Event/EventResult.h>

#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/Engine2D/Core/Node2D.h>


#include <Courier/Foundation/Component.h>

#include <Courier/Visualization/AsyncValidationWakeup.h>
#include <Courier/Visualization/InvalidationViews.h>
#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/Renderer.h>
#include <Courier/Visualization/RenderHint.h>
#include <Courier/Visualization/ViewFacade.h>

#include <FeatStd/Platform/AtomicOp.h>
#include <FeatStd/Platform/Thread.h>
#include <CanderaBehavior/Touch/TouchSession.h>

// Forward declarations
namespace Courier {
    class Message;
    class IViewHandler;
    class ViewFactory;
}

namespace Courier {
    /// @addtogroup COURIER_FOUNDATION
    /// @{
    /**
    * RenderComponentUpdateEvent will be emitted once for each update iteration of the RenderComponent.
    */
    class RenderComponentUpdateEvent : public FeatStd::Event
    {
    public:
        FEATSTD_RTTI_DECLARATION();

        /**
        * RenderComponentUpdateEvent default constructor.
        */
        RenderComponentUpdateEvent()
        {
        }

        /**
        * RenderComponentUpdateEvent destructor.
        */
        virtual ~RenderComponentUpdateEvent()
        {
        }

    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(RenderComponentUpdateEvent);
    };

    /** RenderComponentBase is the base implementation of specific RenderComponents which are controlled by RenderHint objects.
    */
    template<typename T, ComponentId id> class RenderComponentBase : public Component {
        typedef T RenderHint;
        typedef Component Base;

    public:
        ///
        RenderComponentBase() : Base(id)
        {
            Candera::UpdateSystem* updateSystem = Candera::EntityComponentSystem::EntitySystem::Create<Candera::UpdateSystem>();
            FEATSTD_DEBUG_ASSERT(0 != updateSystem);
            FEATSTD_UNUSED(updateSystem);

            Candera::ControllerSystem* controllerSystem = Candera::EntityComponentSystem::EntitySystem::Create<Candera::ControllerSystem>();
            FEATSTD_DEBUG_ASSERT(0 != controllerSystem);
            FEATSTD_UNUSED(controllerSystem);

            static_cast<void>(Internal::InvalidationViews::GetInstance()); // force initialization
        }

        ~RenderComponentBase()
        {
            bool wasSuccessful = Candera::EntityComponentSystem::EntitySystem::Destroy<Candera::ControllerSystem>();
            FEATSTD_DEBUG_ASSERT(wasSuccessful);
            FEATSTD_UNUSED(wasSuccessful);

            wasSuccessful = Candera::EntityComponentSystem::EntitySystem::Destroy<Candera::UpdateSystem>();
            FEATSTD_DEBUG_ASSERT(wasSuccessful);
            FEATSTD_UNUSED(wasSuccessful);
        }

        /** Initializes the RenderComponent.
            @param viewHandler IViewHandler used for rendering and message distribution.
            @return <em>true</em> if initialization was successful,
            <em>false</em> otherwise. */
        bool Init(IViewHandler * viewHandler)
        {
            return mViewFacade.Init(viewHandler);
        }

    protected:
        /// Calls the Render method of the ViewFacade.
        virtual bool OnExecute()
        {
            AcquireDataLock();
            Candera::TouchSession::UpdateEventDispatchResult touchSessionUpdateEventDispatchResult;
            Candera::TouchSession::UpdateEvent touchSessionUpdateEvent;
            Candera::TouchSession::GetInstance().OnEvent(touchSessionUpdateEvent, touchSessionUpdateEventDispatchResult);
            RenderComponentUpdateEvent renderComponentUpdateEvent;
            View::GetEventSource().DispatchEvent(renderComponentUpdateEvent);
            mViewFacade.Update(&mRenderHint);

            Candera::EntityComponentSystem::EntitySystem::HeartBeat();
            Candera::EntityComponentSystem::EntitySystem::Update();

            IViewHandler* viewHandler = mViewFacade.GetViewHandler();
            if (0 != viewHandler) {
                FeatStd::Internal::Vector<ViewId>& views = Internal::InvalidationViews::GetInstance().Swap();
                for (FeatStd::SizeType i = 0; i < views.Size(); ++i) {
                    View* view = viewHandler->FindView(views[i]);
                    if (0 != view) {
                        view->Invalidate();
                    }
                }
                views.Clear();
            }
            bool lRc = mViewFacade.Render(&mRenderHint) || touchSessionUpdateEventDispatchResult.IsFurtherUpdateRequired();
            ReleaseDataLock();
            if (Renderer::GetRenderConfiguration().ShallUseWakeUpRenderMechanism()) {
                lRc = (lRc) || 
                    (!Candera::TextRendering::AsyncTextRenderDispatcher::GetInstance().IsDispatcherEmpty()) ||
                    (Internal::InvalidationViews::GetInstance().ContainsUnhandledViews());
                return lRc;
            }
            // always keep running independent of Render return value.
            return true;
        }

        /// Delegates render specific messages to the ViewFacade. 
        virtual bool OnMessage(const Message & msg)
        {
            // we process the render message here, but do not change any data of view tree.
            // therefore lock the rendering
            AcquireDataLock();
            bool lRc = mViewFacade.OnRenderComponentMessage(id, &mRenderHint, msg);
            ReleaseDataLock();
            lRc = false; // continue messaging
            return lRc;
        }
        
    private:
        /// Delegates rendering related functionality.
        ViewFacade mViewFacade;
        /// RenderHint controls if 2D / 3D or both shall be rendered by this component.
        RenderHint mRenderHint;


        Internal::AsyncValidationWakeup mTextValidationWakeup;


    };

    /// RenderComponent renders both 2D and 3D Views.
    typedef RenderComponentBase<RenderHintAll, ComponentType::Renderer>  RenderComponent;
    /// RenderComponent2D renders only 2D Views.
    typedef RenderComponentBase<RenderHint2D, ComponentType::Renderer2D> RenderComponent2D;
    /// RenderComponent3D renders only 3D Views.
    typedef RenderComponentBase<RenderHint3D, ComponentType::Renderer3D> RenderComponent3D;

    /// @}
}

#endif
