//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Gdu.h"
#include <Courier/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitMetaInfo.h>
#include <Courier/Visualization/Renderer.h>
#include <Courier/Visualization/View.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>

namespace Courier {
    namespace Internal {
        const FeatStd::SizeType COURIER_GDU_NOT_UPLOADED =  0;
    }

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;

// ------------------------------------------------------------------------
Gdu::Gdu(SizeType layerIndex) :
#if defined(CANDERA_3D_ENABLED)
    mOwnerState(OwnerInvalid),
#endif // CANDERA_3D_ENABLED
    mGdu(0),
    m2DRT(0),
    m3DRT(0),
#ifdef CANDERA_2D_ENABLED
    mClearData2D(),
#endif // CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
    mClearData3D(),
#endif // CANDERA_3D_ENABLED
    mImageSourceHandle(0),
    mSurfaceHandle(0),
    mLayerId(0),
    mLayerIndex(layerIndex),
    mUnitCategory(DevicePackageDescriptor::DisplayTarget3D),
    mInvalidate(false),
    mIsActive(false),
    mIsOffscreen(false),
    mIsLoaded(false),
    mHasCameraWithRenderStrategy(false),
    mUploadRequestNo(Internal::COURIER_GDU_NOT_UPLOADED),
    mRenderState(RenderTargetValid),
    mLastRenderJob(0)
{
}

// ------------------------------------------------------------------------
Gdu::Gdu(Candera::GraphicDeviceUnit * gdu, SizeType layerIndex) :
#if defined(CANDERA_3D_ENABLED)
    mOwnerState(OwnerInvalid),
#endif
    mGdu(gdu),
    m2DRT(0),
    m3DRT(0),
#ifdef CANDERA_2D_ENABLED
    mClearData2D(),
#endif // CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
    mClearData3D(),
#endif // CANDERA_3D_ENABLED
    mImageSourceHandle(0),
    mSurfaceHandle(0),
    mLayerId(0),
    mLayerIndex(layerIndex),
    mUnitCategory(DevicePackageDescriptor::DisplayTarget3D),
    mInvalidate(false),
    mIsActive(true),
    mIsOffscreen(false),
    mIsLoaded(false),
    mHasCameraWithRenderStrategy(false),
    mUploadRequestNo(Internal::COURIER_GDU_NOT_UPLOADED),
    mRenderState(RenderTargetValid),
    mLastRenderJob(0)
{
    FEATSTD_PANIC_IF(mGdu==0, "Candera::GraphicDeviceUnit not valid");

    m2DRT = gdu->ToRenderTarget2D();
    m3DRT = gdu->ToRenderTarget3D();

    FEATSTD_PANIC_IF(((m3DRT==0) && (m2DRT==0)), "No valid 2D or 3D Rendertarget");

    GraphicDeviceUnitTypeHandle typeHandle = mGdu->GetUnitType();
    const MetaInfo::GraphicDeviceUnitMetaInfo * metaInfo = DevicePackageDescriptor::GetMetaInformation(typeHandle);
    if(metaInfo!=0) {
        mUnitCategory = DevicePackageDescriptor::GetUnitCategory(typeHandle);
        mIsOffscreen = IsOffScreen(mUnitCategory);
        MetaInfo::GraphicDeviceUnitPropertyMetaInfo * propMetaInfo = metaInfo->LookupItem("Layer");
        if(0==propMetaInfo) {
            propMetaInfo = metaInfo->LookupItem("LayerId");
        }
        if (propMetaInfo!=0) {
            Char buffer[32];
            buffer[0] = '\0';
            if(propMetaInfo->Get(mGdu,buffer,32)) {
                ::sscanf(buffer,"%d",&mLayerId);
                //mLayerId = ::atoi(buffer);
            }
        }
    }
}

// ------------------------------------------------------------------------
void Gdu::Finalize()
{
    Candera::GraphicDeviceUnit * canderaGdu = GetGdu();
#if defined(CANDERA_3D_ENABLED)
    if ((0 != mOwnerAccess.GetCompositionCamera()) && (0 != mOwnerAccess.GetCompositionCamera()->GetParent())) {
        static_cast<void>(mOwnerAccess.GetCompositionCamera()->RemoveNodeListener(this));
    }
    RenderTarget3D * renderTarget3D = canderaGdu->ToRenderTarget3D();
    if (0 != renderTarget3D) {
        mClearData3D.Finalize();
        (void)renderTarget3D->Activate();
    }
#endif
#if defined(CANDERA_2D_ENABLED)
    FEATSTD_LINT_NEXT_EXPRESSION(774, "No, it does not evaluate always to true")
    RenderTarget2D * renderTarget2D = canderaGdu->ToRenderTarget2D();
    if (0 != renderTarget2D) {
        mClearData2D.Finalize();
        (void)renderTarget2D->Activate();
    }
#endif
    if (Internal::COURIER_GDU_NOT_UPLOADED != mUploadRequestNo) {
        mUploadRequestNo = Internal::COURIER_GDU_NOT_UPLOADED;

        ContextResourcePool& crp = ContextResourcePool::GetActive();
        canderaGdu->Unload();
        static_cast<void>(crp.Activate());

        COURIER_LOG_DEBUG("Unloaded Gdu %s; mUploadRequestNo was reset to %d", canderaGdu->GetName(), mUploadRequestNo);
    }
}

// ------------------------------------------------------------------------
bool Gdu::IsOffScreen(DevicePackageDescriptor::UnitCategory unitCategory)
{
    bool rc = false;
    switch(unitCategory) {
        case DevicePackageDescriptor::OffscreenTarget3D:
        case DevicePackageDescriptor::OffscreenTarget2D:
        case DevicePackageDescriptor::Mixed2D3DOffscreenTarget:
        case DevicePackageDescriptor::OffscreenTarget2Dto3D:
        case DevicePackageDescriptor::OffscreenTarget3Dto2D:
            rc = true;
            break;
        case DevicePackageDescriptor::DisplayTarget3D:
        case DevicePackageDescriptor::DisplayTarget2D:
        case DevicePackageDescriptor::Mixed2D3DDisplayTarget:
        case DevicePackageDescriptor::ExternalSource:
        case DevicePackageDescriptor::CustomObject:
            rc = false;
            break;
    }
    return rc;
}

// ------------------------------------------------------------------------
#if defined(CANDERA_2D_ENABLED)
Camera2D * Gdu::GetClearCamera2D()
{
    FEATSTD_DEBUG_ASSERT(Is2D());
    mClearData2D.Init(this);
    return mClearData2D.GetClearCamera();
}
#endif

// ------------------------------------------------------------------------
#if defined(CANDERA_3D_ENABLED)
Camera * Gdu::GetClearCamera3D()
{
    FEATSTD_DEBUG_ASSERT(Is3D());
    mClearData3D.Init(this);
    return mClearData3D.GetClearCamera();
}

// ------------------------------------------------------------------------
Candera::Internal::GraphicDeviceUnitOwnerAccess& Gdu::GetOwnerAccess()
{
    if (OwnerInvalid == mOwnerState) {
        if (mIsLoaded) {
            // a reupload is required to update the owner
            static_cast<void>(mGdu->Upload());
            // no need to update the mUploadIdx as the existing one should be valid as gdu is already uploaded (mIsLoaded == true)
        }
        if ((0 != mOwnerAccess.GetCompositionCamera()) && (0 != mOwnerAccess.GetCompositionCamera()->GetParent())) {
            static_cast<void>(mOwnerAccess.GetCompositionCamera()->RemoveNodeListener(this));
        }
        mOwnerAccess = (0 != mGdu) ? Candera::Internal::GraphicDeviceUnitOwnerAccess(mGdu) : Candera::Internal::GraphicDeviceUnitOwnerAccess();
        if (0 != mOwnerAccess.GetCompositionCamera()) {
            static_cast<void>(mOwnerAccess.GetCompositionCamera()->AddNodeListener(this));
        }
        switch (mOwnerAccess.GetOwnerType()) {
        case Candera::Internal::GraphicDeviceUnitOwnerAccess::NoOwner:
            mOwnerState = NoOwner;
            break;
        case Candera::Internal::GraphicDeviceUnitOwnerAccess::GraphicDeviceUnitOwner:
            mOwnerState = GraphicDeviceUnitOwner;
            break;
        case Candera::Internal::GraphicDeviceUnitOwnerAccess::ContextResourcePoolOwner:
            mOwnerState = ContextResourcePoolOwner;
            break;
        }
    }
    return mOwnerAccess;
}

// ------------------------------------------------------------------------
void Gdu::OnNodeRemoved(Candera::Node* /*parent*/, Candera::Node* node)
{
    if (node == mOwnerAccess.GetCompositionCamera()) {
        mOwnerState = OwnerInvalid;
        mOwnerAccess = Candera::Internal::GraphicDeviceUnitOwnerAccess();
    }
}

#endif

// ------------------------------------------------------------------------
void Gdu::SetActivate(bool active)
{
    mIsActive = active;
    if (active) {
        (void)View::IncrementInvalidationId();
    }
}

// ------------------------------------------------------------------------
bool Gdu::Upload(Renderer* renderer)
{
    FEATSTD_PANIC_IF(0 == mGdu, "GDU not valid");

    if (! mIsLoaded) {
#if defined(CANDERA_3D_ENABLED)
        (void)GetOwnerAccess();
#endif
        mIsLoaded = mGdu->Upload();
        mGdu->ApplyChanges();
        
        if (mIsLoaded) {
            static FeatStd::SizeType uploadRequestNo = Internal::COURIER_GDU_NOT_UPLOADED;
            mUploadRequestNo = ++uploadRequestNo;
            COURIER_LOG_DEBUG("Uploaded Gdu %s; mUploadRequestNo = %d", mGdu->GetName(), mUploadRequestNo);
        }

        if (mIsLoaded && mIsOffscreen) {
#if defined(CANDERA_3D_ENABLED)
            ImageSource3D * imageSource3D = mGdu->ToImageSource3D();
            if (0 != imageSource3D) {
                mImageSourceHandle = imageSource3D->GetVideoMemoryHandle();
                COURIER_LOG_INFO("Found VideoMemoryHandle(%08x)", mImageSourceHandle);
            }
#endif
#if defined(CANDERA_2D_ENABLED)
            ImageSource2D * imageSource2D = mGdu->ToImageSource2D();
            if (0 != imageSource2D) {
                mSurfaceHandle = imageSource2D->GetSurfaceHandle();
                COURIER_LOG_INFO("Found SurfaceHandle(%08x)", mSurfaceHandle);
            }
#endif
        }

        if (renderer->GetRenderConfiguration().ShallClearRenderTargetOnUpload()) {
            bool isMixed2D3D = Is3D() && Is2D();
#if defined(CANDERA_3D_ENABLED)
            if (Is3D() && 
                ((!isMixed2D3D) || (!renderer->Is2DRequest()))) {
                Camera * clearCamera3D = GetClearCamera3D();
                if (0 != clearCamera3D) {
                    renderer->Queue3DJob(0, clearCamera3D, cCOURIER_DEFAULT_RENDER_3D_COUNT, FeatStd::Optional<Candera::Rectangle>(), true);
                }
            }
#endif
#if defined(CANDERA_2D_ENABLED)
            if (Is2D() && 
                ((!isMixed2D3D) || renderer->Is2DRequest())) {
                Camera2D * clearCamera2D = GetClearCamera2D();
                if (0 != clearCamera2D) {
                    renderer->Queue2DJob(0, clearCamera2D, cCOURIER_DEFAULT_RENDER_2D_COUNT, FeatStd::Optional<Candera::Rectangle>(), true);
                }
            }
#endif
        }
        return mIsLoaded;
    }
    return true;
}

// ------------------------------------------------------------------------
void Gdu::Unload(Renderer* renderer)
{
    FEATSTD_PANIC_IF( 0 == mGdu, "GDU not valid");

    renderer->DeleteRenderJobs(this);

    if (mIsLoaded) {
        mUploadRequestNo = Internal::COURIER_GDU_NOT_UPLOADED;

        ContextResourcePool& crp = ContextResourcePool::GetActive();
        mGdu->Unload();
        static_cast<void>(crp.Activate());

        mImageSourceHandle = 0;
        COURIER_LOG_DEBUG("Unloaded Gdu %s; mUploadRequestNo was reset to %d", mGdu->GetName(), mUploadRequestNo);
    }
    mIsLoaded = false;
}

// ------------------------------------------------------------------------
void Gdu::Activate() const
{
    if(mIsLoaded) {
        Candera::GraphicDeviceUnit * canderaGdu = GetGdu();
#if defined(CANDERA_3D_ENABLED)
        RenderTarget3D * renderTarget3D = canderaGdu->ToRenderTarget3D();
        if(renderTarget3D!=0) {
            (void)renderTarget3D->Activate();
        }
#endif
#if defined(CANDERA_2D_ENABLED)
        FEATSTD_LINT_NEXT_EXPRESSION(774, "No, it does not evaluate always to true")
        RenderTarget2D * renderTarget2D = canderaGdu->ToRenderTarget2D();
        if(renderTarget2D!=0) {
            (void)renderTarget2D->Activate();
        }
#endif
    }
}

// ------------------------------------------------------------------------
void Gdu::Sync() const
{
    if(mIsLoaded) {
        if(ShallSync()) {
#if defined(CANDERA_3D_ENABLED)
            Candera::GraphicDeviceUnit * canderaGdu = GetGdu();
            RenderTarget3D * renderTarget3D = canderaGdu->ToRenderTarget3D();
            if(renderTarget3D!=0) {
                renderTarget3D->Sync();
            }
#endif
        }
    }
}

// ------------------------------------------------------------------------
bool Gdu::ShallSync() const
{
    return mUnitCategory==Candera::DevicePackageDescriptor::Mixed2D3DDisplayTarget;
}

// ------------------------------------------------------------------------
void Gdu::Log() const
{
#ifdef FEATSTD_LOG_ENABLED
    if(mGdu!=0) {
        Int displayId = mGdu->GetDisplay();
        GraphicDeviceUnitTypeHandle typeHandle = mGdu->GetUnitType();
        DevicePackageDescriptor::UnitCategory unitCategory = DevicePackageDescriptor::GetUnitCategory(typeHandle);
        const MetaInfo::GraphicDeviceUnitMetaInfo * metaInfo = DevicePackageDescriptor::GetMetaInformation(typeHandle);
        const Char * name = (metaInfo!=0) ? metaInfo->GetName() : "<invalid>";
        const Char * dim = (((m3DRT!=0) && (m2DRT!=0))) ? "2D3D" : ((m3DRT!=0) ? "3D" : ((m2DRT!=0) ? "2D" : "invalid"));
        COURIER_LOG_INFO("Display(#%d) Layer(#%d) Dimension(%s) 2D-RT(%08x) 3D-RT(%08x) Category(%d) Name(%s)",
                          displayId ,  mLayerId,  dim,          m2DRT,      m3DRT,      unitCategory,name);
    } else {
        COURIER_LOG_ERROR("invalid Gdu, cant log info");
    }
#endif
}

}
