//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_CameraGroupHandler_h)
#define Courier_Visualization_CameraGroupHandler_h

#include <Candera/Candera.h>
#include <Candera/EngineBase/Common/CameraGroup.h>
#if defined(CANDERA_2D_ENABLED)
#include <Candera/Engine2D/Core/Node2DListener.h>
#endif
#if defined(CANDERA_3D_ENABLED)
#include <Candera/Engine3D/Core/NodeListener.h>
#endif

#include <Courier/Util/Util.h>
#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/ViewId.h>
#include <FeatStd/Util/String.h>
#include <CanderaPlatform/OS/StringPlatform.h>

namespace Courier { 
    namespace UnitTest {
        class CameraGroupHandlerWrapper;
    }

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The CameraGroupHandler.
*/
#if defined(CANDERA_3D_ENABLED)
    class ViewScene3D;
#endif

#if defined(CANDERA_2D_ENABLED)
    class ViewScene2D;
#endif

class CameraGroupHandler
{
    friend class Courier::UnitTest::CameraGroupHandlerWrapper;

    public:
        ///
        CameraGroupHandler();
        
        ///
        virtual ~CameraGroupHandler();
        
        ///
        bool Init(IViewHandler * viewHandler);

        /// Frees resources.
        void Finalize();

        /// 
        bool CreateCameraGroupViews(ViewAction::Enum action, const ItemId & cameraGroupId, bool loadOnCreation);

        /// 
        bool ActivateCameraGroupCameras(const ItemId & cameraGroupId, bool activate, bool enableCameras, bool onlyCameraGroupCameras);

        /// 
        bool LoadCameraGroupViews(const ItemId & cameraGroupId, bool load);

        ///
        void OnViewInitContent(bool init, View& view);

    private:
        struct NodesPathNames {
        public:
            NodesPathNames(const ViewId& viewId, const Char* sceneName, const Char* cameraName):
                mViewId(viewId),
                mSceneName(sceneName),
                mCameraName(cameraName) {}

            ViewId mViewId;
            FeatStd::String mSceneName;
            FeatStd::String mCameraName;

        };

        typedef Vector<NodesPathNames> NodesPathNamesVct;

        class Group:
#if defined(CANDERA_3D_ENABLED)
            public Candera::NodeListener
            #if defined(CANDERA_2D_ENABLED)
            ,
            #endif
#endif
#if defined(CANDERA_2D_ENABLED)
            public Candera::Node2DListener
#endif 
        {
            public:
                Group();
                Group(const ItemId & id, Candera::CameraGroup* cg, IViewHandler * viewHandler);
                Group(const Group & ref);

                ~Group();
                const ItemId & GetId() const { return mId; }
                const Candera::CameraGroup* GetCameraGroup() const { return mCameraGroup; }

                bool IsRenderingEnabled() const { return mIsRenderingEnabled; }
                void SetRenderingEnabled(bool enable) { mIsRenderingEnabled = enable; }

                bool IsActivated() const { return mIsActivated; }
                void SetActivated(bool enable) {mIsActivated = enable;}

                typedef Candera::CanderaObject* (*GetParentCallback)(Candera::CanderaObject* object);

                void StoreNames(const ViewScene* viewScene, Candera::CanderaObject* parent, Candera::CanderaObject* camera, GetParentCallback getParentCallback);

                FeatStd::SizeType GetNamesVctCount() const {return mNameVct.Size();}
                NodesPathNames* GetNodesPathNames(FeatStd::SizeType idx);
                void RetrieveLostCamerasFromScene(ViewScene* viewScene);

#if defined(CANDERA_3D_ENABLED)
                virtual void OnTransformChange(Candera::Node*) {};
                virtual void OnNodeRemoved(Candera::Node* parent, Candera::Node* node);

                FeatStd::SizeType GetCamera3DCount() const;
                const Char* GetScene3DName(Int index) const;
                const Char* GetCamera3DName(Int index) const;
                Candera::Camera* GetCamera3D(Int index) const;
                ViewScene3D* FindViewScene3D(const Char* sceneName) const;
                void ActivateView(ViewScene3D& viewScene, Int cameraId, bool activate) const;
                void EnableRendering(ViewScene3D& viewScene, Int cameraId, bool enable, bool onlyCameraGroupCameras) const;
                void Invalidate(ViewScene3D& viewScene, Int cameraId, bool onlyCameraGroupCameras, const FeatStd::Optional<Candera::Rectangle>& dirtyArea) const;
#endif
#if defined(CANDERA_2D_ENABLED)
                virtual void OnTransformChange(Candera::Node2D*) {}
                virtual void OnNodeRemoved(Candera::Node2D* parent, Candera::Node2D* node);

                FeatStd::SizeType GetCamera2DCount() const;
                const Char* GetScene2DName(Int index) const;
                const Char* GetCamera2DName(Int index) const;
                Candera::Camera2D* GetCamera2D(Int index) const;
                ViewScene2D* FindViewScene2D(const Char* sceneName) const;
                void ActivateView(ViewScene2D& viewScene, Int cameraId, bool activate) const;
                void EnableRendering(ViewScene2D& viewScene, Int cameraId, bool enable, bool onlyCameraGroupCameras) const;
                void Invalidate(ViewScene2D& viewScene, Int cameraId, bool onlyCameraGroupCameras, const FeatStd::Optional<Candera::Rectangle>& dirtyArea) const;
#endif

            private:
                void RegisterAsListener();

                template<typename T>
                static T* Valid(T* p);
                static const Char* GetName(const Candera::CanderaObject* object);
                static bool ValidIndex(FeatStd::OffsetType index, FeatStd::OffsetType count);

                ItemId mId;
                Candera::CameraGroup* mCameraGroup;
                /// The used ViewHandler.
                IViewHandler * mViewHandler;
                /// bool flag for insuring consecutive calls to EnableRendering with the same value for enable param are only processed the first time.
                bool mIsRenderingEnabled;
                /// bool flag for insuring consecutive calls to ActivateView with the same value for activate param are only processed the first time.
                bool mIsActivated;
                NodesPathNamesVct mNameVct;
        };

        ///
        void EnableCameras(bool is2D, const Char * sceneName, bool enableCameras);
        ///
        void LoadView(bool is2D, const Char * sceneName, bool load);
        ///
        CameraGroupHandler::Group * GetGroup(const ItemId & cameraGroupId);
        ///
        bool Load(const ItemId & cameraGroupId);
#if defined(CANDERA_3D_ENABLED)
        ///
        bool IsCamera3DMultipleUsed(const Candera::Camera * lookupCamera, const CameraGroupHandler::Group * excludeGroup) const;
#endif
#if defined(CANDERA_2D_ENABLED)
        ///
        bool IsCamera2DMultipleUsed(const Candera::Camera2D * lookupCamera, const CameraGroupHandler::Group * excludeGroup) const;
#endif
        typedef Courier::Vector< Group > GroupVector;
        ///
        GroupVector mGroupVector;

        /// The used ViewHandler.
        IViewHandler * mViewHandler;

};

/// @}
}

#endif
