//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetAccessor.h"

#include <Courier/Visualization/AssetConfiguration.h>

#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <CanderaAssetLoader/AssetLoader.h>
#include <Courier/Diagnostics/Log.h>
#include <Courier/Visualization/ViewId.h>

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;

namespace Internal {
Int AssetLibHelper::GetCount(const DefaultAssetProvider* assetProvider, Candera::AssetLib assetLib)
{
    if (0 != assetProvider) {
        return assetProvider->GetAssetDescriptor().GetAssetObjectCount(assetLib);
    }
    return 0;
}
}

// ------------------------------------------------------------------------
AssetAccessor::AssetAccessor() : mAssetConfiguration(0), mContentLoader(0), mAssetProvider(0)
{
}

// ------------------------------------------------------------------------
FEATSTD_LINT_CURRENT_SCOPE(1579, "member need not to be freed here")
AssetAccessor::~AssetAccessor()
{
    Finalize();
}

// ------------------------------------------------------------------------
AssetAccessor & AssetAccessor::GetInstance()
{
    static AssetAccessor assetAccessor;
    return assetAccessor;
}

// ------------------------------------------------------------------------
void AssetAccessor::Finalize()
{
    if(mAssetConfiguration!=0) {
        if(mAssetProvider!=0) {
            mAssetProvider->Finalize();
            mAssetProvider = 0;
        }
        mContentLoader = 0;
        mAssetConfiguration = 0;
    }
}

// ------------------------------------------------------------------------
bool AssetAccessor::Init(Candera::AssetConfig * assetConfiguration, UInt32 validationFlags)
{
    FEATSTD_DEBUG_ASSERT(assetConfiguration!=0);

    if(assetConfiguration==0) {
        COURIER_LOG_ERROR("AssetConfiguration not set.");
        return false;
    }

    if(mAssetConfiguration==assetConfiguration) {
        COURIER_LOG_INFO("AssetAccessor previously initialized with same AssetConfiguration.");
        return true;
    } 

    if((mAssetConfiguration!=0) && (mAssetConfiguration!=assetConfiguration)) {
        COURIER_LOG_ERROR("AssetAccessor already initialized with different AssetConfiguration, has to be finalized before.");
        return false;
    } 

    if(assetConfiguration->GetAssetRepositoryCount()==0) {
        COURIER_LOG_ERROR("AssetConfiguration has no Candera::AssetRepository added.");
        return false;
    }

    mAssetProvider = &DefaultAssetProvider::GetInstance();
    FEATSTD_LINT_NEXT_EXPRESSION(948, "mContentLoader might be == 0")
    bool rc = (mAssetProvider!=0);
    FEATSTD_LINT_NEXT_EXPRESSION(774, "mAssetProvider might be == 0")
    if(rc) {
        mContentLoader = &ContentLoader::GetInstance();
        FEATSTD_LINT_NEXT_EXPRESSION(948, "mContentLoader might be == 0")
        rc = (mContentLoader!=0);
    }
    FEATSTD_LINT_NEXT_EXPRESSION(774, "mContentLoader might be == 0")
    if(rc) {
        mContentLoader->Init(mAssetProvider);
        rc = mAssetProvider->Initialize(assetConfiguration, validationFlags);
    }
    if(! rc) {
        COURIER_LOG_ERROR("Asset initialization failed.");
        return false;
    }

    const AssetDescriptor &assetDescriptor = mAssetProvider->GetAssetDescriptor();

    const Char * platformName = assetDescriptor.GetPlatformName();
    bool isForTarget = assetDescriptor.IsForTarget();
    COURIER_LOG_INFO("Asset was generated for platform \"%s\" %s\n",
                      ((platformName!=0) ? platformName : "unknown"),
                      (isForTarget?"TARGET":"HOST"));
    FEATSTD_UNUSED2(platformName, isForTarget); // if logging is disabled, variable/parameter is not used

    mAssetConfiguration = assetConfiguration;
    return rc;
}

// ------------------------------------------------------------------------
Int AssetAccessor::GetDisplayCount() const
{
    return Internal::AssetLibHelper::GetCount(mAssetProvider, DisplayLib);
}

// ------------------------------------------------------------------------
bool AssetAccessor::GetDisplayData(Int idx, DisplayData & displayData)
{
    displayData = DisplayData();
    if (0 != mAssetProvider){
        Int currentIdx = 0;
        for (AssetDescriptor::AssetIdIterator displayIt = mAssetProvider->GetAssetDescriptor().GetAssetIdIterator(DisplayLib); displayIt.IsValid(); ++displayIt) {
            if (currentIdx == idx) {
                displayData.mDisplayName = mAssetProvider->GetNameById(DisplayLib, *displayIt, 0);
                if (displayData.mDisplayName==0) {
                    FEATSTD_LOG_DEBUG("GetNameById is not valid");
                }
                displayData.mDisplayId = mAssetProvider->GetDisplayDataById(*displayIt, displayData.mWidth, displayData.mHeight);
                break;
            }
            ++currentIdx;
        }
        if (0 == displayData.mDisplayName) {
            displayData.mDisplayName = "INVALID";
        }
        COURIER_LOG_INFO("Display Idx(%d) (#%d, %s(%d,%d))", idx, displayData.mDisplayId, displayData.mDisplayName, displayData.mWidth, displayData.mHeight);
        return true;
    }
    return false;
}
}
