//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_ObjectList_h)
#define Courier_Util_ObjectList_h

#include <Courier/Base.h>
#include <Courier/Util/Traits.h>

namespace Courier { namespace Internal {
    /// @addtogroup COURIER_UTIL
    /// @{
    /** DefaultAccessor defines default accessor functions to the given type Next and SetNext methods
        @param T type providing a "T* Next() const" and "void SetNext(T *)" method */
    template<typename T> struct DefaultAccessor {
        /** retrieve next prointer of the given object
            The expected getter signature is 'T* Next() const'
            @param current pointer to current object
            @return pointer to the object following current or null */
        static inline T* Next(T *current) {
            return current->Next();
        }

        /** Set the next pointer of the given object
            the expected setter signature is 'void SetNext(T *)'
            @param current the object which next pointer should be set
            @param next the pointer to the object following current or null */
        static inline void SetNext(T *current, T *next) {
            current->SetNext(next);
        }
    };

    /** ObjectListFunc exports functions to modify intrusive linked lists
        @param T node type maintained in list
        @param Accessor accessor class to read and write next pointer to a node object */
    template<typename T, typename Accessor = DefaultAccessor<T> > class ObjectListFunc {
        public:
            typedef T NodeType;             ///< type of objects maintained by the list
            typedef Accessor NodeAccessor;  ///< access object type

            /** add the given object to the list
                the node is pushed to the front of the list
                @param head the head of the list
                @param node the node to be added to the list
                @return true if the node could be added to the list, false otherwise */
            static bool Add(T *&head, T *node) {
                FEATSTD_DEBUG_ASSERT(node != 0);
                bool ok = Accessor::Next(node) == 0;
                if (ok) {
                    Accessor::SetNext(node, head);
                    head = node;
                }
                return ok;
            }

            /** append the given object to the list
                @param head the head of the list
                @param node the node to be appended to the list
                @return true if the node could be added to the list, false otherwise */
            static bool Append (T *&head, T *node) {
                FEATSTD_DEBUG_ASSERT(node != 0);
                bool ok = Accessor::Next(node) == 0;
                if (ok) {
                    T *prev = 0;
                    T *p = head;
                    while (p != 0) {
                        prev = p;
                        p = Accessor::Next(p);
                    }
                    if (prev == 0) {
                        head = node;
                    }
                    else {
                        Accessor::SetNext(node, 0);
                        Accessor::SetNext(prev, node);
                    }
                }
                return ok;
            }

            /** append the given object to the list
                providing the lastNode parameter significantly improves append performance
                @param head the head of the list
                @param node the node to be appended to the list
                @param lastNode the last node in the list
                @return true if the node could be added to the list, false otherwise */
            static bool Append (T *&head, T *node, T *lastNode) {
                FEATSTD_DEBUG_ASSERT(node != 0);
                bool ok = Accessor::Next(node) == 0;
                if (ok) {
                    if (lastNode == 0) {
                        head = node;
                    }
                    else {
                        Accessor::SetNext(node, 0);
                        Accessor::SetNext(lastNode, node);
                    }
                }
                return ok;
            }

            /** Remove the given node from the list
                @param head the head of the list
                @param node the node to be removed from the list
                @return true if the node could be located and removed, false otherwise */
            static bool Remove(T *&head, T *node) {
                T *prev;
                T *p = Locate(head, node, prev);
                if (p != 0) {
                    if (p == head) {
                        head = Accessor::Next(head);
                    }
                    else {
                        Accessor::SetNext(prev, Accessor::Next(node));
                    }
                    Accessor::SetNext(node, 0);
                }
                return p != 0;
            }

            /** locate the given node in the list
                @param head the head of the list
                @param node the node to be located in the list
                @return the node object if the node could be located, null otherwise */
            static T* Locate(T *head, T *node) {
                T *prev;
                return Locate(head, node, prev);
            }

            /** count the number of nodes in the given list
                @param head the head of the list
                @return the number of nodes in the list */
            static UInt32 Count(T *head) {
                UInt32 n = 0;
                while (head != 0) {
                    ++n;
                    head = Accessor::Next(head);
                }
                return n;
            }

            /** Remove all elements from the list
                @param head the head of the list */
            static void Clear(T *&head) {
                while (head != 0) {
                    T *p = head;
                    head = Accessor::Next(head);
                    Accessor::SetNext(p, 0);
                }
                head = 0;
            }

        private:
            /** locate the given node in the list
                @param head the head of the list
                @param node the node to be located in the list
                @param prev the node prior to the located node or null
                @return the node object if the node could be located, null otherwise */
            static T *Locate(T *head, T *node, T *&prev) {
                prev = 0;
                T *p = head;
                while ((p != 0) && (p != node)) {
                    prev = p;
                    p = Accessor::Next(p);
                }
                return p;
            }
    };

    /** SortedObjectListFunc exports functions to modify intrusive sorted linked lists
        @param T node type maintained in list
        @param SortPredicate function pointer to sort predicate
        @param Accessor accessor class to read and write next pointer to a node object */
    template<typename T, bool (*SortPredicate)(const T&, const T&), typename Accessor = DefaultAccessor<T> > class SortedObjectListFunc {
        public:

            typedef T NodeType;             ///< type of objects maintained by the list
            typedef Accessor NodeAccessor;  ///< access object type

            /** add the given object to the list
                @param head the head of the list
                @param node the node to be added to the list */
            static bool Add(T *&head, T *node) {
                FEATSTD_DEBUG_ASSERT(node != 0);
                bool ok = Accessor::Next(node) == 0;
                if (ok) {
                    T *prev;
                    T *p = Locate(head, node, prev);
                    if (p == 0) {
                        if (prev == 0) {
                            Accessor::SetNext(node, head);
                            head = node;
                        }
                        else {
                            Accessor::SetNext(node, Accessor::Next(prev));
                            Accessor::SetNext(prev, node);
                        }
                    }
                }
                return ok;
            }

            /** Remove the given node from the list
                @param head the head of the list
                @param node the node to be removed from the list
                @return true if the node could be located and removed, false otherwise */
            static bool Remove(T *&head, T *node) {
                T *prev;
                T *p = Locate(head, node, prev);
                if (p != 0) {
                    if (p == head) {
                        head = Accessor::Next(head);
                    }
                    else {
                        Accessor::SetNext(prev, Accessor::Next(node));
                    }
                    Accessor::SetNext(node, 0);
                }
                return p != 0;
            }

            /** locate the given node in the list
                @param head the head of the list
                @param node the node to be located in the list
                @return the node object if the node could be located, null otherwise */
            static T* Locate(T *head, T *node) {
                T *prev;
                return Locate(head, node, prev);
            }

            /** count the number of nodes in the given list
                @param head the head of the list
                @return the number of nodes in the list */
            static inline UInt32 Count(T *head) {
                return ObjectListFunc<T>::Count(head);
            }

            /** Remove all elements from the list
                @param head the head of the list */
            static void Clear(T *&head) {
                ObjectListFunc<T>::Clear(head);
            }

        private:
            /** locate the given node in the list
                @param head the head of the list
                @param node the node to be located in the list
                @param prev the node prior to the located node or null
                @return the node object if the node could be located, null otherwise */
            static T *Locate(T *head, T *node, T *&prev) {
                prev = 0;
                T *p = head;
                while ((p != 0) && (p != node) && SortPredicate(*node, *p)) {
                    prev = p;
                    p = Accessor::Next(p);
                }
                return (p == node) ? p : 0;
            }
    };

    /** Iterator for ObjectLists
        @param T the node type
        @param Accessor accessor object to get and set the next pointer of nodes */
    template<typename T, typename Accessor> class ObjectListIterator {
        public:
            typedef typename Internal::ConstRemover<T>::Type NonConstType;

            //@{
            ///
            inline ObjectListIterator() {
                mCurrent = 0;
            };

            /** ObjectListIterator.
                @param head the head of the list or null */
            inline ObjectListIterator(NonConstType *head) {
                mCurrent = head;
            };
            //@}

            /** Return the current node.
                @return current node object */
            inline T* operator*() const {
                return mCurrent;
            }

            //@{
            /** Advance to the next object in sequence. */
            ObjectListIterator& operator++() {
                if (mCurrent != 0) {
                    mCurrent = Accessor::Next(mCurrent);
                }
                return *this;
            }

            ObjectListIterator operator++(int) {
                ObjectListIterator it(mCurrent);
                if (mCurrent != 0) {
                    mCurrent = Accessor::Next(mCurrent);
                }
                return it;
            }
            //@}

            //@{
            /** comparison operators */
            inline bool operator==(const ObjectListIterator &it) const {
                return mCurrent == it.mCurrent;
            }

            inline bool operator!=(const ObjectListIterator &it) const {
                return mCurrent != it.mCurrent;
            }
            //@}

        private:
            NonConstType *mCurrent;     ///< pointer to the current object
    };

    /** Foundation of ObjectList class
        @param Func type providing ObjectList functions (i.e. ObjectListFunc or SortedObjectListFunc) */
    template<typename Func> class ObjectListBase {
        public:
            /// the object type managed by the list
            typedef typename Func::NodeType T;
            /// iterator
            typedef ObjectListIterator<T, typename Func::NodeAccessor> Iterator;
            /// const iterator
            typedef ObjectListIterator<const T, typename Func::NodeAccessor> ConstIterator;

            /** Constructs an empty ObjectList. */
            ObjectListBase() {
                mHead = 0;
            }

            /** Add the given node to the list.
                @param node the object to add to the list */
            inline bool Add(T *node) {
                return Func::Add(mHead, node);
            }

            /** Append the given node to the list.
                Append is not available for sorted lists
                @param node the object to append to the list */
            inline bool Append(T *node) {
                return Func::Append(mHead, node);
            }

            /** Append the given node to the list.
                Append is not available for sorted lists
                @param node the object to append to the list
                @param lastNode the last node in the list */
            inline bool Append(T *node, T *lastNode) {
                return Func::Append(mHead, node, lastNode);
            }

            /** Remove the given node from the list.
                @return true if the node could be removed, false otherwise */
            inline bool Remove(T *node) {
                return Func::Remove(mHead, node);
            }

            /** returns the first element in the list */
            inline T* Front() const {
                return mHead;
            }

            /** Locate the given node in the list.
                @param node the object to locate in the list
                @return the located node or null if not contained in list */
            inline T* Locate(T *node) const {
                return Func::Locate(mHead, node);
            }

            inline UInt32 Count() {
                return Func::Count(mHead);
            }

            inline void Clear() {
                Func::Clear(mHead);
            }

            //@{
            /** returns begin iterator
                @return iterator object pointing to the begin of the list */
            inline Iterator Begin() const {
                return Iterator(mHead);
            }

            inline ConstIterator ConstBegin() const {
                return ConstIterator(mHead);
            }
            //@}

            //@{
            /** returns end iterator
                @return iterator object pointing to the end of the list */
            inline Iterator End() const {
                return Iterator(0);
            }

            inline ConstIterator ConstEnd() const {
                return ConstIterator(0);
            }
            //@}

        private:
            T *mHead;

            //@{
            /// inhibit copy constructor and assignment
            ObjectListBase(const ObjectListBase&) { }
            ObjectListBase& operator=(const ObjectListBase&) { return *this; }
            //@}
    };

    /** Intrusive list of objects */
    template<typename T, typename Accessor = DefaultAccessor<T> > class ObjectList : public ObjectListBase<ObjectListFunc<T, Accessor> > {
    };

    /** Intrusive sorted list of objects */
    template<typename T, bool (*SortPredicate)(const T&, const T&), typename Accessor = DefaultAccessor<T> > class SortedObjectList : public ObjectListBase<SortedObjectListFunc<T, SortPredicate, Accessor> > {
    };

    /// @}
}}

#endif
